/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/***************************************************************************
 *            commands.c
 *
 *  Wed Aug 25 22:17:41 2004
 *  Copyright  2004  Gergely POLONKAI
 *  polesz@techinfo.hu
 ****************************************************************************/
#define _GNU_SOURCE
#include <ctype.h>
#include <netdb.h>
#include <stdlib.h>
#include <netinet/in.h>
#include <string.h>
#include <arpa/inet.h>
#include <stdio.h>
#include <errno.h>

#include "typedefs.h"
#include "functions.h"
#include "defines.h"
#include "variables.h"

/* I have to prototype these, but I don't want to define them yet. Don't ask me why :) */
BC_COMMAND(CommandExit);
BC_COMMAND(CommandMode);
BC_COMMAND(CommandConnect);

/*
 * command[]
 * Command list
 */
command_t command[] = {
	{"QUIT",    CT_INTERNAL,	CommandExit},
	{"EXIT",    CT_INTERNAL,	CommandExit},
	{"MODE",    CT_INTERNAL,	CommandMode},
	{"CONNECT", CT_INTERNAL,	CommandConnect},
};

struct sockaddr_in ConnectData;

#define num_commands (sizeof(command) / sizeof(command_t))

/*
 * BC_COMMAND(*)
 * Callbacks for the commands.
 */
BC_COMMAND(CommandExit)
{
	if (ConfigData.debug_mode)
	{
		FeedInfoToTerminal(TRUE, "Exiting...");
	}
	ExitCleanly();
	return TRUE;
}

BC_COMMAND(CommandMode)
{
	gboolean ret = TRUE;

	if (paramnum == 0)
	{
		FeedErrorToTerminal(TRUE, "Usage: MODE (M|E|B)");
		return FALSE;
	}
	
	switch (toupper(*param[0]))
	{
		case 'B':
			ChangeMode('B');
			break;
		case 'M':
			ChangeMode('M');
			break;
		case 'E':
			ChangeMode('E');
			break;
		default:
			FeedErrorToTerminal(TRUE, "Usage: MODE (M|E|B)");
  		ret = FALSE;
			break;
	}

	return ret;
}

BC_COMMAND(CommandConnect)
{
	gchar *hostname;
	unsigned short port;

	if (paramnum < 1)
	{
		FeedErrorToTerminal(TRUE, "Usage: CONNECT botname or CONNECT host port [user [password]]");
		return FALSE;
	}
	if (paramnum < 2)
	{
		FeedInfoToTerminal(TRUE, "Sorry, connecting to a pre-defined bot is not implemented yet.");
		hostname = NULL;
		port = 0;
		return TRUE;
	}
	else
	{
		hostname = g_strdup(param[0]);
		if (!is_numeric(param[1]))
		{
      struct protoent *prot;
      struct servent *serv;

      if ((prot = getprotobyname("tcp")) == NULL)
      {
        FeedErrorToTerminal(TRUE, "Unable to get TCP protocol!");
        return FALSE;
      }

      if ((serv = getservbyname(param[1], prot->p_name)) == NULL)
      {
        FeedErrorToTerminal(TRUE, "Unknown port %s", param[1]);
        return FALSE;
      }

      port = ntohs(serv->s_port);
		}
		else
		{
			port = atoi(param[1]);
		}
	}
	
	if (Connect(hostname, port) < 0)
	{
		return FALSE;
	}
	
	StaticData.socket_tag = gdk_input_add(StaticData.socket, GDK_INPUT_READ, DataReady, NULL);

	return TRUE;
}

/*
 * ProcessLine()
 * Processes the full command line.
 */
void
ProcessLine(gchar *text)
{
	gchar ThisMode;
	
	ThisMode = StaticData.CurrentMode;
	if (*text == '/')
	{
		ThisMode = 'B';
	}
	else if (*text == '.')
	{
		ThisMode = 'E';
	}

	switch (ThisMode)
	{
		case 'E':
			dprintf(StaticData.socket, ".%s\n", (*text == '.') ? (text + 1) : text);
			break;
		case 'B':
			{
				char **words;
				int num, i;
				gboolean found = FALSE;
				
				if ((num = WrapString(text, " \t", &words)) == -1)
					return;

				if (*words[0] == '/')
				{
					words[0] = (words[0] + 1);
				}

				for (i = 0; i < num_commands; i++)
				{
					if (str_eq(command[i].command, words[0], FALSE))
					{
						found = TRUE;
						if (!(command[i].func)(num - 1, words + 1))
						{
							if (ConfigData.debug_mode)
							{
								FeedErrorToTerminal(TRUE, "Error while processing command.");
							}
						}
					}
				}
				if (!found)
				{
					FeedErrorToTerminal(TRUE, "Unknown command.");
				}
			}
			break;
		case 'M':
		default:
			{
				/* The following code is from php source code. I know I have to clean it, but it works :) */
				char *src = g_strdup(text);
				int src_len = strlen(text);
				int nl;
				int *new_len = &nl;
				
				char *outbuf = 0;
				
				if (src && src_len)
				{
					int outlen = src_len;
		      int outlenleft = src_len;
					int inlenleft = src_len;
					iconv_t ic = iconv_open("ISO-8859-2", "UTF-8");
					char* src_ptr = (char*)src;
					char* out_ptr = 0;

					if(ic != (iconv_t)-1)
					{
						size_t st;
						outbuf = (char*)g_malloc(outlen + 1);

						if(outbuf)
						{
							out_ptr = (char*)outbuf;
							while(inlenleft)
							{
								st = iconv(ic, &src_ptr, &inlenleft, &out_ptr, &outlenleft);
								if(st == -1)
								{
                  if(errno == E2BIG)
									{
                    int diff = out_ptr - outbuf;
                    outlen += inlenleft;
                    outlenleft += inlenleft;
                    outbuf = (char*)realloc(outbuf, outlen + 1);
                    if(!outbuf)
										{
                      break;
                    }
                    out_ptr = outbuf + diff;
                  }
                  else
									{
                    free(outbuf);
                    outbuf = 0;
                    break;
                  }
								}
							}
						}
						iconv_close(ic);
					}
					outlen -= outlenleft;

					if(new_len)
					{
						*new_len = outbuf ? outlen : 0;
					}
					if(outbuf)
					{
						outbuf[outlen] = 0;
					}
				}
				dprintf(StaticData.socket, "%s%s\n", (*text == '.') ? " " : "", outbuf);
				break;
			}
	}
	
}
