/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/***************************************************************************
 *            widgets.c
 *
 *  Sat Aug 21 23:56:24 2004
 *  Copyright  2004  Gergely Polonkai
 *  polesz@techinfo.hu
 ****************************************************************************/
#include "botcommander.h"

#include <gtk/gtk.h>
#include <vte/vte.h>
#include <string.h>

#include "variables.h"
#include "functions.h"

/*
 * Menu*()
 * Callback functions for the menu items.
 */
static void
MenuExit(GtkWidget *widget, gpointer data)
{
	ExitCleanly();
}

static void
MenuChangeMode(gpointer data, guint action, GtkWidget *widget)
{
	static char modes[4] = "MEB";
	if ((GTK_CHECK_MENU_ITEM(widget)->active) && (StaticData.CurrentMode != modes[action - 1]))
	{
		StaticData.CurrentMode = modes[action - 1];
	}
}

/*
 * CreateMenu()
 * Creates the main menubar.
 */
GtkWidget *
CreateMenu(void)
{
	static GtkItemFactoryEntry menu_items[] = {
	/*  Text                           Hotkey        Callback action type */
		{ "/_Bot",                       NULL,         NULL,           0, "<Branch>"                           },
		{ "/Bot/_Connect",               "<control>C", NULL,           0, "<StockItem>", GTK_STOCK_NETWORK     },
		{ "/Bot/_Disconnect",            "<control>D", NULL,           0, "<Item>"                             },
		{ "/Bot/sep1",                   NULL,         NULL,           0, "<Separator>"                        },
		{ "/Bot/E_xit",                  "<control>Q", MenuExit,       0, "<StockItem>", GTK_STOCK_QUIT        },
		{ "/_Mode",                      NULL,         NULL,           0, "<Branch>"                           },
		{ "/Mode/_Message",              "F5",         MenuChangeMode, 1, "<RadioItem>"                        },
		{ "/Mode/_Eggdrop command",      "F6",         MenuChangeMode, 2, "/Mode/Message"                      },
		{ "/Mode/_BotCommander command", "F7",         MenuChangeMode, 3, "/Mode/Message"                      },
		{ "/_Setup",                     NULL,         NULL,           0, "<Branch>"                           },
		{ "/Setup/Manage _list",         "<control>L", NULL,           0, "<Item>"                             },
		{ "/Setup/_Settings",            "<control>S", NULL,           0, "<StockItem>", GTK_STOCK_PREFERENCES },
		{ "/_Help",                      NULL,         NULL,           0, "<Branch>"                           },
		{ "/Help/_About...",             NULL,         NULL,           0, "<Item>"                             }
	};
	static gint nmenu_items = sizeof(menu_items) / sizeof(menu_items[0]);
	GtkAccelGroup *accel_group;
	
	accel_group = gtk_accel_group_new();
	
	StaticData.main_factory = gtk_item_factory_new(GTK_TYPE_MENU_BAR, "<main>", accel_group);
	
	gtk_item_factory_create_items(StaticData.main_factory, nmenu_items, menu_items, NULL);
	
	gtk_window_add_accel_group(GTK_WINDOW(StaticData.main_window), accel_group);
	
	return gtk_item_factory_get_widget(StaticData.main_factory, "<main>");
}

/*
 * CreateMainWindow()
 * Creates the main window with all of its components.
 */
gboolean
CreateMainWindow(void)
{
	/* Create the main window and set its properties */
	StaticData.main_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	g_signal_connect(G_OBJECT(StaticData.main_window), "delete_event", G_CALLBACK(delete_event), NULL);
	g_signal_connect(G_OBJECT(StaticData.main_window), "destroy", G_CALLBACK(destroy_event), NULL);
	
  /* Create the vertical box, which contains every widgets */
	StaticData.main_vbox = gtk_vbox_new(FALSE, 0);
	
	StaticData.main_menu = CreateMenu();
	gtk_box_pack_start(GTK_BOX(StaticData.main_vbox), StaticData.main_menu, FALSE, TRUE, 0);
	gtk_widget_show(StaticData.main_menu);
	
	/* TODO Create a toolbar */
	
	/* Create the virtual terminal window and pack it into the vertical box */
	StaticData.main_vte = (VteTerminal *)vte_terminal_new();
	vte_terminal_set_encoding(VTE_TERMINAL(StaticData.main_vte), "UTF-8");
	vte_terminal_set_allow_bold(VTE_TERMINAL(StaticData.main_vte), ALLOW_BOLD);
	vte_terminal_set_scroll_on_output(VTE_TERMINAL(StaticData.main_vte), ConfigData.scroll_on_output);
	vte_terminal_set_background_transparent(VTE_TERMINAL(StaticData.main_vte), ConfigData.transparent_background);
	vte_terminal_set_background_saturation(VTE_TERMINAL(StaticData.main_vte), ConfigData.background_saturation);
	vte_terminal_set_audible_bell(VTE_TERMINAL(StaticData.main_vte), AUDIBLE_BELL);
	vte_terminal_set_visible_bell(VTE_TERMINAL(StaticData.main_vte), VISIBLE_BELL);
	vte_terminal_set_font_from_string(VTE_TERMINAL(StaticData.main_vte), ConfigData.vte_font);
	gtk_signal_connect(GTK_OBJECT(StaticData.main_vte), "clicked", GTK_SIGNAL_FUNC(vte_clicked), NULL);
	GTK_WIDGET_UNSET_FLAGS(GTK_WIDGET(StaticData.main_vte), GTK_CAN_FOCUS);
	gtk_signal_connect(GTK_OBJECT(StaticData.main_vte), "focus-in-event", GTK_SIGNAL_FUNC(grab_focus), NULL);
	gtk_box_pack_start(GTK_BOX(StaticData.main_vbox), GTK_WIDGET(StaticData.main_vte), TRUE, TRUE, 0);
	gtk_widget_show(GTK_WIDGET(StaticData.main_vte));
	
	/* Create the command line and pack it into the vertical box */
	StaticData.main_commandline = gtk_entry_new();
	g_signal_connect(G_OBJECT(StaticData.main_commandline), "activate", G_CALLBACK(activate_event), NULL);
	gtk_signal_connect(GTK_OBJECT(StaticData.main_commandline), "key_press_event", GTK_SIGNAL_FUNC(commandline_keypress), NULL);
	gtk_box_pack_start(GTK_BOX(StaticData.main_vbox), StaticData.main_commandline, FALSE, TRUE, 0);
	gtk_widget_show(StaticData.main_commandline);
	
	/* TODO Create a status bar */
	
	/* Put the vertical box into the main window */
	gtk_container_add(GTK_CONTAINER(StaticData.main_window), StaticData.main_vbox);
	gtk_widget_show(StaticData.main_vbox);
	
	gtk_widget_grab_focus(GTK_WIDGET(StaticData.main_commandline));

	gtk_widget_show(StaticData.main_window);
	
	return TRUE;
}

/*
 * FeedLineToTerminal()
 * Feeds some text to the terminal widget.
 */
void
FeedLineToTerminal(gboolean NeedCRLF, gchar *line, va_list list)
{
	gchar *string;
	gint len;
	
	len = valist_to_string(&string, line, list);

	vte_terminal_feed((VteTerminal *)StaticData.main_vte, string, strlen(string));
	if (NeedCRLF)
	{
		vte_terminal_feed((VteTerminal *)StaticData.main_vte, "\r\n", 2);
	}
}

/*
 * FeedCRLFToTerminal()
 * Feeds a CD LF pair to the terminal, if needed.
 */
void
FeedCRLFToTerminal(gboolean NeedIt)
{
	if (NeedIt)
	{
		vte_terminal_feed((VteTerminal *)StaticData.main_vte, "\r\n", 2);
	}
}

/*
 * FeedMessageToTerminal()
 * Feeds some informational (un-coloured) text to the terminal, always appending
 * a line-feed.
 */
void
FeedMessageToTerminal(gboolean NeedCRLF, gchar *line, ...)
{
	va_list pl;
	
	va_start(pl, line);
	FeedLineToTerminal(FALSE, line, pl);
	va_end(pl);
	FeedCRLFToTerminal(NeedCRLF);
}

/*
 * FeedErrorToTerminal()
 * Feeds some error (red) text to the terminal, always appending a line-feed
 */
void
FeedErrorToTerminal(gboolean NeedCRLF, gchar *line, ...)
{
	va_list pl;

	va_start(pl, line);
	FeedMessageToTerminal(FALSE, "\033[1;31m");
	FeedLineToTerminal(FALSE, line, pl);
	va_end(pl);
	FeedMessageToTerminal(NeedCRLF, "\033[0m");
}

/*
 * FeedInfoToTerminal()
 * Feeds some informational (green) text to the terminal, always appending a
 * line-feed.
 */
void
FeedInfoToTerminal(gboolean NeedCRLF, gchar *line, ...)
{
	va_list pl;

	va_start(pl, line);
	FeedMessageToTerminal(FALSE, "\033[1;32m");
	FeedLineToTerminal(FALSE, line, pl);
	va_end(pl);
	FeedMessageToTerminal(NeedCRLF, "\033[0m");
}
