"""Base Window Class

Based on MHGui work
by Mario Latronico and Manuel Bastioni
mbastioni@tiscalinet.it
vers. 1.12

Graphics class to draw widget primitives, like lines, rectangles,
picture, text, etc.

"""

#    This file is part of brad (Blended RADiance).
#
#    brad is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    brad is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with Foobar; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

from Blender.Draw import *
from Blender.BGL  import *
from math import *

class BaseWindowClass:

	borderColor = {'R':0, 'G':0, 'B':0} # default RGB = (0,0,0)
	boxColor    = {'R':1, 'G':1, 'B':1} # default RGB = (1,1,1)
	titleColor  = {'R':0, 'G':0, 'B':0} # default RGB = (0,0,0)
	barColor    = {'R':0.7, 'G':0.7, 'B':0.7} # default RGB = (0.7,0.7,0.7)

	halftone = Buffer(GL_BYTE,128)
	halftone[:] =[0xC0, 0xC0, 0xC0, 0xC0, 0x60, 0x60, 0x60, 0x60,
			0x30, 0x30, 0x30, 0x30, 0x18, 0x18, 0x18, 0x18,
			0x0C, 0x0C, 0x0C, 0x0C, 0x06, 0x06, 0x06, 0x06,
			0x03, 0x03, 0x03, 0x03, 0x81, 0x81, 0x81, 0x81,
			0xC0, 0xC0, 0xC0, 0xC0, 0x60, 0x60, 0x60, 0x60,
			0x30, 0x30, 0x30, 0x30, 0x18, 0x18, 0x18, 0x18,
			0x0C, 0x0C, 0x0C, 0x0C, 0x06, 0x06, 0x06, 0x06,
			0x03, 0x03, 0x03, 0x03, 0x81, 0x81, 0x81, 0x81,
			0xC0, 0xC0, 0xC0, 0xC0, 0x60, 0x60, 0x60, 0x60,
			0x30, 0x30, 0x30, 0x30, 0x18, 0x18, 0x18, 0x18,
			0x0C, 0x0C, 0x0C, 0x0C, 0x06, 0x06, 0x06, 0x06,
			0x03, 0x03, 0x03, 0x03, 0x81, 0x81, 0x81, 0x81,
			0xC0, 0xC0, 0xC0, 0xC0, 0x60, 0x60, 0x60, 0x60,
			0x30, 0x30, 0x30, 0x30, 0x18, 0x18, 0x18, 0x18,
			0x0C, 0x0C, 0x0C, 0x0C, 0x06, 0x06, 0x06, 0x06,
			0x03, 0x03, 0x03, 0x03, 0x81, 0x81, 0x81, 0x81]

	halftone2 = Buffer(GL_BYTE,128)
	halftone2[:] =[0xAA, 0xAA, 0xAA, 0xAA, 0x55, 0x55, 0x55, 0x55,
			0xAA, 0xAA, 0xAA, 0xAA, 0x55, 0x55, 0x55, 0x55,
			0xAA, 0xAA, 0xAA, 0xAA, 0x55, 0x55, 0x55, 0x55,
			0xAA, 0xAA, 0xAA, 0xAA, 0x55, 0x55, 0x55, 0x55,
			0xAA, 0xAA, 0xAA, 0xAA, 0x55, 0x55, 0x55, 0x55,
			0xAA, 0xAA, 0xAA, 0xAA, 0x55, 0x55, 0x55, 0x55,
			0xAA, 0xAA, 0xAA, 0xAA, 0x55, 0x55, 0x55, 0x55,
			0xAA, 0xAA, 0xAA, 0xAA, 0x55, 0x55, 0x55, 0x55,
			0xAA, 0xAA, 0xAA, 0xAA, 0x55, 0x55, 0x55, 0x55,
			0xAA, 0xAA, 0xAA, 0xAA, 0x55, 0x55, 0x55, 0x55,
			0xAA, 0xAA, 0xAA, 0xAA, 0x55, 0x55, 0x55, 0x55,
			0xAA, 0xAA, 0xAA, 0xAA, 0x55, 0x55, 0x55, 0x55,
			0xAA, 0xAA, 0xAA, 0xAA, 0x55, 0x55, 0x55, 0x55,
			0xAA, 0xAA, 0xAA, 0xAA, 0x55, 0x55, 0x55, 0x55,
			0xAA, 0xAA, 0xAA, 0xAA, 0x55, 0x55, 0x55, 0x55,
			0xAA, 0xAA, 0xAA, 0xAA, 0x55, 0x55, 0x55, 0x55]

	def __init__(self, LocX, LocY, Width, Height):

		self.LocX   = LocX
		self.LocY   = LocY
		self.Width  = Width
		self.Height = Height
		self.isSelected=0
		self.grabEvent=0
		self.detailVisibility=1
		self.Visibility=1

	#-------------------------------------------#
	#    DRAW A SIMPLE TRIANGLE              #
	#-------------------------------------------#

	def drawTriangle(self,x1,y1,x2,y2,x3,y3,r,g,b,a):
		"""
		DESCRIPTION:
                draw a triangle,
		antialiased (GL_BLEND),(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA),
		tra i tre vertici (x1,y1),(x2,y2),(x3,y3).
                Transparency is determined according to the value of the a variable

		SYNOPSIS:
		drawTriangle(x1,y1,x2,y2,x3,y3,r,g,b,a)

		PARAMETERS:
		x1

		y1

		x2

		y2

		x3

		y3

		r = red [0:1]

		g = green [0:1]

		b = blue [0:1]

		a = transparency [0:1]


		RETURN VALUES:
		Nothing
		"""
		glColor4f(r,g,b,a)
		glPolygonMode(GL_FRONT_AND_BACK,GL_FILL)
		glEnable(GL_BLEND)
		glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA)
		glEnable(GL_POLYGON_SMOOTH)
		glBegin(GL_POLYGON)
		glVertex2d(x1,y1)
		glVertex2d(x2,y2)
		glVertex2d(x3,y3)
		glEnd()
		glDisable(GL_POLYGON_SMOOTH)
		glDisable(GL_BLEND)

	#-------------------------------------------#
	#    DRAW A SIMPLE RECTANGLE                #
	#-------------------------------------------#

	def drawRectangleSimple(self,x1,y1,x2,y2,r,g,b,a):

		glColor4f(r,g,b,a)
		glPolygonMode(GL_FRONT_AND_BACK,GL_FILL)
		glEnable(GL_BLEND)
		glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA)
		glEnable(GL_LINE_SMOOTH)
		glBegin(GL_LINE_LOOP)
		glVertex2d(x1,y1)
		glVertex2d(x2,y1)
		glVertex2d(x2,y2)
		glVertex2d(x1,y2)
		glEnd()
		glDisable(GL_LINE_SMOOTH)
		glDisable(GL_BLEND)

        #-------------------------------------------#
	#    DRAW A FILLED RECTANGLE                #
	#-------------------------------------------#

	def drawRectangle(self,x1,y1,x2,y2,r,g,b,a):

		glEnable(GL_BLEND)
		glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA)
		glColor4f(r,g,b,a)
		glRectf(x1,y1,x2,y2)
		glDisable(GL_BLEND)

        #-------------------------------------------#
	#    DRAW A SMOOTH LINE                     #
	#-------------------------------------------#

	def drawLine(self,x1,y1,x2,y2,r,g,b,w,a):

		glEnable(GL_BLEND)
		glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA)
		glEnable(GL_LINE_SMOOTH)

		#glEnable(GL_LINE_STIPPLE)
		#glLineStipple(1,0x00FF)

		glLineWidth(w)
		glColor4f(r,g,b,a)
		glBegin(GL_LINES)
		glVertex2d(x1,y1)
		glVertex2d(x2,y2)
		glEnd()

		#glDisable(GL_LINE_STIPPLE)

		glDisable(GL_LINE_SMOOTH)
		glDisable(GL_BLEND)

	#-------------------------------------------#
	#    DRAW A NON SMOOTH LINE              #
	#-------------------------------------------#

	def drawLineHard(self,x1,y1,x2,y2,r,g,b,w,a):

                glLineWidth(w)
		glColor4f(r,g,b,a)
		glBegin(GL_LINES)
		glVertex2d(x1,y1)
		glVertex2d(x2,y2)
		glEnd()

	#-------------------------------------------#
	#    DRAW A SMOOTH SOLID FILLED POLYGON     #
	#-------------------------------------------#

	def drawPolygonF(self,x,y,sides,radius,r,g,b,a):

		glColor4f(r,g,b,a)

		step=360/sides
		oY=y
		oX=x

		vY1=oY
		vX1=oX
		glPolygonMode(GL_FRONT_AND_BACK,GL_FILL)
		glEnable(GL_BLEND)
		glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA)
		glEnable(GL_POLYGON_SMOOTH)
		glBegin(GL_POLYGON)
		alpha=0
		while alpha<=360:
			alpha=alpha+step
			angle=alpha/57.29577957795135
			vX=oX+(radius*sin(angle))
			vY=oY+(radius*cos(angle))
			#glVertex2d(oX,oY)
			glVertex2d(vX1,vY1)
			glVertex2d(vX,vY)
			vY1=vY
			vX1=vX
		glEnd()
		glDisable(GL_POLYGON_SMOOTH)
		glDisable(GL_BLEND)


	#-------------------------------------------#
	#    DRAW A SMOOTH FILLED POLYGON           #
	#-------------------------------------------#

	def drawPolygonFS(self,x,y,sides,radius,r,g,b,a):

		glColor4f(r,g,b,a)

		step=360/sides
		oY=y
		oX=x
		vY1=oY
		vX1=oX+radius
		glPolygonMode(GL_FRONT_AND_BACK,GL_FILL)
		glEnable(GL_BLEND)
		glEnable(GL_POLYGON_STIPPLE)
		glPolygonStipple(self.halftone)
		glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA)
		glEnable(GL_POLYGON_SMOOTH)
		glBegin(GL_POLYGON)
		alpha=0
		while alpha<=360:
			alpha=alpha+step
			angle=alpha/57.29577957795135
			vX=oX+(radius*sin(angle))
			vY=oY+(radius*cos(angle))
			#glVertex2d(oX,oY)
			glVertex2d(vX1,vY1)
			glVertex2d(vX,vY)
			vY1=vY
			vX1=vX
		glEnd()
		glDisable(GL_POLYGON_SMOOTH)
		glDisable(GL_POLYGON_STIPPLE)
		glDisable(GL_BLEND)



	#-------------------------------------------#
	#    DRAW A SMOOTH NON-FILLED POLYGON       #
	#-------------------------------------------#

	def drawPolygon(self,x,y,sides,radius,r,g,b,w,a):

                #glColor4f(r,g,b,a)
		step=360/sides
		oY=y
		oX=x

		vY1=oY
		vX1=oX+radius

		alpha=0
		while alpha<=360:
			alpha=alpha+step
			angle=alpha/57.29577957795135
			vX=oX+(radius*sin(angle))
			vY=oY+(radius*cos(angle))
			self.drawLine(vX1,vY1,vX,vY,r,g,b,w,a)
			glVertex2d(vX1,vY1)
			glVertex2d(vX,vY)
			vY1=vY
			vX1=vX


	#-------------------------------------------#
	#    DRAW A "SKETCHED" RECTANGLE            #
	#-------------------------------------------#

	def drawRectangleS(self,x1,y1,x2,y2,r,g,b,a):

		self.drawRectangle(x1,y1,x2,y2,r,g,b,a)
		sketch1LineX=abs((x1-x2)/10)
		sketch2LineX=abs((x1-x2)/10)

		sketch1LineY=abs((y1-y2)/10)
		sketch2LineY=abs((y1-y2)/10)

		self.drawLine(x1-sketch1LineX,y1,x2+sketch1LineX,y1,0,0,0,1,1)
		self.drawLine(x1,y1+sketch1LineY,x1,y2-sketch1LineY,0,0,0,1,1)
		self.drawLine(x1-sketch1LineX,y2,x2+sketch1LineX,y2,0,0,0,1,1)
		self.drawLine(x2,y1+sketch1LineY,x2,y2-sketch1LineY,0,0,0,1,1)

	#-------------------------------------------#
	#    DRAW A self.halftone2 FILLED RECTANGLE #
	#-------------------------------------------#

	def drawRectangleHF(self,x1,y1,x2,y2,r,g,b,a):

		glColor4f(r,g,b,a)

		glPolygonMode(GL_FRONT_AND_BACK,GL_FILL)
		glEnable(GL_BLEND)
		glEnable(GL_POLYGON_STIPPLE)
		glPolygonStipple(self.halftone2)
		glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA)
		glEnable(GL_POLYGON_SMOOTH)
		glBegin(GL_POLYGON)

		glVertex2d(x1,y1)
		glVertex2d(x2,y1)
		glVertex2d(x2,y2)
		glVertex2d(x1,y2)

		glEnd()
		glDisable(GL_POLYGON_SMOOTH)
		glDisable(GL_POLYGON_STIPPLE)
		glDisable(GL_BLEND)



	#---------------------------------------------#
	#    DRAW A self.halftone FILLED RECTANGLE    #
	#---------------------------------------------#

	def drawRectangleSF(self,x1,y1,x2,y2,r,g,b,a):

		glColor4f(r,g,b,a)

		glPolygonMode(GL_FRONT_AND_BACK,GL_FILL)
		glEnable(GL_BLEND)
		glEnable(GL_POLYGON_STIPPLE)
		glPolygonStipple(self.halftone)
		glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA)
		glEnable(GL_POLYGON_SMOOTH)
		glBegin(GL_POLYGON)

		glVertex2d(x1,y1)
		glVertex2d(x2,y1)
		glVertex2d(x2,y2)
		glVertex2d(x1,y2)

		glEnd()
		glDisable(GL_POLYGON_SMOOTH)
		glDisable(GL_POLYGON_STIPPLE)
		glDisable(GL_BLEND)

	#-------------------------------------------#
	#    DRAW A SIMPLE TEXT                     #
	#-------------------------------------------#

	def drawText(self,x,y,r,g,b,textString):

		glColor3f(r,g,b)
		glRasterPos2d(x,y)
		Text(textString)


	#-------------------------------------------#
	#    DRAW A BEZIER LINE                     #
	#                                           #
	# Draw a Bezier line from x1,y1 to x2,y2    #
	# and controlled by the x3,y3 and x4,y4     #
	#                                           #
	#-------------------------------------------#

	def drawBezier(self,x1,y1,x2,y2,x3,y3,x4,y4,r,g,b,width,alpha,point,lines):

		Ax = x1; Ay = y1
		Bx = x3; By = y3
		Cx = x4; Cy = y4
		Dx = x2; Dy = y2

		# parameter
		a = 1.0;
		b = 1.0 - a;

		#Visualizzation of construction point
		if point == 1:
			glPointSize(5)
			glBegin(GL_POINTS)
			glVertex2d(Ax,Ay)
			glVertex2d(Bx,By)
			glVertex2d(Cx,Cy)
			glVertex2d(Dx,Dy)
			glEnd()

		#Visualizzation of construction point
		if lines == 1:
			self.drawLine(Ax,Ay,Bx,By,0,0,0,1,1)
			self.drawLine(Cx,Cy,Dx,Dy,0,0,0,1,1)


		glEnable(GL_BLEND)
		glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA)
		glEnable(GL_LINE_SMOOTH)
		glLineWidth(width)
		glColor4f(r,g,b,alpha)
		glBegin(GL_LINE_STRIP)

		for i in range(21):

			# Get a point on the curve
			X = Ax*a*a*a + Bx*3*a*a*b + Cx*3*a*b*b + Dx*b*b*b;
			Y = Ay*a*a*a + By*3*a*a*b + Cy*3*a*b*b + Dy*b*b*b;

			# Draw the line from point to point (assuming OGL is set up properly)
			glVertex2d(X, Y);

			# Change the variable
			a = a - 0.05;
			b = 1.0 - a;

		glEnd()
		glDisable(GL_LINE_SMOOTH)
		glDisable(GL_BLEND)

	#-------------------------------------------#
	#    SET THE BORDER COLOR                   #
	#-------------------------------------------#

	def setBorderColor(self,r,g,b):

                self.borderColor['R'] = r
		self.borderColor['G'] = g
		self.borderColor['B'] = b

	#-------------------------------------------#
	#    SET THE BOX COLOR		            #
	#-------------------------------------------#

	def setBoxColor(self,r,g,b):

                self.boxColor['R'] = r
		self.boxColor['G'] = g
		self.boxColor['B'] = b

	#-------------------------------------------#
	#    SET THE TITLE COLOR                    #
	#-------------------------------------------#

	def setTitleColor(self,r,g,b):

                self.titleColor['R'] = r
		self.titleColor['G'] = g
		self.titleColor['B'] = b


	#-------------------------------------------#
	#    SET THE TOPBAR COLOR                   #
	#-------------------------------------------#

	def setTopColor(self,r,g,b):

                self.barColor['R'] = r
		self.barColor['G'] = g
		self.barColor['B'] = b

	#-------------------------------------------#
	#    DRAW A TITLE                        #
	#-------------------------------------------#

	def drawTitle(self,title):

                self.drawText(self.LocX+5,
			self.LocY-15,
			self.titleColor['R'],
			self.titleColor['G'],
			self.titleColor['B'],
			title
			)

	#-------------------------------------------#
	#    DRAW A VERTICAL TITLE               #
	#-------------------------------------------#

	def drawTitleV(self,title):

                verticalSpacing=10
		textLocY=self.LocY
		for letter in title:
			textLocY=textLocY-verticalSpacing
			self.drawText(self.LocX+self.Width+10,
						textLocY,
						self.titleColor['R'],
						self.titleColor['G'],
						self.titleColor['B'],
						letter
					)



	#-------------------------------------------#
	#    DRAW A WINDOW BOX WITH SHADOW       #
	#-------------------------------------------#

	def drawBox(self):

		self.drawRectangle(
			self.LocX+2,
			self.LocY-4,
			self.LocX+self.Width+6,
			self.LocY-self.Height-6,
			0.4,0.4,0.4,0.3
			)

		self.drawRectangle(
			self.LocX,
			self.LocY-2,
			self.LocX+self.Width+4,
			self.LocY-self.Height-4,
			0.3,0.3,0.3,0.3
			)

		self.drawRectangle(
			self.LocX-2,
			self.LocY+2,
			self.LocX+self.Width+2,
			self.LocY-self.Height-2,
			self.borderColor['R'],
			self.borderColor['G'],
			self.borderColor['B'],
			0.3
			)

		self.drawRectangle(
			self.LocX,
			self.LocY,
			self.LocX+self.Width,
			self.LocY-self.Height,
			self.boxColor['R'],
			self.boxColor['G'],
			self.boxColor['B'],
			0.3
			)

		self.drawRectangle(
			self.LocX+1,
			self.LocY-1,
			self.LocX+self.Width-1,
			self.LocY-20,
			self.barColor['R'],
			self.barColor['G'],
			self.barColor['B'],
			0.3
			)


	#-------------------------------------------#
	#    DRAW A WINDOW BOX SKETCH STYLE      #
	#-------------------------------------------#

	def drawBoxS(self):


		self.drawRectangleS(
			self.LocX,
			self.LocY,
			self.LocX+self.Width,
			self.LocY-self.Height,
			self.boxColor['R'],
			self.boxColor['G'],
			self.boxColor['B'],
			1.0
			)

		self.drawRectangleSF(
			self.LocX+1,
			self.LocY-1,
			self.LocX+self.Width-1,
			self.LocY-20,
			0.6,
			0.4,
			0.4,
			1.0
			)



	#-------------------------------------------#
	#    DRAW A WINDOW BOX self.halftone2 STYLE   #
	#-------------------------------------------#
	#Used for selection

	def drawBoxHF(self):

                self.drawRectangle(
			self.LocX,
			self.LocY,
			self.LocX+self.Width,
			self.LocY-self.Height,
			self.boxColor['R'],
			self.boxColor['G'],
			self.boxColor['B'],
			0.3
			)

		self.drawRectangleHF(
			self.LocX+1,
			self.LocY-1,
			self.LocX+self.Width-1,
			self.LocY-20,
			1.0,
			1.0,
			1.0,
			1.0
			)


