/*******************************************************************************
 * capi20wrapper - Collection of functions and data structures for accessing
 *                 a shared library which implements the interface as described
 *                 in section 'Linux' in part two of CAPI 2.0
 * 
 * Written in 2014 by Swen Lünig.
 * 
 * To the extent possible under law, the author(s) have dedicated all copyright
 * and related and neighboring rights to this software to the public domain
 * worldwide. This software is distributed without any warranty.
 * 
 * You should have received a copy of the CC0 Public Domain Dedication along
 * with this software. If not, see
 * <http://creativecommons.org/publicdomain/zero/1.0/>.
 ******************************************************************************/

/******************************************************************************
 * This little example tries to establish a connection.
 * The following sequences are implemented:
 * CONNECT_REQ, CONNECT_CONF
 * CONNECT_REQ, CONNECT_CONF, DISCONNECT_IND, DISCONNECT_RESP
 * CONNECT_REQ, CONNECT_CONF, CONNECT_ACTIVE_IND, CONNECT_ACTIVE_RESP, DISCONNECT_REQ, DISCONNECT_CONF, DISCONNECT_IND, DISCONNECT_RESP
 * Only the last sequence counts as a successful try of establishing a
 * connection and the program returns 0. In all other cases the program returns
 * 1.
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <capi20.h>

#include "capi20wrapper.h"

static
int wasTheTrySuccessful = 0;

static
int didTheSequenceFinish = 0;

static
unsigned int get_connect_conf ( unsigned int ApplID,
				capi20wrapper_Word_t Message_number,
				capi20wrapper_DWord_t PLCI,
				capi20wrapper_Word_t Info )
{
  printf ( "--------------------------------\n" );
  printf ( "got a CONNECT_CONF\n" );
  printf ( "PLCI = 0x%08lx\n",
	   PLCI );
  printf ( "Info = 0x%04x\n",
	   Info );
  
  if ( 0x0000 != Info )
    {
      printf ( "Oops! Info is not 0x0000, terminating the process.\n" );
      didTheSequenceFinish = 1;
    }
  
  return 0x0000;
}

static
unsigned int get_connect_active_ind ( unsigned int ApplID,
				      capi20wrapper_Word_t Message_number,
				      capi20wrapper_DWord_t PLCI,
				      const capi20wrapper_Struct_t * Connected_number,
				      const capi20wrapper_Struct_t * Connected_subaddress,
				      const capi20wrapper_Struct_t * LLC )
{
  printf ( "--------------------------------\n" );
  printf ( "got a CONNECT_ACTIVE_IND\n" );
  printf ( "PLCI = 0x%08lx\n",
	   PLCI );
  capi20wrapper_printf_Struct ( "Connected_number",
				Connected_number );
  capi20wrapper_printf_Struct ( "Connected_subaddress",
				Connected_subaddress );
  capi20wrapper_printf_Struct ( "LLC",
				LLC );
  
  wasTheTrySuccessful = 1;
  
  unsigned int result = capi20wrapper_put_connect_active_resp ( ApplID,
								Message_number,
								PLCI );
  printf ( "capi20wrapper_put_connect_active_resp (...) = 0x%04x\n",
	   result );
  
  result = capi20wrapper_put_disconnect_req ( ApplID,
					      0x1234,
					      PLCI,
					      NULL );
  printf ( "capi20wrapper_put_connect_active_resp (...) = 0x%04x\n",
	   result );
  
  return result;
}

static
unsigned int get_disconnect_ind ( unsigned int ApplID,
				  capi20wrapper_Word_t Message_number,
				  capi20wrapper_DWord_t PLCI,
				  capi20wrapper_Word_t Reason )
{
  printf ( "--------------------------------\n" );
  printf ( "got a DISCONNECT_IND\n" );
  printf ( "PLCI = 0x%08lx\n",
	   PLCI );
  printf ( "Reason = 0x%04x\n",
	   Reason );
  
  printf ( "Terminating the process.\n" );
  didTheSequenceFinish = 1;
  
  unsigned int result = capi20wrapper_put_disconnect_resp ( ApplID,
							    Message_number,
							    PLCI );
  printf ( "capi20wrapper_put_disconnect_resp (...) = 0x%04x\n",
	   result );
  return result;
}

static
void get_unknown_message ( unsigned int ApplID,
			   capi20wrapper_Word_t Message_number,
			   capi20wrapper_Byte_t Command,
			   capi20wrapper_Byte_t Subcommand )
{
  printf ( "--------------------------------\n" );
  printf ( "got an unknown message\n" );
  printf ( "Command = 0x%02x\n",
	   Command );
  printf ( "Subommand = 0x%02x\n",
	   Subcommand );
  
  printf ( "Terminating the process.\n" );
  didTheSequenceFinish = 1;
}

static
capi20wrapper_Callbacks_t theCallbacks = {
  .get_connect_active_ind = get_connect_active_ind,
  .get_connect_conf       = get_connect_conf,
  .get_disconnect_ind     = get_disconnect_ind,
  .get_unknown_message    = get_unknown_message
};

int
main ()
{
  unsigned int const isinstalled = capi20_isinstalled ();

  printf ( "capi20_isinstalled () = 0x%04x\n",
	   isinstalled );
  if ( 0x0000 != isinstalled )
    {
      printf ( "Oops! capi20 is not installed, terminating the process.\n" );
      return 1;
    }

  unsigned int ApplID = 0;
  unsigned int result = 0x0000;
  
  const char * valuestring = NULL;
  
#define MAX_SIZE_CALLED_PARTY_NUMBER 32
  capi20wrapper_Byte_t mem_Called_party_number [ MAX_SIZE_CALLED_PARTY_NUMBER ] = {
    0x81
  };
  
  valuestring = getenv ( "CAPI20_Called_party_number" );
  if ( NULL != valuestring )
    {
      if ( ( MAX_SIZE_CALLED_PARTY_NUMBER - 1 ) < strlen ( valuestring ) )
	{
	  printf ( "Oops! CAPI20_Called_party_number is too long, terminating the process.\n" );
	  return 1;
	}
      memmove ( & mem_Called_party_number [ 1 ],
		valuestring,
		strlen ( valuestring ) );
    }
  
  capi20wrapper_Struct_t Called_party_number = {
    strlen ( valuestring ) + 1,
    mem_Called_party_number
  };
  
  capi20wrapper_DWord_t Controller = 0x00000001;
  
  valuestring = getenv ( "CAPI20_Controller" );
  if ( NULL != valuestring )
    {
      Controller = strtoul ( valuestring,
			     NULL,
			     16 );
    }
  
  capi20wrapper_DWord_t CIP_Value = 0x00000001;
  
  valuestring = getenv ( "CAPI20_CIP_Value" );
  if ( NULL != valuestring )
    {
      CIP_Value = strtoul ( valuestring,
			    NULL,
			    16 );
    }
  
  result = capi20_register ( 1,
			     7,
			     2048,
			     & ApplID );
  printf ( "capi20_register (...) = 0x%04x\n",
	   result );
  if ( 0x0000 != result )
    {
      printf ( "Oops! capi20_register did not succeed, terminating the process.\n" );
      return 1;
    }
  printf ( "ApplID = %d\n",
	   ApplID );
  
  capi20wrapper_Struct_t B_protocol = {
    0,
    0
  };
  /* see ETS 300 083
   */
  capi20wrapper_Byte_t mem_BC [] = { 0x80, 0x90, 0xa3 };
  capi20wrapper_Struct_t BC = {
    sizeof ( mem_BC ),
    mem_BC
  };
  capi20wrapper_Byte_t mem_LLC [] = { 0x80, 0x90 };
  capi20wrapper_Struct_t LLC = {
    sizeof ( mem_LLC ),
    mem_LLC
  };
  capi20wrapper_Byte_t mem_HLC [] = { 0x91, 0x81 };
  capi20wrapper_Struct_t HLC = {
    sizeof ( mem_HLC ),
    mem_HLC
  };
  capi20wrapper_Byte_t mem_Additional_Info [] = { 0x00, 0x00, 0x00, 0x00, 0x00 };
  capi20wrapper_Struct_t Additional_Info = {
    sizeof ( mem_Additional_Info ),
    mem_Additional_Info
  };
  
  result = capi20wrapper_put_connect_req ( ApplID,
					   0x0077,
					   Controller,
					   CIP_Value,
					   & Called_party_number,
					   NULL,
					   NULL,
					   NULL,
					   & B_protocol,
					   & BC,
					   & LLC,
					   & HLC,
					   & Additional_Info );
  printf ( "capi20wrapper_put_connect_req (...) = 0x%04x\n",
	   result );
  
  /* Now we wait for messages and handle them.
   */
  while ( 0x0000 == result
	  && ! didTheSequenceFinish )
    {
      result = capi20_waitformessage ( ApplID,
				       0 );
      printf ( "capi20_waitformessage (...) = 0x%04x\n",
	       result );
      if ( 0x0000 == result )
	{
	  result = capi20wrapper_get_message ( ApplID,
					       & theCallbacks );
	  printf ( "capi20wrapper_get_message (...) = 0x%04x\n",
		   result );
	}
    }
  
  result = capi20_release ( ApplID );
  printf ( "capi20_release (...) = 0x%04x\n",
	   result );
  
  if ( wasTheTrySuccessful )
    {
      return 0;
    }
  else
    {
      return 1;
    }
}
