/* PairMap.java -- A map using a heterogenous pair as the key.
 Copyright (C) 2005  The University of Sheffield.

 This file is part of the CASheW-s editor.

 The CASheW-s editor is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.
 
 The CASheW-s editor is distributed in the hope that it will be useful, but
 WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with The CASheW-s editor; see the file COPYING.  If not, write to the
 Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 02111-1307 USA.
*/

package nongnu.cashews.commons;

import java.util.HashMap;

/**
 * A map of a key, consisting of a pair of values, to a value.
 * 
 * @author Andrew John Hughes (gnu_andrew@member.fsf.org)
 * @see java.util.Map
 */
public class PairMap<A,B,V>
  extends HashMap<Pair<A,B>,V>
{

  /**
   * Serialization ID.
   */
  private static final long serialVersionUID = -3133725540569322674L;

  /**
   * Constructs an empty <code>PairMap</code> with the default initial
   * capacity (16) and the default load factor (0.75).
   */
  public PairMap()
  {
    this(16);
  }

  /**
   * Constructs an empty <code>PairMap</code> using the supplied initial
   * capacity and the default load factor (0.75).
   *
   * @param initialCapacity the initial capacity of the map.
   */
  public PairMap(int initialCapacity)
  {
    this(initialCapacity, 0.75f);
  }

  /**
   * Constructs an empty <code>PairMap</code> using the supplied initial
   * capacity and the supplied load factor.
   *
   * @param initialCapacity the initial capacity of the map.
   * @param loadFactor the load factor of the map.
   */
  public PairMap(int initialCapacity, float loadFactor)
  {
    super(initialCapacity, loadFactor);
  }

  /**
   * Constructs a <code>PairMap</code> using the contents of the supplied
   * pair map.
   *
   * @param m the map whose contents are to be used as the start of the
   *          contents for the new map.
   */
  public PairMap(PairMap<A, B, ? extends V> m)
  {
    super(m);
  }

  /**
   * Returns true if the map contains a mapping for the specified key.
   * The key is constructed as the pair of the two supplied values.
   *
   * @param left the left-hand value of the key.
   * @param right the right-hand value of the key.
   * @return true if the map contains a mapping for the key.
   */
  public boolean containsKey(A left, B right)
  {
    return super.containsKey(new Pair<A,B>(left,right));
  }

  /**
   * Retrieves the value associated with the given key.  The key
   * is constructed as the pair of the two supplied values.
   *
   * @param left the left-hand value of the key.
   * @param right the right-hand value of the key.
   * @return the value associated with the key.
   */
  public V get(A left, B right)
  {
    return super.get(new Pair<A,B>(left,right));
  }

  /**
   * Adds a mapping to the specified value from the supplied key.  The key
   * is constructed as the pair of the two supplied values.
   *
   * @param left the left-hand value of the key.
   * @param right the right-hand value of the key.
   * @param value the value which maps from the given key.
   * @return the previous value assigned to the supplied key, or
   *         <code>null</code> if there was no previous value.
   *         <code>null</code> may also be returned if the map allows
   *         <code>null</code> values. 
   */
  public V put(A left, B right, V value)
  {
    return super.put(new Pair<A,B>(left,right), value);
  }

  /**
   * Removes the mapping associated with the specified key.  The key
   * is constructed as the pair of the two supplied values.
   *
   * @param left the left-hand value of the key.
   * @param right the right-hand value of the key.
   * @return the previous value assigned to the supplied key, or
   *         <code>null</code> if there was no previous value.
   *         <code>null</code> may also be returned if the map allows
   *         <code>null</code> values. 
   */
  public V remove(A left, B right)
  {
    return super.remove(new Pair<A,B>(left,right));
  }

}
