/* DiagramEditor.java -- Representation of the diagram editor.
   Copyright (C) 2005  The University of Sheffield.

This file is part of the CASheW-s editor Eclipse plug-in.

The CASheW-s editor Eclipse plug-in is free software; you may copy, modify,
and redistribute it under the terms of the GNU General Public License
version 2 (or, at your option, any later version), and/or the Eclipse
Public License version 1.0.
 
The CASheW-s editor is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with The CASheW-s editor; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
02111-1307 USA.

The University of Sheffield makes available all content in this plug-in
("Content"). Unless otherwise indicated below, the Content is provided to
you under the terms and conditions of the Eclipse Public License Version
1.0 ("EPL"). A copy of the EPL is available at
http://www.eclipse.org/legal/epl-v10.html. For purposes of the EPL,
"Program" will mean the Content.

If you did not receive this Content directly from the University of Sheffield,
the Content is being redistributed by another party ("Redistributor") and
different terms and conditions may apply to your use of any object code in
the Content. Check the Redistributor's license that was provided with the
Content. If no such license exists, contact the Redistributor. Unless
otherwise indicated below, the terms and conditions of the EPL still apply
to any source code in the Content.

*/

package nongnu.cashews.eclipse.composer.ui;

import java.io.InputStream;
import java.util.EventObject;

import nongnu.cashews.eclipse.composer.dnd.DiagramTemplateTransferDropTargetListener;
import nongnu.cashews.eclipse.composer.model.Diagram;
import nongnu.cashews.eclipse.composer.model.ElementFactory;
import nongnu.cashews.eclipse.composer.parts.PartFactory;
import nongnu.cashews.eclipse.composer.parts.TreePartFactory;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.gef.DefaultEditDomain;
import org.eclipse.gef.KeyHandler;
import org.eclipse.gef.KeyStroke;
import org.eclipse.gef.dnd.TemplateTransferDragSourceListener;
import org.eclipse.gef.dnd.TemplateTransferDropTargetListener;
import org.eclipse.gef.editparts.ScalableFreeformRootEditPart;
import org.eclipse.gef.palette.PaletteRoot;
import org.eclipse.gef.requests.CreationFactory;
import org.eclipse.gef.ui.actions.ActionRegistry;
import org.eclipse.gef.ui.actions.GEFActionConstants;
import org.eclipse.gef.ui.parts.ContentOutlinePage;
import org.eclipse.gef.ui.parts.GraphicalEditorWithPalette;
import org.eclipse.gef.ui.parts.GraphicalViewerKeyHandler;
import org.eclipse.gef.ui.parts.TreeViewer;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.eclipse.ui.dialogs.SaveAsDialog;
import org.eclipse.ui.part.FileEditorInput;
import org.eclipse.ui.part.IPageSite;
import org.eclipse.ui.part.PageBook;
import org.eclipse.ui.views.contentoutline.IContentOutlinePage;

public class DiagramEditor extends GraphicalEditorWithPalette
{

  private PaletteRoot paletteRoot;

  private KeyHandler sharedKeyHandler;

  private boolean savePreviouslyNeeded;

  private Diagram diagram;

  public DiagramEditor()
  {
    diagram = new Diagram();
    // an EditDomain is a "session" of editing which contains things
    // like the CommandStack
    setEditDomain(new DefaultEditDomain(this));
    System.out.println("Diagram Editor created");
  }

  public Diagram getDiagram()
  {
    return this.diagram;
  }

  protected KeyHandler getCommonKeyHandler()
  {
    if (sharedKeyHandler == null)
      {
        sharedKeyHandler = new KeyHandler();
        sharedKeyHandler.put(
                             KeyStroke.getPressed(SWT.DEL, 127, 0),
                             getActionRegistry().getAction(
                                                           GEFActionConstants.DELETE));
      }
    return sharedKeyHandler;
  }

  // ------------------------------------------------------------------------
  // Overridden from GraphicalEditor

  protected void configureGraphicalViewer()
  {
    super.configureGraphicalViewer();
    getGraphicalViewer().setRootEditPart(new ScalableFreeformRootEditPart());

    // set the factory to use for creating EditParts for elements in the model
    getGraphicalViewer().setEditPartFactory(new PartFactory());

    getGraphicalViewer().setKeyHandler(
                                       new GraphicalViewerKeyHandler(
                                                                     getGraphicalViewer()).setParent(getCommonKeyHandler()));
  }

  public void commandStackChanged(EventObject event)
  {
    if (isDirty())
      {
        if (!this.savePreviouslyNeeded)
          {
            this.savePreviouslyNeeded = true;
            firePropertyChange(IEditorPart.PROP_DIRTY);
          }
      }
    else
      {
        savePreviouslyNeeded = false;
        firePropertyChange(IEditorPart.PROP_DIRTY);
      }
    super.commandStackChanged(event);
  }

  public Object getAdapter(Class type)
  {
    if (type == IContentOutlinePage.class)
      return new OutlinePage();
    return super.getAdapter(type);
  }

  // ------------------------------------------------------------------------
  // Overridden from EditorPart

  protected void setInput(IEditorInput input)
  {
    super.setInput(input);

    IFile file = ((IFileEditorInput) input).getFile();
    try
      { // attempt to read from a file
        InputStream istream = file.getContents(false);
        diagram = Diagram.makeFromStream(istream);
      }
    catch (Exception e)
      { // but if there's an error, create a new diagram
        e.printStackTrace();
        diagram = new Diagram();
      }
  }

  // ------------------------------------------------------------------------
  // Abstract methods from GraphicalEditor

  protected void initializeGraphicalViewer()
  {
    // this uses the PartFactory set in configureGraphicalViewer
    // to create an EditPart for the diagram and sets it as the
    // content for the viewer
    getGraphicalViewer().setContents(this.diagram);

    getGraphicalViewer().addDropTargetListener(
                                               new TemplateTransferDropTargetListener(getGraphicalViewer())
					       {
						 protected CreationFactory getFactory(Object template)
						 {
						   return new ElementFactory(template);
						 }
					       });
  }

  // ------------------------------------------------------------------------
  // Overridden methods from GraphicalEditorWithPalette

  protected void initializePaletteViewer()
  {
    super.initializePaletteViewer();
    getPaletteViewer().addDragSourceListener(
                                             new TemplateTransferDragSourceListener(
                                                                                    getPaletteViewer()));
  }

  // ------------------------------------------------------------------------
  // Abstract methods from EditorPart

  public void doSave(IProgressMonitor monitor)
  {
    try
      {
        IFile file = ((IFileEditorInput) getEditorInput()).getFile();
        //file.setContents(this.diagram.getAsStream(), true, false, monitor);
        file.setContents(this.diagram.getAsStream(), true, false, monitor);
        getCommandStack().markSaveLocation();
      }
    catch (Exception e)
      {
        e.printStackTrace();
      }
  }

  public void doSaveAs()
  {
    SaveAsDialog dialog = new SaveAsDialog(
                                           getSite().getWorkbenchWindow().getShell());
    dialog.setOriginalFile(((IFileEditorInput) getEditorInput()).getFile());
    dialog.open();
    IPath path = dialog.getResult();

    if (path == null)
      return;

    IWorkspace workspace = ResourcesPlugin.getWorkspace();
    final IFile file = workspace.getRoot().getFile(path);

    WorkspaceModifyOperation op = new WorkspaceModifyOperation()
    {
      public void execute(final IProgressMonitor monitor) throws CoreException
      {
        try
          {
            file.create(diagram.getAsStream(), true, monitor);
          }
        catch (Exception e)
          {
            e.printStackTrace();
          }
      }
    };

    try
      {
        new ProgressMonitorDialog(getSite().getWorkbenchWindow().getShell()).run(
                                                                                 false,
                                                                                 true,
                                                                                 op);
        setInput(new FileEditorInput((IFile) file));
        getCommandStack().markSaveLocation();
      }
    catch (Exception e)
      {
        e.printStackTrace();
      }
  }

  public void gotoMarker(IMarker marker)
  {
  }

  public boolean isDirty()
  {
    // rely on the command stack to determine dirty flag
    return getCommandStack().isDirty();
  }

  public boolean isSaveAsAllowed()
  {
    // allow Save As
    return true;
  }

  // ------------------------------------------------------------------------
  // Abstract methods from GraphicalEditorWithPalette

  protected PaletteRoot getPaletteRoot()
  {
    if (this.paletteRoot == null)
      {
        this.paletteRoot = PaletteFactory.createPalette();
      }
    return this.paletteRoot;
  }

  // ========================================================================
  // Inner Classes

  public class OutlinePage extends ContentOutlinePage
  {

    private PageBook pageBook;

    private Control outline;

    public OutlinePage()
    {
      super(new TreeViewer());
    }

    public void init(IPageSite pageSite)
    {
      super.init(pageSite);
      ActionRegistry registry = getActionRegistry();
      IActionBars bars = pageSite.getActionBars();
      String id = IWorkbenchActionConstants.UNDO;
      bars.setGlobalActionHandler(id, registry.getAction(id));
      id = IWorkbenchActionConstants.REDO;
      bars.setGlobalActionHandler(id, registry.getAction(id));
      id = IWorkbenchActionConstants.DELETE;
      bars.setGlobalActionHandler(id, registry.getAction(id));
      bars.updateActionBars();
    }

    public void createControl(Composite parent)
    {
      pageBook = new PageBook(parent, SWT.NONE);
      outline = getViewer().createControl(pageBook);
      pageBook.showPage(outline);
      getSelectionSynchronizer().addViewer(getViewer());
      getViewer().setEditDomain(getEditDomain());
      getViewer().setEditPartFactory(new TreePartFactory());
      getViewer().setKeyHandler(getCommonKeyHandler());
      getViewer().setContents(getDiagram());
    }

    public Control getControl()
    {
      return this.pageBook;
    }

    public void dispose()
    {
      getSelectionSynchronizer().removeViewer(getViewer());
      super.dispose();
    }
  }
}
