/* SoapOperation.java -- Representation of a SOAP operation.
 Copyright (C) 2005  The University of Sheffield.

 This file is part of the CASheW-s editor.

 The CASheW-s editor is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.
 
 The CASheW-s editor is distributed in the hope that it will be useful, but
 WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with The CASheW-s editor; see the file COPYING.  If not, write to the
 Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 02111-1307 USA.
*/

package nongnu.cashews.language.grounding;

import java.net.URI;
import java.net.URISyntaxException;

import javax.xml.namespace.QName;

import nongnu.cashews.xml.XmlField;
import nongnu.cashews.xml.Xmlizable;

/**
 * An implementation of <code>Grounding</code> for the Simple
 * Object Access Protocol (SOAP).  SOAP operations have an
 * endpoint, a namespace, an input message and an output
 * message, which specify the location, naming and data
 * transmission of the operation respectively.
 *
 * @author Andrew John Hughes (gnu_andrew@member.fsf.org)
 * @see Grounding
 */
public class SoapOperation
  implements Grounding, Xmlizable
{

  /**
   * Serialization fields which specify how to serialize this class.
   */
  private static final XmlField[] serialPersistentFields
    = new XmlField[]
    {
      new XmlField("endpoint",URI.class),
      new XmlField("namespace",URI.class),
      new XmlField("inputMessage",SoapMessage.class,false,true,false),
      new XmlField("outputMessage",SoapMessage.class,false,true,false)     
    };

  /**
   * The endpoint where the operation is located.
   *
   * @serial the endpoint of the operation.
   */
  private URI endpoint;

  /**
   * The namespace of this operation.
   *
   * @serial the operation namespace.
   */
  private URI namespace;

  /**
   * The message format for the input.
   *
   * @serial the input message format.
   */
  private SoapMessage inputMessage;

  /**
   * The message format for the output.
   *
   * @serial the output message format.
   */
  private SoapMessage outputMessage;

  /**
   * Constructs a new empty operation.
   */
  public SoapOperation()
  {
  }

  /**
   * Constructs a new operation using the specified endpoint.
   * The endpoint is also used as the namespace of the operation.
   *
   * @param endpoint the endpoint of this SOAP operation.
   * @throws URISyntaxException if the supplied endpoint is not a valid URI.
   */
  public SoapOperation(String endpoint)
    throws URISyntaxException
  {
    this(endpoint, endpoint);
  }

  /**
   * Constructs a new operation using the specified endpoint
   * and namespace.
   *
   * @param endpoint the endpoint of this SOAP operation.
   * @param namespace the namespace of this SOAP operation.
   * @throws URISyntaxException if the supplied endpoint is not a valid URI.
   */
  public SoapOperation(String endpoint, String namespace)
    throws URISyntaxException
  {
    setEndpoint(endpoint);
    setNamespace(namespace);
  }

  /**
   * Sets the endpoint of this SOAP operation to that specified.
   *
   * @param endpoint the endpoint of the operation.
   * @throws URISyntaxException if the supplied endpoint is not a valid URI.
   */
  public void setEndpoint(String endpoint)
    throws URISyntaxException
  {
    setEndpoint(new URI(endpoint));
  }

  /**
   * Sets the endpoint of this SOAP operation to that specified.
   *
   * @param endpoint the endpoint of the operation.
   */
  public void setEndpoint(URI endpoint)
  {
    this.endpoint = endpoint;
  }

  /**
   * Sets the namespace of this SOAP operation to that specified.
   *
   * @param namespace the namespace of the operation.
   * @throws URISyntaxException if the supplied namespace is not a valid URI.
   */
  public void setNamespace(String namespace)
    throws URISyntaxException
  {
    setNamespace(new URI(namespace));
  }

  /**
   * Sets the namespace of this SOAP operation to that specified.
   *
   * @param namespace the namespace of the operation.
   */
  public void setNamespace(URI namespace)
  {
    this.namespace = namespace;
  }

  /**
   * Sets the input message of this SOAP operation to that specified.
   *
   * @param inputMessage the input message of this operation.
   */
  public void setInputMessage(SoapMessage inputMessage)
  {
    this.inputMessage = inputMessage;
  }

  /**
   * Sets the output message of this SOAP operation to that specified.
   *
   * @param outputMessage the output message of this operation.
   */
  public void setOutputMessage(SoapMessage outputMessage)
  {
    this.outputMessage = outputMessage;
  }

  /**
   * Retrieves the endpoint of this operation.
   *
   * @return the operation endpoint.
   */
  public URI getEndpoint()
  {
    return endpoint;
  }

  /**
   * Retrieves the input message of this operation.
   *
   * @return the operation's input message.
   */
  public SoapMessage getInputMessage()
  {
    return inputMessage;
  }

  /**
   * Returns a <code>String</code> representation of this SOAP operation.
   *
   * @return a textual representation.
   */
  public String toString()
  {
    return getClass().getName() +
      "[endpoint=" +
      endpoint +
      ",namespace=" +
      namespace +
      ",inputMessage=" +
      inputMessage +
      ",outputMessage=" +
      outputMessage +
      "]";
  }

  /**
   * Returns "soapOperation" as the element name.
   *
   * @return <code>soapOperation</code>
   */
  public String getElementName()
  {
    return "soapOperation";
  }

  /**
   * Retrieves an array of QNames which specifies the namespaces to
   * declare when serializing this element as XML.
   *
   * @return an array of QNames for namespace declaration.
   */
  public QName[] getDeclaredNamespaces()
  {
    if (inputMessage != null)
      {
	return new QName[]
	  {
	    inputMessage.getName()
	  };
      }
    else
      return null;
  }

}

