/* Consume.java -- Consumes input for use in a composite process.
 Copyright (C) 2005  The University of Sheffield.

 This file is part of the CASheW-s editor.

 The CASheW-s editor is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.
 
 The CASheW-s editor is distributed in the hope that it will be useful, but
 WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with The CASheW-s editor; see the file COPYING.  If not, write to the
 Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 02111-1307 USA.
*/

package nongnu.cashews.language.process;

import java.io.Serializable;

import java.net.URI;
import java.net.URISyntaxException;

/**
 * Consumes input for use in a <code>CompositeProcess</code>.
 * The consumer associates the inputs of the composite process
 * with a particular performance inside the process.
 *
 * @author Andrew John Hughes (gnu_andrew@member.fsf.org)
 * @see Performance
 */
public class Consume
  implements Serializable
{

  /**
   * The input on the composite process.
   *
   * @serial the source input.
   */
  private URI fromInput;

  /**
   * The internal performance where the input goes.
   *
   * @serial the destination performance.
   */
  private URI toPerformance;

  /**
   * The input on the destination performance.
   *
   * @serial the destination input.
   */
  private URI toInput;

  /**
   * The index of the input on the destination performance.
   *
   * @serial the index of the destination input.
   */
  private int toIndex;

  /**
   * Construct a new <code>Consume</code> with the specified
   * destination performance, source and destination inputs,
   * along with the supplied index.
   *
   * @param fromInput the input on the composite process.
   * @param toPerformance the performance to send the input to.
   * @param toInput the input on the destination performance.
   * @param toIndex the index on the input on the destination performance.
   * @throws URISyntaxException if one of the supplied names is not a valid
   *                            URI.
   */
  public Consume(String fromInput, String toPerformance, String toInput,
		 int toIndex)
    throws URISyntaxException
  {
    setFromInput(fromInput);
    setToPerformance(toPerformance);
    setToInput(toInput);
    setToIndex(toIndex);
  }

  /**
   * Construct a new <code>Consume</code> with the specified
   * destination performance, source and destination inputs,
   * along with the supplied index.
   *
   * @param fromInput the input on the composite process.
   * @param toPerformance the performance to send the input to.
   * @param toInput the input on the destination performance.
   * @param toIndex the index on the input on the destination performance.
   */
  public Consume(URI fromInput, URI toPerformance, URI toInput,
		 int toIndex)
  {
    setFromInput(fromInput);
    setToPerformance(toPerformance);
    setToInput(toInput);
    setToIndex(toIndex);
  }

  /**
   * Sets the source input on the composite process to that specified.
   *
   * @param fromInput the new source input on the composite process.
   * @throws URISyntaxException if the supplied input is not a valid URI.
   */
  public void setFromInput(String fromInput)
    throws URISyntaxException
  {
    setFromInput(new URI(fromInput));
  }

  /**
   * Sets the source input on the composite process to that specified.
   *
   * @param fromInput the new source input on the composite process.
   */
  public void setFromInput(URI fromInput)
  {
    this.fromInput = fromInput;
  }

  /**
   * Sets the destination performance to that specified.
   *
   * @param toPerformance the new destination performance.
   * @throws URISyntaxException if the supplied performance is not a valid URI.
   */
  public void setToPerformance(String toPerformance)
    throws URISyntaxException
  {
    setToPerformance(new URI(toPerformance));
  }

  /**
   * Sets the destination performance to that specified.
   *
   * @param toPerformance the new destination performance.
   */
  public void setToPerformance(URI toPerformance)
  {
    this.toPerformance = toPerformance;
  }

  /**
   * Sets the destination input on the performance to that specified.
   *
   * @param toInput the new destination input on the performance.
   * @throws URISyntaxException if the supplied input is not a valid URI.
   */
  public void setToInput(String toInput)
    throws URISyntaxException
  {
    setToInput(new URI(toInput));
  }

  /**
   * Sets the destination input on the performance to that specified.
   *
   * @param toInput the new destination input on the performance.
   */
  public void setToInput(URI toInput)
  {
    this.toInput = toInput;
  }

  /**
   * Sets the index of the input on the performance to that specified.
   * 
   * @param toIndex the new index of the input on the performance.
   */
  public void setToIndex(int toIndex)
  {
    this.toIndex = toIndex;
  }

  /**
   * Returns a <code>String</code> representation of this performance.
   *
   * @return a textual representation.
   */
  public String toString()
  {
    return getClass().getName() +
      "[fromInput=" +
      fromInput +
      ",toPerformance=" +
      toPerformance +
      ",toInput=" +
      toInput +
      ",toIndex=" +
      toIndex +
      "]";
  }

}

