/* LogicLanguage.java -- Representation of an OWL-S logic language.
 Copyright (C) 2005  The University of Sheffield.

 This file is part of the CASheW-s editor.

 The CASheW-s editor is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.
 
 The CASheW-s editor is distributed in the hope that it will be useful, but
 WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with The CASheW-s editor; see the file COPYING.  If not, write to the
 Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 02111-1307 USA.
*/

package nongnu.cashews.owls.expression;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.List;

/**
 * This class represents a particular logical formalism, such as KIF, DRS or
 * SWRL. At present, it simply allows for the specification of zero or more
 * URIs, which point to the documentation for the formalism.
 * 
 * @author Andrew John Hughes (gnu_andrew@member.fsf.org)
 */
public class LogicLanguage
{

  /**
   * A representation of the logical formalism, SWRL, as defined by the W3C.
   */
  public static final LogicLanguage SWRL = new SWRL();

  /**
   * A representation of the logical formalism, DRS, as defined by the DAML
   * team.
   */
  public static final LogicLanguage DRS = new DRS();

  /**
   * A representation of the logical formalism, KIF, as defined by Stanford
   * University.
   */
  public static final LogicLanguage KIF = new KIF();

  /**
   * A list of <code>URI</code> s which specify the location of the
   * documentation for this formalism.
   * 
   * @serial the URIs of this formalism's documentation.
   * @see java.net.URI
   */
  private List<URI> uris;

  /**
   * The default constructor for a <code>LogicLanguage</code> instance.
   */
  public LogicLanguage()
  {
    uris = new ArrayList<URI>();
  }

  /**
   * Adds a <code>URI</code> to the list specifying the documentation for this
   * formalism.
   * 
   * @param uri the URI to add.
   * @see java.net.URI
   */
  public void addURI(URI uri)
  {
    uris.add(uri);
  }

  /**
   * Returns the list of <code>URI</code> s for this logic formalism.
   * 
   * @return a cloned list of the <code>URI</code> s of this formalism.
   */
  public List<URI> getURIs()
  {
    List<URI> clonedURIs = new ArrayList<URI>();
    for (URI uri : uris)
      {
        try
          {
            URI newURI = new URI(uri.toString());
            clonedURIs.add(newURI);
          }
        catch (URISyntaxException e)
          {
            throw new IllegalStateException("The list of URIs includes "
                                            + "an invalid URI.", e);
          }
      }
    return clonedURIs;
  }

  /**
   * A specific subclass of <code>LogicLanguage</code>, which represents the
   * SWRL logic formalism. Instances of this are not modifiable.
   * 
   * @author Andrew John Hughes (gnu_andrew@member.fsf.org)
   * @see LogicLanguage
   */
  private static final class SWRL extends LogicLanguage
  {

    /**
     * The default constructor for the SWRL language.
     */
    public SWRL()
    {
    }

    /**
     * This method always throws an exception in this subclass in order to
     * prevent derivations of the SWRL formalism.
     * 
     * @throws UnsupportedOperationException as the formalism can
     *         not be modified.
     */
    public void addURI(URI uri)
    {
      throw new UnsupportedOperationException("The SWRL logic language "
                                              + "instance is not modifable.");
    }

    /**
     * Returns the list of <code>URI</code> s for the SWRL logic formalism.
     * 
     * @return the SWRL <code>URI</code>s.
     */
    public List<URI> getURIs()
    {
      List<URI> swrl = new ArrayList<URI>();
      try
        {
          swrl.add(new URI("http://www.w3.org/2003/11/swrl"));
        }
      catch (URISyntaxException e)
        {
          throw new IllegalStateException("The SWRL URL is invalid.", e);
        }
      return swrl;
    }

  }

  /**
   * A specific subclass of <code>LogicLanguage</code>, which represents the
   * DRS logic formalism. Instances of this are not modifiable.
   * 
   * @author Andrew John Hughes (gnu_andrew@member.fsf.org)
   * @see LogicLanguage
   */
  private static final class DRS extends LogicLanguage
  {

    /**
     * The default constructor for the DRS language, which stores the URIs for
     * its specification documents.
     */
    public DRS()
    {
    }

    /**
     * This method always throws an exception in this subclass in order to
     * prevent derivations of the DRS formalism.
     * 
     * @throws UnsupportedOperationException as the formalism can
     *         not be modified.
     */
    public void addURI(URI uri)
    {
      throw new UnsupportedOperationException("The DRS logic language "
                                              + "instance is not modifable.");
    }

    /**
     * Returns the list of <code>URI</code> s for the DRS logic formalism.
     * 
     * @return the DRS <code>URI</code>s.
     */
    public List<URI> getURIs()
    {
      List<URI> drs = new ArrayList<URI>();
      try
        {
          drs.add(new URI("http://www.daml.org/services/owl-s/1.1/"
                          + "generic/drs.owl"));
        }
      catch (URISyntaxException e)
        {
          throw new IllegalStateException("The DRS URL is invalid.", e);
        }
      return drs;
    }

  }

  /**
   * A specific subclass of <code>LogicLanguage</code>, which represents the
   * KIF logic formalism. Instances of this are not modifiable.
   * 
   * @author Andrew John Hughes (gnu_andrew@member.fsf.org)
   * @see LogicLanguage
   */
  private static final class KIF extends LogicLanguage
  {

    /**
     * The default constructor for the KIF language, which stores the URIs for
     * its specification documents.
     */
    public KIF()
    {
    }

    /**
     * This method always throws an exception in this subclass in order to
     * prevent derivations of the KIF formalism.
     * 
     * @throws UnsupportedOperationException as the formalism can not be
     *         modified.
     */
    public void addURI(URI uri)
    {
      throw new UnsupportedOperationException("The KIF logic language "
                                              + "instance is not modifable.");
    }

    /**
     * Returns the list of <code>URI</code> s for the KIF logic formalism.
     * 
     * @return the KIF <code>URI</code>s.
     */
    public List<URI> getURIs()
    {
      List<URI> kif = new ArrayList<URI>();
      try
        {
          kif.add(new URI("http://logic.stanford.edu/kif/kif.html"));
        }
      catch (URISyntaxException e)
        {
          throw new IllegalStateException("The KIF URL is invalid.", e);
        }
      return kif;
    }

  }

}
