/* Graph.java -- Representation of a RDF graph.
 Copyright (C) 2005  The University of Sheffield.

 This file is part of the CASheW-s editor.

 The CASheW-s editor is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.
 
 The CASheW-s editor is distributed in the hope that it will be useful, but
 WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with The CASheW-s editor; see the file COPYING.  If not, write to the
 Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 02111-1307 USA.
*/

package nongnu.cashews.rdf;

import java.io.Serializable;
import java.util.HashSet;
import java.util.Set;

/**
 * This class represents an RDF graph, formally described as a set of RDF
 * triples. The graph is made up of nodes (the subjects and objects of the
 * triples) and the links between these nodes (the predicates of the triples).
 * 
 * @author Andrew John Hughes (gnu_andrew@member.fsf.org)
 * @see Triple
 * @see Node
 * @see Subject
 * @see RDFObject
 * @see Predicate
 */
public class Graph 
  implements Serializable
{

  /**
   * Serialization UID for compatability.
   */
  private static final long serialVersionUID = -4009151165788339408L;

  /**
   * The set of RDF triples which make up this particular RDF 
   * <code>Graph</code> instance.
   * 
   * @see Triple
   * @serial the graph of RDF triples.
   */
  private Set<Triple> graph;

  /**
   * Constructs a new empty graph.
   */
  public Graph()
  {
    graph = new HashSet<Triple>();
  }

  /**
   * Returns true if the specified object is either of type <code>Graph</code>,
   * or a sub-type, and contains an equivalent set of triples. If the specified
   * object is null, false is returned.
   * 
   * @param obj
   *          the object to compare this object with.
   * @return true if the two objects are equivalent.
   */
  public boolean equals(Object obj)
  {
    if (obj == null)
      return false;
    if (getClass() == obj.getClass())
      {
        Graph g = (Graph) obj;
        return graph.equals(g.getGraph());
      }
    else
      return false;
  }

  /**
   * Returns the hashcode of this RDF graph. This is the code for the 
   * underlying collection of <code>Triple</code>s.
   * 
   * @return the hashcode for the RDF graph.
   */
  public int hashCode()
  {
    return graph.hashCode();
  }

  /**
   * Returns a clone of the graph of RDF triples.
   * 
   * @return a clone of the RDF triple graph.
   */
  public Set<Triple> getGraph()
  {
    Set<Triple> clonedSet = new HashSet<Triple>();
    for (Triple triple : graph)
      clonedSet.add(triple.clone());
    return clonedSet;
  }

  /**
   * Adds a triple to the graph.
   *
   * @param triple the triple to add.
   */
  public void addTriple(Triple triple)
  {
    graph.add(triple);
  }

  /**
   * Returns a textual representation of the graph.
   *
   * @return a textual representation.
   */
  public String toString()
  {
    return getClass().getName() +
      "[graph = " +
      graph +
      "]";
  }
 
}
