/* Literal.java -- Representation of a RDF literal.
 Copyright (C) 2005  The University of Sheffield.

 This file is part of the CASheW-s editor.

 The CASheW-s editor is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.
 
 The CASheW-s editor is distributed in the hope that it will be useful, but
 WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with The CASheW-s editor; see the file COPYING.  If not, write to the
 Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 02111-1307 USA.
*/

package nongnu.cashews.rdf;

/**
 * <p>
 * This class represents an RDF literal. A literal has at least a lexical form.
 * Untyped literals include an optional lower-case language tag, taken from the
 * values in RFC3066. Typed literals instead include a datatype URI, which
 * specifies the type of the data used in the literal.
 * </p>
 * 
 * @author Andrew John Hughes (gnu_andrew@member.fsf.org)
 * @see Type
 */
public class Literal 
  implements RDFObject
{

  /**
   * The lexical form of the literal.
   * 
   * @serial the literal's lexical form. This may not be null.
   */
  private String lexicalForm;

  /**
   * The optional language tag for this literal.
   * 
   * @serial a lower-case RFC3066 language tag. May be null.
   */
  private String language;

  /**
   * The optional type of the literal.
   * 
   * @see Type
   * @serial the type of the literal's data. May be null.
   */
  private Type type;

  /**
   * Constructs a <code>Literal</code> using the specified lexical form.
   *
   * @param lexicalForm the lexical form of this literal.
   */
  public Literal(String lexicalForm)
  {
    this.lexicalForm = lexicalForm;
  }

  /**
   * Constructs a <code>Literal</code> using the specified lexical form
   * and type.
   *
   * @param lexicalForm the lexical form of this literal.
   * @param type the type of this literal.
   */
  public Literal(String lexicalForm, Type type)
  {
    this(lexicalForm);
    setType(type);
  }

  /**
   * Returns a textual representation of the literal.
   *
   * @return a textual representation.
   */
  public String toString()
  {
    return getClass().getName() +
      "[lexicalForm = " +
      lexicalForm +
      ", language = " +
      language +
      ", type = " +
      type + 
      "]";
  }

  /**
   * Returns a deep copy of this literal.
   *
   * @return a clone of the literal.
   */
  public Literal clone()
  {
    try
      {
	Object clonedObject = super.clone();
	Literal clone = (Literal) clonedObject;
	clone.setType(type.clone());
	return clone;
      }
    catch (CloneNotSupportedException e)
      {
	throw new IllegalStateException("Unexpected exception: " + e, e);
      }
  }

  /**
   * Sets the type used by the literal to the one specified.
   *
   * @param type the new type to use.
   */
  public void setType(Type type)
  {
    this.type = type;
  }

}
