/* Triple.java -- Representation of a RDF triple.
 Copyright (C) 2005  The University of Sheffield.

 This file is part of the CASheW-s editor.

 The CASheW-s editor is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.
 
 The CASheW-s editor is distributed in the hope that it will be useful, but
 WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with The CASheW-s editor; see the file COPYING.  If not, write to the
 Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 02111-1307 USA.
*/

package nongnu.cashews.rdf;

import java.io.Serializable;

/**
 * <p>
 * This class represents an RDF triple. Each triple consists of:
 * </p>
 * <ul>
 * <li>a <strong>subject </strong>, which is a URI reference or a blank node.
 * </li>
 * <li>a <strong>predicate </strong>, which is a URI reference.</li>
 * <li>an <strong>object </strong>, which is a URI reference, literal or blank
 * node.</li>
 * </ul>
 * <p>
 * A set of these triples form a <code>Graph</code> object.
 * 
 * @author Andrew John Hughes (gnu_andrew@member.fsf.org)
 * @see Graph
 * @see Node
 * @see Subject
 * @see RDFObject
 * @see Predicate
 */
public class Triple 
  implements Serializable, Cloneable
{

  /**
   * Serialization UID for compatability.
   */
  private static final long serialVersionUID = -6440582411545169192L;

  /**
   * The subject of this triple.
   * 
   * @see Subject
   * @serial the triple's subject.
   */
  private Subject subject;

  /**
   * The predicate of this triple.
   * 
   * @see Predicate
   * @serial the triple's predicate.
   */
  private Predicate predicate;

  /**
   * The object of this triple.
   * 
   * @see RDFObject
   * @serial the triple's object.
   */
  private RDFObject object;

  /**
   * Constructs a new <code>Triple</code> using the specified
   * subject, predicate and object.
   *
   * @param subject the new subject.
   * @param predicate the new predicate.
   * @param object the new object.
   */
  public Triple(Subject subject, Predicate predicate, RDFObject object)
  {
    this.subject = subject;
    this.predicate = predicate;
    this.object = object;
  }

  /**
   * Returns a textual representation of the triple.
   *
   * @return a textual representation.
   */
  public String toString()
  {
    return getClass().getName() +
      "[subject = " +
      subject +
      ", predicate = " +
      predicate +
      ", object = " +
      object +
      "]";
  }

  /**
   * Returns a deep copy of this triple.
   *
   * @return a clone of the triple.
   */
  public Triple clone()
  {
    try
      {
	Object clonedObject = super.clone();
	Triple clone = (Triple) clonedObject;
	clone.setSubject(subject.clone());
	clone.setPredicate(predicate.clone());
	clone.setObject(object.clone());
	return clone;
      }
    catch (CloneNotSupportedException e)
      {
	throw new IllegalStateException("Unexpected exception: " + e, e);
      }
  }

  /**
   * Sets the subject of this triple to that specified.
   *
   * @param subject the new subject for this triple.
   */
  public void setSubject(Subject subject)
  {
    this.subject = subject;
  }

  /**
   * Sets the predicate of this triple to that specified.
   *
   * @param predicate the new predicate for this triple.
   */
  public void setPredicate(Predicate predicate)
  {
    this.predicate = predicate;
  }

  /**
   * Sets the object of this triple to that specified.
   *
   * @param object the new object for this triple.
   */
  public void setObject(RDFObject object)
  {
    this.object = object;
  }

  /**
   * Returns true if the specified triple is equal to this one.
   *
   * @param object the object to compare.
   */
  public boolean equals(Object obj)
  {
    if (obj == null)
      return false;
    if (obj == this)
      return true;
    if (obj.getClass() == getClass())
      {
	Triple triple = (Triple) obj;
	return subject.equals(triple.getSubject()) 
	  && predicate.equals(triple.getPredicate())
	  && object.equals(triple.getObject());
      }
    return false;
  }

  /**
   * Returns the hashcode of this triple. This is calculated using
   * the hash codes of the subject, predicate and object.
   * 
   * @return the hashcode for the triple.
   */
  public int hashCode()
  {
    return super.hashCode() 
      + 13 * subject.hashCode()
      + 17 * predicate.hashCode()
      + 19 * object.hashCode();
  }

  /**
   * Returns a clone of the triple's subject.
   *
   * @return a clone of the subject.
   */
  public Subject getSubject()
  {
    return subject.clone();
  }

  /**
   * Returns a clone of the triple's predicate.
   *
   * @return a clone of the predicate.
   */
  public Predicate getPredicate()
  {
    return predicate.clone();
  }

  /**
   * Returns a clone of the triple's object.
   *
   * @return a clone of the object.
   */
  public RDFObject getObject()
  {
    return object.clone();
  }

}
