/* TypeFactory.java -- Factory for generating type instances.
 Copyright (C) 2005  The University of Sheffield.

 This file is part of the CASheW-s editor.

 The CASheW-s editor is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.
 
 The CASheW-s editor is distributed in the hope that it will be useful, but
 WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with The CASheW-s editor; see the file COPYING.  If not, write to the
 Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 02111-1307 USA.
*/

package nongnu.cashews.rdf;

import java.lang.reflect.InvocationTargetException;
import java.util.HashMap;
import java.util.Map;

/**
 * <p>
 * This factory generates a type instance for a URI. Type instances provide
 * the conversion between the literal value and the actual value
 * represented, by applying a lexical-to-value mapping.  Unrecognised URIs
 * are mapped to an instance of <code>UnknownType</code>.
 * </p>
 * 
 * @author Andrew John Hughes (gnu_andrew@member.fsf.org)
 * @see UnknownType
 */
public class TypeFactory
{

  /**
   * The map of URI strings to classes implementing that type.
   */
  private static Map<String,Class<? extends Type>> typeClasses;

  /**
   * Static initializer to create the map of URIs to type classes.
   */
  static
  {
    typeClasses = new HashMap<String,Class<? extends Type>>();
  }

  /**
   * Private constructor to prevent class instantiation.
   */
  private TypeFactory()
  {
  }

  /**
   * Retrieve a <code>Type</code> instance corresponding to the supplied
   * URI.
   *
   * @param uri the URI to lookup, in <code>String</code> form.
   */
  public static Type getInstance(String URI)
  {
    Class<? extends Type> typeClass = typeClasses.get(URI);
    if (typeClass == null)
      return new UnknownType(URI);
    else
      try
	{
	  return typeClass.getConstructor().newInstance();
	}
      catch (NoSuchMethodException e)
	{
	  throw new IllegalArgumentException("There is no type for the "+
					     "specified URI with a "+
					     "no argument constructor:" + e,
					     e);
	}
      catch (IllegalAccessException e)
	{
	  throw new IllegalArgumentException("There is no type for the "+
					     "specified URI with an "+
					     "accessible constructor: " + e,
					     e);
	}
      catch (InstantiationException e)
	{
	  throw new IllegalArgumentException("A Type instance for this URI "+
					     "could not be instantiated: " + e,
					     e);
	}
      catch (InvocationTargetException e)
	{
	  throw new IllegalArgumentException("The constructor of the Type "+
					     "instance for this URI "+
					     "threw an exception: " + e,
					     e);
	}
  }

}
