/* Serializer.java -- Recursively serializes an object as XML.
 Copyright (C) 2005  The University of Sheffield.

 This file is part of the CASheW-s editor.

 The CASheW-s editor is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.
 
 The CASheW-s editor is distributed in the hope that it will be useful, but
 WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with The CASheW-s editor; see the file COPYING.  If not, write to the
 Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 02111-1307 USA.
*/

package nongnu.cashews.xml;

import java.io.ObjectStreamField;
import java.io.OutputStream;
import java.io.Serializable;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;

import java.net.URISyntaxException;

import java.util.Collection;

import static javax.xml.XMLConstants.XMLNS_ATTRIBUTE;
import static javax.xml.XMLConstants.XMLNS_ATTRIBUTE_NS_URI;
import static javax.xml.XMLConstants.W3C_XML_SCHEMA_NS_URI;

import javax.xml.namespace.QName;

import nongnu.cashews.commons.Pair;
import nongnu.cashews.commons.PairList;

import static nongnu.cashews.services.Processes.TEST_COMPOSITE_SEQUENCE;

import nongnu.cashews.xml.schema.TypeMapper;
import nongnu.cashews.xml.schema.XsdType;

import org.w3c.dom.Document;
import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Element;
import org.w3c.dom.bootstrap.DOMImplementationRegistry;
import org.w3c.dom.ls.DOMImplementationLS;
import org.w3c.dom.ls.LSOutput;
import org.w3c.dom.ls.LSSerializer;

/**
 * Generically serializes an object to an XML tree. 
 *
 * @author Andrew John Hughes (gnu_andrew@member.fsf.org)
 */
public class Serializer
{

  /**
   * Implementation of the DOM for use within this class.
   */
  private static DOMImplementation domImpl;

  /**
   * The document namespaces.
   */
  private static final QName[] DOCUMENT_NAMESPACES = new QName[]
    {
      new QName(W3C_XML_SCHEMA_NS_URI, "", "xsd")
    };

  /**
   * Initialize the DOM implementation (one time operation only).
   *
   * @throws InstantiationException if the implementation class couldn't
   *         be instantiated.
   * @throws IllegalAccessException if the implementation class can't be
   *         accessed.
   * @throws ClassNotFoundException if the implementation class can't be
   *         found.
   */
  private static void initializeImpl()
    throws InstantiationException, IllegalAccessException,
	   ClassNotFoundException
  {
    if (domImpl == null)
      {
	DOMImplementationRegistry registry = 
	  DOMImplementationRegistry.newInstance();
	domImpl = registry.getDOMImplementation("LS 3.0");
      }
  }

  /**
   * Serializes the specified object into an XML tree.  If the supplied
   * root argument is not <code>null</code>, the resulting tree
   * is appended to it.  Otherwise, the top-level element of the
   * class is the root.
   *
   * @param object the object to serialize.
   * @param root the root node to append to, or <code>null</code>
   *        if a new root node should be created.
   * @param document the document to use for creation.
   * @return the serialized object in XML form.
   * @throws IllegalAccessException if a field can't be accessed.
   */
  public static Element serialize(Serializable object, Element root,
				  Document document)
    throws IllegalAccessException
  {
    return serialize(object, root, document, true);
  }

  /**
   * Serializes the specified object into an XML tree.  If the supplied
   * root argument is not <code>null</code>, the resulting tree
   * is appended to it.  Otherwise, the class name node is used.
   *
   * @param object the object to serialize.
   * @param root the root node to append to, or <code>null</code>
   *        if a new root node should be created.
   * @param document the document to use for creation.
   * @param includeClassNameElement a flag which, when <code>false</code>,
   *        causes the root node with the name of the class to be
   *        suppressed.
   * @return the serialized object in XML form.
   * @throws IllegalAccessException if a field can't be accessed.
   * @throws IllegalStateException if the root node is null and the
   *         class name node is suppressed.
   */
  public static Element serialize(Serializable object, Element root,
				  Document document,
				  boolean includeClassNameElement)
    throws IllegalAccessException
  {
    PairList<XmlField,Field> fields = new PairList<XmlField,Field>();
    Class clazz = object.getClass();
    Xmlizable customObject = null;
    Element objRoot;
    if (!includeClassNameElement)
      {
	if (root == null)
	  throw new IllegalStateException("Impossible to serialize a class "+
					  "with no supplied root node or "+
					  "class name root node.");
	objRoot = root;
	root = null;
      }
    else
      {
	String elementName = null;
	if (object instanceof Xmlizable)
	  {
	    customObject = (Xmlizable) object;
	    elementName = customObject.getElementName();
	  }
	if (elementName == null)
	  elementName = clazz.getSimpleName();
	objRoot = createElement(document, elementName);
      }
    if (customObject != null)
      addNamespaceDeclarations(customObject.getDeclaredNamespaces(), objRoot);
    while (clazz != null)
      {
	PairList<XmlField,Field> newFields = new PairList<XmlField,Field>();
	try
	  {
	    Field serialField = 
	      clazz.getDeclaredField("serialPersistentFields");
	    ObjectStreamField[] osFields = (ObjectStreamField[])
	      serialField.get(null);
	    for (ObjectStreamField osField : osFields)
	      {
		XmlField xField;
		if (osField instanceof XmlField)
		  xField = (XmlField) osField;
		else
		  xField = new XmlField(osField);
		try
		  {
		    newFields.add(xField,
			       clazz.getDeclaredField(xField.getName()));
		  }
		catch (NoSuchFieldException e)
		  {
		    throw new
		      IllegalStateException("Inconsistency between "+
					    "listed and actual serializable " +
					    "fields in " + osField.getName()
					    + ".",e);
		  }
	      }
	  }
	catch (NoSuchFieldException e)
	  {
	    Field[] clFields = clazz.getDeclaredFields();
	    for (Field field : clFields)
	      newFields.add(new XmlField(field.getName(),
					 field.getType()),field);
	  }
	fields.addAll(0,newFields);
	clazz = clazz.getSuperclass();
      }
    TypeMapper mapper = new TypeMapper();
    for (Pair<XmlField,Field> pair: fields)
      {
	XmlField xField = pair.getLeft();
	Field field = pair.getRight();
	if (Modifier.isTransient(field.getModifiers()))
	  continue;
	serializeValue(field.getName(), field.get(object),
		       xField.isFieldNameSerialized(),
		       xField.isClassNameSerialized(),
		       mapper, document, objRoot);
      }
    if (root != null)
      {
	root.appendChild(objRoot);
	return root;
      }
    return objRoot;
  }

  /**
   * Converts an XML document to a <code>String</code>.
   *
   * @param document the document to convert.
   * @return a <code>String</code> containing the serialized document.
   */  
  public static String convertDocumentToString(Document document)
  {
    DOMImplementationLS loadAndSave = (DOMImplementationLS) domImpl;
    LSSerializer serializer = loadAndSave.createLSSerializer();
    return serializer.writeToString(document);
  }

  /**
   * Creates an element with the appropriate naming schema.
   *
   * @param document the document for creating elements.
   * @param name the proposed document name.
   * @return the created element.
   */
  private static Element createElement(Document document, String name)
  {
    char firstChar = name.charAt(0);
    if (Character.isUpperCase(firstChar))
      name = Character.toLowerCase(firstChar) 
	+ name.substring(1, name.length());
    return document.createElement(name);
  }

  /**
   * Adds namespace declarations to an element.
   *
   * @param qnames the qualified names for which namespace declarations
   *               should be made.
   * @param element the element to which to add declarations.
   */
  private static void addNamespaceDeclarations(QName[] qnames, Element element)
  {
    if (qnames != null)
      for (QName qname : qnames)
	element.setAttributeNS(XMLNS_ATTRIBUTE_NS_URI,
			       XMLNS_ATTRIBUTE + ":" + qname.getPrefix(),
			       qname.getNamespaceURI());
  }

  /**
   * Finalizes an XML document by adding the specified namespace
   * declarations and root element.
   *
   * @param document the document to finalize.
   * @param root the root element of the document.
   * @param namespaces the namespaces to declare as part of the document.
   *                   This may be <code>null</code> if there are no
   *                   namespaces to declare.
   * @return the XML document.
   */
  public static Document finalizeXmlDocument(Document document, Element root,
					     QName[] namespaces)
  {
    addNamespaceDeclarations(namespaces, root);
    document.appendChild(root);
    return document;
  }

  /**
   * Retrieves an empty XML document for content generation.
   *
   * @return a blank XML document.
   * @throws InstantiationException if the implementation class couldn't
   *         be instantiated.
   * @throws IllegalAccessException if the implementation class can't be
   *         accessed.
   * @throws ClassNotFoundException if the implementation class can't be
   *         found.
   */
  public static Document getXmlDocument()
    throws InstantiationException, IllegalAccessException,
	   ClassNotFoundException
  {
    initializeImpl();
    return domImpl.createDocument(null,null,null);
  }

  /**
   * Serializes an XML document to an <code>OutputStream</code>.
   *
   * @param document the document to convert.
   * @param stream the stream to serialize to.
   * @return true if serialization was successful.
   */  
  public static boolean serializeToStream(Document document,
					  OutputStream stream)
  {
    DOMImplementationLS loadAndSave = (DOMImplementationLS) domImpl;
    LSSerializer serializer = loadAndSave.createLSSerializer();
    LSOutput output = loadAndSave.createLSOutput();
    output.setByteStream(stream);
    output.setEncoding("utf-8");
    return serializer.write(document, output);
  }

  /**
   * Serializes a value to XML.  The default of no field names and
   * class names is used.
   * 
   * @param name the name of the field for this value.
   * @param value the value itself.
   * @param mapper for converting between types.
   * @param document the document to create nodes with or add nodes to.
   * @param objRoot the root to which the serialized document should be
   *                presented.
   */
  public static void serializeValue(String name, Object value,
				    TypeMapper mapper, Document document,
				    Element objRoot)
    throws IllegalAccessException
  {
    serializeValue(name, value, false, true, mapper, document, objRoot);
  }

  /**
   * Serializes a value to XML.
   * 
   * @param name the name of the field for this value.
   * @param value the value itself.
   * @param includeFieldName <code>true</code> means that an element
   *                         with the name of the field is serialized.
   * @param includeTypeName <code>true</code> means that an element
   *                         with the name of its class is serialized.
   * @param mapper for converting between types.
   * @param document the document to create nodes with or add nodes to.
   * @param objRoot the root to which the serialized document should be
   *                presented.
   */
  public static void serializeValue(String name, Object value,
				    boolean includeFieldName,
				    boolean includeTypeName,
				    TypeMapper mapper, Document document,
				    Element objRoot)
    throws IllegalAccessException
  {
    System.out.println("field: " + name);
    if (value == null)
      return;
    Class valueClazz = value.getClass();
    System.out.println("value: " + value + ", " + valueClazz);
    XsdType schemaType = mapper.map(valueClazz);
    if (schemaType != null)
      {
	Element element = createElement(document, name);
	element.appendChild(schemaType.translateValue(document, value));
	objRoot.appendChild(element);
      }
    else if (value instanceof Collection)
      {
	Collection collection = (Collection) value;
	for (Object obj : collection)
	  if (obj instanceof Serializable)
	    serialize((Serializable) obj, objRoot, document);
      }
    else if (value instanceof Serializable)
      {
	Element element;
	if (includeFieldName)
	  {
	    element = createElement(document, name);
	    objRoot.appendChild(element);
	  }
	else
	  element = objRoot;
	serialize((Serializable) value, element, document, includeTypeName);
      }
    else
      {
	Element element = createElement(document, name);
	element.appendChild(document.createTextNode(value.toString()));
	objRoot.appendChild(element);
      }
  }

  /**
   * A simple test harness to ensure that objects can be successfully
   * converted to XML.
   *
   * @param args the command-line arguments.
   * @throws InstantiationException if a class couldn't be
   *         instantiated.
   * @throws IllegalAccessException if a class can't be accessed.
   * @throws ClassNotFoundException if a class can't be found.
   * @throws URISyntaxException if one of the names is not a valid URI.
   */
  public static void main(String[] args)
    throws InstantiationException, IllegalAccessException,
	   ClassNotFoundException, URISyntaxException
  {
    Document document = getXmlDocument();
    Element root = Serializer.serialize(TEST_COMPOSITE_SEQUENCE, null,
					document);
    finalizeXmlDocument(document, root, DOCUMENT_NAMESPACES);
    System.out.println(convertDocumentToString(document));
  }

}

