# Catkin/Entry.pm
# Copyright (C) 2002-2003 colin z robertson
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

package Catkin::Entry;

use strict;
use CGI::Carp;
use XML::DOM;
use XML::DOMWrap;
use DateTime;
use Catkin::Comment;
use Catkin::Util;
use Data::Dumper;
use Fcntl ':flock';
use Catkin::Item;
use vars qw( $AUTOLOAD @ISA );

@ISA = qw(Catkin::Item);

my $ID_REGEX = '[-_a-z0-9]+';

sub new {
	my $proto = shift;
	my $class = ref($proto) || $proto;
	my $self  = {};
	bless $self, $class;
	
	my ($elem) = @_;
	$self->{DOM_NODE} = $elem || (new XML::DOM::Document())->createElement("entry");
	my @comments = ();
	foreach my $comment (&XML::DOMWrap::get_child_elements($elem,"comment")) {
		push @comments, new Catkin::Comment($comment);
	}
	$self->comments([@comments]);
    $self->date(DateTime->now) unless $self->date;
	
	return $self;
}

sub comments {
	my $self = shift;
	my ($comment_list) = @_;
	if (defined $comment_list) {
		foreach my $old_comment (XML::DOMWrap::get_child_elements($self->node,"comment")) {
			$self->node->removeChild($old_comment);
		}
		$self->{comments} = $comment_list;
		foreach my $new_comment (@{$comment_list}) {
			$new_comment->parent($self);
			$new_comment->node->setOwnerDocument($self->node->getOwnerDocument);
			$self->node->appendChild($new_comment->node);
		}
	}
	return $self->{comments};
}

sub add_comment {
    my $self = shift;
	my ($comment,$parent_id) = @_;
	my $new_id = 1;
	if ($self->count()) {
		my @comment_list = $self->comments_flat_list();
		my @id_list = map $_->id(), @comment_list;
		$new_id = (sort { $b <=> $a } @id_list)[0] + 1;
	}
	$comment->id($new_id);
	if ($parent_id) {
		my ($attach_to);
		if (!($attach_to = $self->get_comment($parent_id))) {
			carp "No such comment $parent_id\n";
			return 0;
		}
		$attach_to->add_comment($comment);
	} else {
		$self->comments([@{$self->{comments}},$comment]);
	}
	return 1;
}

sub remove_comment {
    my $self = shift;
	my ($comment_id) = @_;
	my $found;
	my @new_comments;
	foreach my $comment(@{$self->comments}) {
		if ($comment->id eq $comment_id) {
			$found = 1;
		} else {
			push @new_comments, $comment;
			if ($comment->remove_comment($comment_id)) {
				$found = 1;
			}
		}
	}
	$self->comments(\@new_comments);
	return $found;
}

sub comments_flat_list {
	my $self = shift;
	return _comments_flat_list_recursive(@{$self->comments});
}

sub _comments_flat_list_recursive {
	my @list = @_;
	my @append;
	foreach my $item (@list) {
		push @append, _comments_flat_list_recursive(@{$item->comments});
	}
	return @list, @append;
}

sub get_comment {
    my $self = shift;
	my ($id) = @_;
	return (grep $_->id eq $id, $self->comments_flat_list)[0];
}

sub count {
	my $self = shift;
	my $count = 0;
	foreach my $comment (@{$self->comments}) {
		$count += $comment->count;
	}
	return $count;
}

sub id {
	my $self = shift;
	my ($new_id) = @_;
	if (!$new_id) {
		return ($self->simple_element("id"))[0];
	} elsif ($new_id =~ /^($ID_REGEX)$/) {
		my $id = $1;
		return ($self->simple_element("id",$id))[0];
	} else {
		warn "New id \"$new_id\" is not valid.\n";
		return;
	}
}

sub validated_id {
	my $self = shift;
	my $id = $self->id;
	if ($id =~ /^($ID_REGEX)$/) {
		return $1;
	} else {
		return;
	}
}

sub title {
	my $self = shift;
	return ($self->simple_element("title",@_))[0];
}

sub text {
	my $self = shift;
	return ($self->simple_element("text",@_))[0];
}

sub type {
	return "entry";
}

1;
