# Catkin/Formatter.pm
# Copyright (C) 2002-2003 colin z robertson
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

package Catkin::Formatter;

use Catkin::Template;
use Catkin::Index;
use Catkin::Entry;
use Catkin::Comment;
use Catkin::Config;
use Catkin::Util;
use XML::Escape;
use CGI::Carp;
use Text::Convert;
use File::Spec::Functions;
use Data::Dumper;
use DateTime;
use strict;
use vars qw($AUTOLOAD);  # it's a package global

my %fields = (
	config => undef,
	index  => undef,
);

sub new {
    my $proto = shift;
    my $class = ref($proto) || $proto;
    my $self  = {
		_permitted => \%fields,
		%fields,
	};
    bless ($self, $class);
	
	my ($config,$index) = @_;
	$self->config($config);
	$self->index($index);
	
	return $self;
}

sub write {
	my $self = shift;
	my @dirty_entries = @_;
	my @entry_templates = get_templates(catdir($self->config->key('priv_dir'),"templates","entry"));
	foreach my $template (@entry_templates) {
		my @write_entries = ();
		my @template_dependencies = $template->dependencies;
		ENTRY:
		foreach my $dirty_entry (@dirty_entries) {
			foreach my $dependency (@template_dependencies) {
				if ($dependency =~ /^self((\+|-)\d+)?$/) { # relative
					my $modifier = $1 || 0;
					my $write_entry = $dirty_entry + $modifier;
					push @write_entries, $write_entry if ($write_entry >= 0 && $write_entry < $self->index->size);
				} elsif ($dependency =~ /^last(-\d+)?$/) { # absolute
					# If a template depends on an entry at a specific point in
					# the list and the dirty entry is at that specific point
					# then all entries will have to be rebuilt with this
					# template.
					my $distance = $self->index->size + (($1 || 0) - 1);
					if ($dirty_entry == $distance) {
						push @write_entries, $self->index->all_entries;
						last ENTRY;
					}
				} elsif ($dependency =~ /^first(\+\d+)?$/) { # absolute
					my $distance = ($1 || 0);
					if ($dirty_entry == $distance) {
						push @write_entries, $self->index->all_entries;
						last ENTRY;
					}
				} elsif ($dependency eq 'all') { # always updated
						push @write_entries, $self->index->all_entries;
						last ENTRY;
				} else { # unrecognised
					warn "Incorrect dependency rule: $dependency\n";
				}
			}
		}
		@write_entries = uniq(@write_entries);
		foreach my $entry_num (@write_entries) {
			my $context = $self->generate_context;
			$context->{self} = $entry_num;
			my $entry = ($self->index->list)[$entry_num];
			if (my $id = $entry->validated_id) {
				my $filename = catfile($self->config->key('html_dir'),$id . "." . $template->extension);
				my $output = $template->process($context);
				Catkin::Util::atomic_write($filename,$output);
			} else {
				warn($entry->id, "not a valid id\n");
			}
		}
	}
	my @index_templates = get_templates(catdir($self->config->key('priv_dir'),"templates","index"));
	foreach my $template (@index_templates) {
		my $context = $self->generate_context;
		my $filename = catdir($self->config->key('html_dir'),$template->filename);
		my $output = $template->process($context);
		Catkin::Util::atomic_write($filename,$output);
	}
}

sub generate_context {
	my $self = shift;
	my $context = {
		blog => {
			entries   => [$self->index->list],
			comments => [$self->index->comments],
			config    => sub { $self->config->key($_[0]) },
		},
	};
	return $context;
}

sub get_templates {
	my ($directory) = @_;
	my @files = grep m/^[^\.].*\.template$/, Catkin::Util::list_dir($directory);
	my @templates;
	foreach my $file (@files) {
		my $template = new Catkin::Template($file);
		if ($template) {
			push @templates, $template;
		} else {
			warn "$file is not a valid template\n";
		}
	}
	return @templates;
}

sub delete {
	my $self = shift;
	my (@ids) = @_;
	my $output_dir = $self->config->key('html_dir') or die "Could not get output directory (html_dir)\n";
	my @list = Catkin::Util::list_dir($output_dir);
	my @delete_files;
	foreach my $id (@ids) {
		my @files = grep(m/[\/\\]$id\.[^\/]+$/,@list);
		push @delete_files,@files;
	}
	foreach my $file (@delete_files) {
		unlink($file) or warn "Failed to delete $file\n";
	}
}

sub reply_page {
	my $self = shift;
	my ($parent, $preview_data, $reply_to) = @_;
	
	my $preview;
	if ($preview_data) {
		$preview = new Catkin::Comment();
		$preview->date(DateTime->now(time_zone => $self->config->key('timezone')));
		$preview->name($preview_data->{name});
		$preview->email($preview_data->{email});
		$preview->url($preview_data->{url});
		$preview->title($preview_data->{title});
		$preview->text($preview_data->{text});
	}
	
	my $template_file = catfile($self->config->key('priv_dir'),"templates","cgi","reply.template");
	my $template = new Catkin::Template($template_file) || return;
	my $context = $self->generate_context;
	$context->{preview}          = $preview;
	$context->{reply_to}         = $parent;
	$context->{form_reply_to}    = $reply_to;
	$context->{form_name}        = $preview_data->{name};
	$context->{form_url}         = $preview_data->{url};
	$context->{form_email}       = $preview_data->{email};
	$context->{form_title}       = $preview_data->{title};
	$context->{form_text}        = $preview_data->{original_text};
	$context->{form_remember}    = $preview_data->{remember};
    $context->{form_mode}        = [
		    {
			    name    => 'text',
				label   => 'Text',
				checked => ((!$preview_data->{mode}) || ($preview_data->{mode} eq 'text')),
			},
		    {
			    name    => 'sloppy_html',
				label   => 'HTML',
				checked => $preview_data->{mode} && ($preview_data->{mode} eq 'sloppy_html'),
			},
	];
	
	my $output = $template->process($context);
	return $output;
}

sub failure_page {
	my $self = shift;
	my ($action, $message) =@_;
	my $file = catfile($self->config->key('priv_dir'), 'templates', 'cgi', 'failure.template');
	my $template = new Catkin::Template($file) || return;
	my $context = $self->generate_context;
	$context->{action} = $action;
	$context->{message} = $message;
	my $output = $template->process($context);
	my $content_type = $template->header('content-type');
	return ($output,$content_type);
}

sub uniq {
	my @list = @_;
	my %hash;
	foreach my $item (@list) {
		$hash{$item} = 1;
	}
	return keys(%hash);
}

sub AUTOLOAD {
	my $self = shift;
	my $type = ref($self) or croak "$self is not an object";

	my $name = $AUTOLOAD;
	$name =~ s/.*://;   # strip fully-qualified portion
	if ($name eq 'DESTROY') { return }

	unless (exists $self->{_permitted}->{$name} ) {
		croak "Can't access `$name' field in class $type";
	}

	if (@_) {
		return $self->{$name} = shift;
	} else {
		return $self->{$name};
	}
}

1;
