//  Copyright (C) 2011, 2014 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include <config.h>
#include <stdbool.h>
#include <stdlib.h>
#include <iostream>
#include <assert.h>
#include <curlpp/cURLpp.hpp>

#include "main.h"
#include "timing.h"
#include "cbc-osd-connector.h"
#include "cbc-show-list.h"
#include "ucompose.hpp"
#include "cbcosd-window.h"

struct Main::Impl
{
    Glib::RefPtr<Gtk::Application> app;
    sigc::connection on_timer_registered(Timing::timer_slot s,
					 int msecs_interval);
    CBCOSDConnector *cbcosd;
    CBCOSDWindow *window;
    CBCShowList *shown_news;
    CBCShowList *shown_sports;
};


static Main *singleton;

Main::Main(int &argc, char **&argv)
    : impl(new Impl)
{
    version_appears_on_the_command_line = false;
    background_appears_on_the_command_line = false;
    singleton = this;

    impl->app = Gtk::Application::create(argc, argv, "org.gnome.cbcosd");
    try
      {
        Glib::OptionContext context("");
        Glib::OptionEntry entry;
        Glib::OptionGroup options (PACKAGE_NAME, "CBC-OSD Options", "Command-line options for CBC-OSD");
        entry.set_long_name ("version");
        entry.set_short_name ('V');
        entry.set_description ("Show version and exit.");
        options.add_entry (entry, version_appears_on_the_command_line);
        entry.set_long_name ("background");
        entry.set_short_name('b');
        entry.set_description ("Run in the background.");
        options.add_entry (entry, background_appears_on_the_command_line);
        context.add_group(options);
        context.set_summary("This program notifies us when there is a live news event streaming on cbc.ca.");

        try
          {
            context.parse(argc, argv);
          }
        catch(const Glib::Error& ex)
          {
            std::cout << ex.what() << std::endl;
            std::cout << "Try `" PACKAGE_NAME 
              " --help' for more information." << std::endl;
            exit(0);
          }

        if (version_appears_on_the_command_line)
          {
            std::cout << PACKAGE_NAME << " " << PACKAGE_VERSION << std::endl;
            std::cout << 
              "This is free software: " <<
              "you are free to change and redistribute it." << std::endl;
            std::cout << 
              "There is NO WARRANTY, to the extent permitted by law." << 
              std::endl;
            exit(0);
          }

        Timing::instance().timer_registered.connect(
	    sigc::mem_fun(*impl, &Main::Impl::on_timer_registered));

        curlpp::initialize();

    }
    catch (const Glib::Error &ex) {
	std::cerr << ex.what() << std::endl;
    }
}

Main::~Main()
{
    delete impl;
    singleton = 0;
}

Main &Main::instance()
{
    assert(singleton != 0);
    return *singleton;
}

void Main::on_connect_succeeded(bool success)
{
  if (!success)
    stop_main_loop();
  impl->cbcosd->news_polled.connect(sigc::bind(sigc::mem_fun(*this, &Main::on_polled), "News", impl->shown_news));
  impl->cbcosd->sports_polled.connect(sigc::bind(sigc::mem_fun(*this, &Main::on_polled), "Sports", impl->shown_sports));
}

void Main::stop_polling()
{
  impl->cbcosd->stop_polling();
}

void Main::on_polled(const char *data, int datalen, Glib::ustring category_id, CBCShowList *shown)
{
  CBCShowList *shows = CBCShowList::create_from_html(data, datalen);
  for (CBCShowList::iterator i = shows->begin(); i != shows->end(); i++)
    {
      if ((*i).is_showing())
        {
          if (shown->contains((*i).get_id()) == false)
            {
              Glib::ustring url = 
                String::ucompose("%1%2/ID/%3/", CBC_VIDEO_BASE, category_id,
                                 (*i).get_id());
              impl->cbcosd->notify((*i).get_title(), (*i).get_description(), (*i).get_id(), url);
              shown->push_back(*i);
            }
        }
    }
  delete shows;
}

void Main::start_main_loop()
{
    try
      {
        impl->cbcosd = CBCOSDConnector::create(true, true);
        impl->shown_news = new CBCShowList();
        impl->shown_sports = new CBCShowList();
        impl->cbcosd->connect_succeeded.connect(sigc::mem_fun(*this, &Main::on_connect_succeeded));
        impl->cbcosd->connect();
        if (background_appears_on_the_command_line == false)
          {

            impl->window = new CBCOSDWindow();
            impl->app->run(impl->window->get_window(), 0, NULL);
          }
        else
          {
            daemon(0, 0);
            restart_polling();
            impl->app->hold();
            impl->app->run(0, NULL);
          }
      }
    catch (const Glib::Error &ex) {
	std::cerr << ex.what() << std::endl;
    }
}
        
void Main::stop_main_loop()
{
    try
    {
        impl->app->quit();
        delete impl->cbcosd;
        delete impl->shown_news;
        delete impl->shown_sports;
    }
    catch (const Glib::Error &ex) {
	std::cerr << ex.what() << std::endl;
    }
}

std::string Main::get_data_path()
{
  return CBCOSD_DATADIR;
}

sigc::connection Main::Impl::on_timer_registered(Timing::timer_slot s,
						 int msecs_interval)
{
    return Glib::signal_timeout().connect(s, msecs_interval);
}

bool Main::get_watching_news()
{
  return Main::instance().impl->cbcosd->get_watch_news();
}

bool Main::get_watching_sports()
{
  return Main::instance().impl->cbcosd->get_watch_sports();
}
    
bool Main::restart_polling()
{
  impl->cbcosd->stop_polling();
  delete impl->shown_news;
  impl->shown_news = new CBCShowList();
  delete impl->shown_sports;
  impl->shown_sports = new CBCShowList();
  impl->cbcosd->connect_succeeded.connect(sigc::mem_fun(*this, &Main::on_connect_succeeded));
  return impl->cbcosd->start_polling(CBCOSDConnector::get_default_polling_period());
}
