#!/usr/bin/python
# -*- coding: utf-8 -*-

## ----------------------------------------------------------------------------
## CERTI - HLA RunTime Infrastructure
## Copyright (C) 2002-2005  ONERA
##
## This program is free software ; you can redistribute it and/or
## modify it under the terms of the GNU Lesser General Public License
## as published by the Free Software Foundation ; either version 2 of
## the License, or (at your option) Any later version.
##
## This program is distributed in the hope that it will be useful, but
## WITHOUT ANY WARRANTY ; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
## Lesser General Public License for more details.
##
## You should have received a copy of the GNU Lesser General Public
## License along with this program ; if not, write to the Free Software
## Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
## USA
##
## $Id: GenMsgCXX.py,v 1.20 2010/06/18 13:34:28 erk Exp $
## ----------------------------------------------------------------------------

"""
The CERTI Message Generator.
C++ Backend Generator
"""

import logging
import GenMsgAST
import GenMsgBase
import sys
import os


class CCERTIGenerator(GenMsgBase.CodeGenerator):

    """
    This is a C generator for C{MessageAST}.
    
    """

    def generatorName(cls):
        return 'C'

    generatorName = classmethod(generatorName)

    def __init__(self, MessageAST):
        super(CCERTIGenerator, self).__init__(MessageAST, '//')
        self.included = dict()
        self.typedefed = dict()
        self.builtinTypeMap = {
            'onoff': 'bool',
            'bool': 'bool',
            'string': 'string',
            'byte': 'uint8_t',
            'int8': 'int8_t',
            'uint8': 'uint8_t',
            'int16': 'int16_t',
            'uint16': 'uint16_t',
            'int32': 'int32_t',
            'uint32': 'uint32_t',
            'int64': 'int64_t',
            'uint64': 'uint64_t',
            'float': 'float',
            'double': 'double',
            }
        self.serializeTypeMap = {
            'onoff': 'write_bool',
            'bool': 'write_bool',
            'string': 'write_string',
            'byte': 'write_byte',
            'int8': 'write_int8',
            'uint8': 'write_uint8',
            'int16': 'write_int16',
            'uint16': 'write_uint16',
            'int32': 'write_int32',
            'uint32': 'write_uint32',
            'int64': 'write_int64',
            'uint64': 'write_uint64',
            'float': 'write_float',
            'double': 'write_double',
            }
        self.deserializeTypeMap = {
            'onoff': 'read_bool',
            'bool': 'read_bool',
            'string': 'read_string',
            'byte': 'read_byte',
            'int8': 'read_int8',
            'uint8': 'read_uint8',
            'int16': 'read_int16',
            'uint16': 'read_uint16',
            'int32': 'read_int32',
            'uint32': 'read_uint32',
            'int64': 'read_int64',
            'uint64': 'read_uint64',
            'float': 'read_float',
            'double': 'read_double',
            }
        self.__languageName = 'C'
        self.replacePrefix = None
        self.exportPrefix = ''
        self.serializeBufferType = 'MsgBuffer_t'
        self.messageTypeGetter = 'getType()'
        self.exception = ['string']

    def getTargetTypeName(self, name):
        if name in self.builtinTypeMap.keys():
            return self.builtinTypeMap[name]
        else:
            t = self.AST.getType(name)
        if isinstance(t, GenMsgAST.EnumType):
            prefix = self.AST.name.split('.')[0] + '_'
            return prefix + name
        else:
            return name

    def getRepresentationFor(self, name):
        for native in self.AST.natives:
            if name == native.name:
                representation = native.getRepresentation()
                if representation:
                    return representation
        return None

    def getSerializeMethodName(self, name):
        if name in self.serializeTypeMap.keys():
            return self.serializeTypeMap[name]
        else:
            representation = self.getRepresentationFor(name)
            if representation:
                return self.getSerializeMethodName(representation)
        return None

    def getDeSerializeMethodName(self, name):
        if name in self.deserializeTypeMap.keys():
            return self.deserializeTypeMap[name]
        else:
            representation = self.getRepresentationFor(name)
            if representation:
                return self.getDeSerializeMethodName(representation)
        return None

    def openNamespaces(self, stream):
        if self.AST.hasPackage():
            self.writeComment(stream, self.AST.package)

            # we may have nested namespace

            nameSpaceList = self.AST.package.name.split('.')
            for ns in nameSpaceList:
                stream.write(self.getIndent() + '''package %s 

'''
                             % ns)
                self.indent()

    def closeNamespaces(self, stream):
        if self.AST.hasPackage():

            # we may have nested namespace

            nameSpaceList = self.AST.package.name.split('.')
            nameSpaceList.reverse()
            for ns in nameSpaceList:
                self.unIndent()
                stream.write(self.getIndent() + '} '
                             + self.commentLineBeginWith
                             + ' end of namespace %s \n' % ns)

    def writeOneGetterSetter(self, stream, field,msg):
        targetTypeName = self.getTargetTypeName(field.typeid.name)

        if field.typeid.name == 'onoff':
            if field.qualifier == 'repeated':
                stream.write(self.getIndent())
                stream.write('uint32_t %s_get'
                             + self.upperFirst(field.name)
                             + 'Size(%s %s) const'
                             % (msg.name, msg.name, msg.name))
                stream.write(' {return ' + msg.name + '.' + field.name + '.size();}\n')
                stream.write(self.getIndent())
                stream.write('void %s_set' + self.upperFirst(field.name)
                             + 'Size(%s_t %s,uint32_t num)' % (msg.name,msg.name,msg.name))
                stream.write(' {' + msg.name + '.' + field.name + '.size = num;}\n')

                stream.write(self.getIndent())
                stream.write('const ' + targetTypeName
                             + '*' + msg.name +'_get' + self.upperFirst(field.name)
                             + '(' + msg.name + ') const')
                stream.write(' {return ' + msg.name + '.' + field.name + ';}\n')

                stream.write(self.getIndent())
                stream.write('void %s_' + field.name + 'On(%s_t %s, uint32_t rank)'
			    % (msg.name, msg.name, msg.name))
                stream.write(' {' + msg.name + '.' + field.name + '[rank] = true;}\n')

                stream.write(self.getIndent())
                stream.write('void %s_' + field.name + 'Off(%s_t %s, uint32_t rank)'
			    % (msg.name, msg.name, msg.name))
                        
                stream.write(' {' + msg.name + '.' + field.name + '[rank] = false;}\n')

                stream.write(self.getIndent())
                stream.write(targetTypeName + ' is'
                             + self.upperFirst(field.name)
                             + 'On(uint32_t rank) const')
                stream.write(' {return ' + msg.name + '.' + field.name + '[rank];}\n')
            else:
                stream.write(self.getIndent())
                stream.write('void %s' + field.name + 'On(%s_t %s)'
			    % (msg.name, msg.name,msg.name))
                stream.write(' {' + msg.name + '.' + field.name + ' = true;}\n')

                stream.write(self.getIndent())
                stream.write('void %s_' + field.name + 'Off(%s_t %s)'
			    % (msg.name, msg.name,msg.name))
                stream.write(' {' + msg.name + '.' +field.name + ' = false;}\n')

                stream.write(self.getIndent())
                stream.write(targetTypeName + ' is'
                             + self.upperFirst(field.name)
                             + 'On(%s_t %s) const'
			    % (msg.name,msg.name))
                stream.write(' {return ' + msg.name + '.' + field.name + ';}\n')
        else:
            if field.qualifier == 'repeated':
                stream.write(self.getIndent())
                stream.write('uint32_t get'
                             + self.upperFirst(field.name)
                             + 'Size(%s_t %s) const'
			     % (msg.name, msg.name))
                stream.write(' {return ' + msg.name + '.' + field.name + '.size();}\n')

                stream.write(self.getIndent())
                stream.write('void set' + self.upperFirst(field.name)
                             + 'Size(%s_t %s, uint32_t num)'
                             % (msg.name, msg.name))
                stream.write(' {' + msg.name + '.' + field.name + '.resize(num);}\n')

                stream.write(self.getIndent())
                stream.write('const ' + targetTypeName
                             + '* get' + self.upperFirst(field.name)
                             + '() const')
                stream.write(' {return ' + field.name + ';}\n')

                stream.write(self.getIndent())
                stream.write('const ' + targetTypeName + '* get'
                             + self.upperFirst(field.name)
                             + '(%s_t %s, uint32_t rank) const'
			     % (msg.name,msg.name))
                stream.write(' {return ' + msg.name + '.' + field.name + '[rank];}\n')

                stream.write(self.getIndent())
                stream.write(targetTypeName + '* get'
                             + self.upperFirst(field.name)
                             + '(%s_t %s, uint32_t rank)'
			    % (msg.name,msg.name))
                stream.write(' {return ' + msg.name + '.' + field.name + '[rank];}\n')

                stream.write(self.getIndent())
                stream.write('void set' + self.upperFirst(field.name)
                             + '(%s_t %s, const '
			    % (msg.name, msg.name))
                stream.write(targetTypeName + '* new'
                             + self.upperFirst(field.name)
                             + ', uint32_t rank)')
                stream.write(' {' + msg.name + '.' + field.name + '[rank]=new'
                             + self.upperFirst(field.name) + ';}\n')
                stream.write(self.getIndent())
                stream.write('void remove'
                             + self.upperFirst(field.name)
                             + '(uint32_t rank)')
                stream.write(' {' + msg.name + '.' + field.name + '.erase(' + field.name
                             + '.begin() + rank);}\n')
            else:
                stream.write(self.getIndent())
                stream.write('const ' + targetTypeName + '* get'
                             + self.upperFirst(field.name) + '(%s_t %s) const'
                             % (msg.name, msg.name))
                stream.write(' {return ' + msg.name + '.'  + field.name + ';}\n')

                stream.write(self.getIndent())
                stream.write('void set' + self.upperFirst(field.name)
                             + '(%s_t %s, const '
                             % (msg.name, msg.name))
                stream.write(targetTypeName + '* new'
                             + self.upperFirst(field.name) + ') {')
                if field.qualifier == 'optional':
                    stream.write('\n')
                    self.indent()
                    stream.write(self.getIndent() + '_has%s=true;\n'
                                 % self.upperFirst(field.name))
                    stream.write(self.getIndent() + msg.name + '.' + field.name + '=new'
                                 + self.upperFirst(field.name) + ';\n')
                    self.unIndent()
                    stream.write(self.getIndent())
                else:
                    stream.write(msg.name + '.' + field.name + '=new'
                                 + self.upperFirst(field.name) + ';')
                stream.write('}\n')
                if field.qualifier == 'optional':
                    stream.write(self.getIndent())
                    tmp = self.upperFirst(field.name)
                    stream.write('bool has%s(%s_t %s) {return %s.%s;}\n'
                                 % (tmp, msg.name, msg.name, msg.name,
                                 tmp))

    def writeDeclarationFieldStatement(self, stream, field):
        stream.write(self.getIndent())
        if field.qualifier == 'repeated':
            stream.write('%s* %s;\n'
                         % (self.getTargetTypeName(field.typeid.name),
                         field.name))
            stream.write(self.getIndent() + 'uint32 ' + field.name + 'size;')
        else:
            stream.write('%s %s;'
                         % (self.getTargetTypeName(field.typeid.name),
                         field.name))
        self.writeComment(stream, field)

        # optional field generate another boolean field
        # used to detect whether if the optional field has
        # been given or not.

        if field.qualifier == 'optional':
            stream.write(self.getIndent() + 'bool _has%s;\n'
                         % self.upperFirst(field.name))

    def generateEnum(self, stream, enum):
        self.writeComment(stream, enum)
        stream.write(self.getIndent())
        stream.write('typedef enum %s {\n' % enum.name)
        self.indent()
        first = True
        lastname = enum.values[len(enum.values) - 1].name
        for enumval in enum.values:
            if first:
                stream.write(self.getIndent())
                stream.write('%s = %d, ' % (enumval.name,
                             enumval.value))
                first = False
                self.writeComment(stream, enumval)
            else:
                stream.write(self.getIndent())
                if enumval.name == lastname:
                    stream.write('%s ' % enumval.name)
                    self.writeComment(stream, enumval)
                    stream.write('\n')
                else:
                    stream.write('%s, ' % enumval.name)
                    self.writeComment(stream, enumval)

        self.unIndent()
        stream.write(self.getIndent())
        stream.write('} %s_t; ' % enum.name)
        stream.write(self.commentLineBeginWith + 'end of enum %s \n'
                     % enum.name)

    def generateHeader(self, stream, factoryOnly=False):

        # write the usual header protecting MACRO

        supposedHeaderName = stream.name
        if supposedHeaderName != '<stdout>':
            supposedHeaderName = os.path.basename(supposedHeaderName)
            supposedHeaderName = os.path.splitext(supposedHeaderName)[0]
            headerProtectMacroName = supposedHeaderName
        else:
            (headerProtectMacroName, ext) = \
                os.path.splitext(self.AST.name)

        headerProtectMacroName = '%s_H' \
            % headerProtectMacroName.upper()
        stream.write('#ifndef %s\n' % headerProtectMacroName)
        stream.write('#define %s\n' % headerProtectMacroName)

        # add necessary standard and global includes

        stream.write(self.commentLineBeginWith
                     + ' ****-**** Global System includes ****-****\n')
        stream.write('#include <stdio.h>\n')
        self.included['#include <stdio.h>'] = 1
        stream.write('#include <string.h>\n')
        self.included['#include <string.h>'] = 1

        # add include coming from native type specification

        stream.write(self.commentLineBeginWith
                     + ' ****-**** Includes coming from native types ****-****\n'
                     )
        for native in self.AST.natives:
            if native.hasLanguage('C'):
                for line in native.getLanguageLines('C'):

                    # we are only interested in native "include" statement

                    stmt = line.statement
                    if stmt.find('#include') >= 0 and not stmt \
                        in self.included.keys():
                        self.writeComment(stream, native)
                        stream.write(stmt + '\n')
                        self.included[stmt] = 1

        # Generate namespace for specified package package
        # we may have nested namespace

        self.openNamespaces(stream)

        if not factoryOnly:

            # Native type should be defined in included header

            stream.write(self.getIndent() + self.commentLineBeginWith)
            stream.write(' Native types has been defined:\n')
            stream.write(self.getIndent() + self.commentLineBeginWith)
            stream.write('     - by included headers (see above)\n')
            stream.write(self.getIndent() + self.commentLineBeginWith)
            stream.write('     - with typedef (see below [if any])\n')
            for native in self.AST.natives:
                if native.hasLanguage('C'):
                    for line in native.getLanguageLines('C'):
                        stmt = line.statement

                       # we are only interested in native statement
                       # which are not #include

                        if stmt.find('typedef') >= 0 and not stmt \
                            in self.typedefed.keys():
                            self.writeComment(stream, native)
                            stream.write(self.getIndent() + stmt + '\n')
                            self.typedefed[stmt] = 1

            # Put enum in a namespace in order to avoid conflict

            stream.write(self.getIndent())
            stream.write('typedef %s {\n' % self.AST.name.split('.'
                         )[0])
            self.indent()

            # Generate version

            if self.AST.hasVersion():
                (major, minor) = self.AST.version.number
                stream.write(self.getIndent())
                stream.write('static const uint32_t versionMajor = %d;\n'
                              % major)
                stream.write(self.getIndent())
                stream.write('static const uint32_t versionMinor = %d;\n'
                              % minor)

            # Generate enum

            lastname = ''
            for enum in self.AST.enums:
                self.generateEnum(stream, enum)
            stream.write('\n')

            # close enum namespace

            self.unIndent()
            stream.write(self.getIndent())
            stream.write('}\n')

            # Generate message type

            for msg in self.AST.messages:
                self.writeComment(stream, msg)
                stream.write(self.getIndent())
                stream.write('typedef struct %s %s' % (self.exportPrefix,
                             msg.name))
                if msg.hasMerge():
		    stream.write('{\n')
                    virtual = 'virtual '
                elif msg.hasHeir():
                    stream.write(' {\n')
                    virtual = 'virtual '
                else:
                    stream.write(' {\n')
                    virtual = ''

                self.indent()

                if msg.hasEnum():
                    self.generateEnum(stream, msg.enum)
                    stream.write('\n')

                if msg.hasMerge():
                    stream.write(self.getIndent()
                                 + '%s super;\n'
                                 % msg.merge.name)


                # write fields

                for field in msg.fields:
                    if isinstance(field,
                                  GenMsgAST.MessageType.CombinedField):
                        for cfield in field.fields:
                            self.writeDeclarationFieldStatement(stream,
                                    cfield)
                    else:
                        self.writeDeclarationFieldStatement(stream,
                                field)

		self.unIndent()
		stream.write(self.getIndent() + '} %s_t; \n\n' % msg.name)

                # now write constructor/destructor

                stream.write(self.getIndent() + 'void %s_create();\n' % msg.name)
                stream.write(self.getIndent() + 'void %s_destroy();\n\n' % msg.name)

                # write virtual serialize and deserialize
                # if we have some specific field

                if len(msg.fields) > 0:

                    # serialize/deserialize

                    stream.write(self.getIndent() + virtual
                                 + 'void %s_serialize(%s_t* %s, %s* msgBuffer);\n'
                                 % (msg.name, msg.name, msg.name, self.serializeBufferType))
                    stream.write(self.getIndent() + virtual
                                 + 'void %s_deserialize(%s_t* %s, %s* msgBuffer);\n'
                                 % (msg.name, msg.name, msg.name, self.serializeBufferType))

                    # specific getter/setter

                    stream.write(self.getIndent()
                                 + self.commentLineBeginWith
                                 + ' specific Getter(s)/Setter(s)\n')
                    for field in msg.fields:
                        if isinstance(field,
                                GenMsgAST.MessageType.CombinedField):
                            for cfield in field.fields:
                                self.writeOneGetterSetter(stream,
                                        cfield,msg)
                        else:
                            self.writeOneGetterSetter(stream, field,msg)

                    # the show method

                    stream.write(self.getIndent()
                                 + self.commentLineBeginWith
                                 + ' the show method\n')
                    stream.write(self.getIndent() + virtual
                                 + 'FILE* show(FILE* out);\n'
                                 )

                # end public:

                stream.write(self.getIndent() + '};\n')

        # Generate Factory (if any)
        # @todo

        if self.AST.hasFactory():
            self.writeComment(stream, self.AST.factory)
            stream.write(self.getIndent() + 'typedef struct %s %s {\n'
                         % (self.exportPrefix, self.AST.factory.name))
            self.indent()

            # begin public

            stream.write(self.getIndent() + 'public:\n')
            self.indent()

            stream.write(self.getIndent()
                         + 'static %s* %s(%s type) throw ('
                         % self.AST.factory.creator)
            stream.write('%s' % self.exception[0])
            for exception in self.exception[1:]:
                stream.write(' ,%s' % exception)
            stream.write('); \n')

            if self.AST.factory.hasFactoryReceiver():
                stream.write(self.getIndent()
                             + 'static %s* %s(%s stream) throw ('
                             % self.AST.factory.receiver)
                stream.write('%s' % self.exception[0])
                for exception in self.exception[1:]:
                    stream.write(' ,%s' % exception)
                stream.write('); \n')

            self.unIndent()

            # end public
            # begin protected

            stream.write(self.getIndent() + 'protected:\n')
            self.indent()
            self.unIndent()

            # end protected
            # begin private

            stream.write(self.getIndent() + 'private:\n')
            self.indent()
            self.unIndent()

            # end private

            self.unIndent()
            stream.write(self.getIndent() + '''};

''')

        # may close any open namespaces

        self.closeNamespaces(stream)

        # close usual HEADER protecting MACRO

        stream.write(self.commentLineBeginWith + ' %s\n'
                     % headerProtectMacroName)
        stream.write('#endif\n')

    def writeInitFieldStatement(self, stream, field):
        if field.qualifier == 'optional':
            stream.write(self.getIndent())
            stream.write('_has' + self.upperFirst(field.name)
                         + '=false;\n')
        if field.hasDefaultValue():
            stream.write(self.getIndent())
            stream.write(field.name + '=' + str(field.defaultValue)
                         + ';\n')
        else:
            stream.write(self.getIndent())
            stream.write(self.commentLineBeginWith)
            stream.write(field.name
                         + '= <no default value in message spec using builtin>\n'
                         )

            # FIXME find a default value for every type beside natives

    def writeSerializeFieldStatement(self, stream, field):
        indexField = ''
        if field.qualifier == 'optional':
            stream.write(self.getIndent())
            stream.write('MB_write_bool(msgBuffer,_has%s);\n'
                         % self.upperFirst(field.name))
            stream.write(self.getIndent())
            stream.write('if (_has%s) {\n'
                         % self.upperFirst(field.name))
            self.indent()
        elif field.qualifier == 'repeated':
            indexField = '[i]'
            stream.write(self.getIndent())
            stream.write('uint32_t ' + field.name + 'Size = '
                         + field.name + '.size();\n')
            stream.write(self.getIndent())
            stream.write('MB_write_uint32(msgBuffer,' + field.name
                         + 'Size);\n')
            stream.write(self.getIndent())
            stream.write('for (uint32_t i = 0; i < ' + field.name
                         + 'Size; ++i) {\n')
            self.indent()

        stream.write(self.getIndent())
        methodName = self.getSerializeMethodName(field.typeid.name)
        if None == methodName:  # field has no Serialize Method Name found in the map

            # non native field case

            if field.typeid.name in [m.name for m in self.AST.messages]:
                stream.write(field.name + indexField
                             + '_serialize(msgBuffer);\n')
            elif field.typeid.name in [m.name for m in self.AST.enums]:

            # enum type field case (enum are serialized as uint32)

                methodName = self.getSerializeMethodName('uint32')
                stream.write('MB_' + methodName)
                stream.write('(msgBuffer' + field.name + indexField + ');\n')
            else:

            # native field case

                stream.write(self.commentLineBeginWith
                             + ' FIXME FIXME FIXME\n')
                stream.write(self.getIndent()
                             + self.commentLineBeginWith
                             + " don't know how to serialize native field <%s> of type <%s>\n"
                              % (field.name, field.typeid.name))
        else:

              # field has one Serialize Method Name found in the map

            stream.write('MB_' + methodName)
            stream.write('(msgBuffer,' + field.name + indexField + ');\n')

        if field.qualifier == 'optional':
            self.unIndent()
            stream.write(self.getIndent() + '}\n')
        elif field.qualifier == 'repeated':
            self.unIndent()
            stream.write(self.getIndent() + '}\n')

    def writeShowFieldStatement(self, stream, field):
        indexField = ''
        if field.qualifier == 'optional':
            stream.write(self.getIndent())
            stream.write('fprintf(out,"(opt) %s =" ' % field.name)
        elif field.qualifier == 'repeated':
            indexField = '[i]'
            stream.write(self.getIndent())
            stream.write('fprintf(out,"    %s [] = \n");\n'
                         % field.name)
            stream.write('fflush(out);\n')
            stream.write(self.getIndent())
            stream.write('for (uint32_t i = 0; i < get'
                         + self.upperFirst(field.name)
                         + 'Size(); ++i) {\n')
            self.indent()
            stream.write(self.getIndent() + 'out ')
        else:
            stream.write(self.getIndent())
            stream.write('fprintf(out," %s = " ' % field.name)

        methodName = self.getSerializeMethodName(field.typeid.name)
        if None == methodName:  # field has no Serialize Method Name found in the map

            # non native field case

            if field.typeid.name in [m.name for m in self.AST.messages]:
                stream.write(field.name + indexField
                             + '.show(out)')
            elif field.typeid.name in [m.name for m in self.AST.enums]:

            # enum type field case (enum are serialized as uint32)

                stream.write('+ %s + " " ' % (field.name
                             + indexField))
            else:

                # stream.write(self.commentLineBeginWith+" FIXME FIXME FIXME inherited message\n")
            # native field case

                stream.write('<< "')
                stream.write(self.getIndent()
                             + self.commentLineBeginWith
                             + "FIXME FIXME don't know how to serialize native field <%s> of type <%s>"
                              % (field.name, field.typeid.name))
                stream.write('"')
        else:

              # field has one Serialize Method Name found in the map

            stream.write('+ %s + " " ' % (field.name + indexField))

        if field.qualifier == 'optional':
            stream.write(self.getIndent() + '+ "\\n");\n')
            stream.write(self.getIndent() + 'fflush(out)\n')
        elif field.qualifier == 'repeated':
            stream.write(';\n')
            self.unIndent()
            stream.write(self.getIndent() + '}\n')
            stream.write(self.getIndent() + 'fprintf(out,"\\n");\n')
            stream.write(self.getIndent() + 'fflush(out);\n')
        else:
            stream.write(self.getIndent() + ' + "\\n");\n')
            stream.write(self.getIndent() + 'fflush(out);\n')

    def writeDeSerializeFieldStatement(self, stream, field):
        indexField = ''
        if field.qualifier == 'optional':
            stream.write(self.getIndent())
            stream.write('_has%s = MB_read_bool(msgBuffer);\n'
                         % self.upperFirst(field.name))
            stream.write(self.getIndent())
            stream.write('if (_has%s) {\n'
                         % self.upperFirst(field.name))
            self.indent()
        elif field.qualifier == 'repeated':
            indexField = '[i]'
            stream.write(self.getIndent())
            stream.write('uint32_t ' + field.name
                         + 'Size = MB_read_uint32(msgBuffer);\n')
            stream.write(self.getIndent())
            stream.write(field.name + '.resize(' + field.name
                         + 'Size);\n')
            stream.write(self.getIndent())
            stream.write('for (uint32_t i = 0; i < ' + field.name
                         + 'Size; ++i) {\n')
            self.indent()

        stream.write(self.getIndent())
        methodName = self.getDeSerializeMethodName(field.typeid.name)
        if None == methodName:  # field has no Deserialize Method Name found in the map

         # non native field case

            if field.typeid.name in [m.name for m in self.AST.messages]:
                stream.write(field.name + indexField
                             + '_deserialize(msgBuffer);\n')
            elif field.typeid.name in [m.name for m in self.AST.enums]:

            # enum type field case (enum are deserialized as uint32)

                targetTypeName = \
                    self.getTargetTypeName(field.typeid.name)
                methodName = self.getDeSerializeMethodName('uint32')

                # We should check if the uint32 value is in enum range before casting it into enumtype

                stream.write(field.name + indexField + ' = static_cast<'
                              + targetTypeName + '>(MB_'
                             + methodName + ' ());\n')
            else:

            # native field case

                stream.write(self.commentLineBeginWith
                             + ' FIXME FIXME FIXME\n')
                stream.write(self.getIndent()
                             + self.commentLineBeginWith
                             + " don't know how to deserialize native field <%s> of type <%s>\n"
                              % (field.name, field.typeid.name))
        else:

              # field has one Deserialize Method Name found in the map

            if methodName == 'read_string':
                stream.write('MB_' + methodName + '(msgBuffer,'
                             + field.name + indexField + ');\n')
            else:

                # We may have to vast in order to enforce conversion

                if isinstance(field.typeid, GenMsgAST.NativeType):
                    stream.write(field.name + indexField
                                 + ' = static_cast<'
                                 + field.typeid.name + '>(MB_'
                                 + methodName + '(msgBuffer));\n')
                else:
                    stream.write(field.name + indexField
                                 + ' = MB_' + methodName
                                 + '(msgBuffer);\n')

        if field.qualifier == 'optional':
            self.unIndent()
            stream.write(self.getIndent() + '}\n')
        elif field.qualifier == 'repeated':
            self.unIndent()
            stream.write(self.getIndent() + '}\n')

    def writeFactoryCreator(self, stream):
        creator = (self.AST.factory.creator[0], self.AST.factory.name) \
            + self.AST.factory.creator[1:]
        stream.write(self.getIndent() + '%s* %s_%s(%s type) throw ('
                     % creator)
        stream.write('%s' % self.exception[0])
        for exception in self.exception[1:]:
            stream.write(' ,%s' % exception)
        stream.write(') { \n')

        self.indent()
        stream.write(self.getIndent() + '''%s* msg = NULL;

'''
                     % creator[0])
        stream.write(self.getIndent() + 'switch (type) {\n')
        self.indent()
        for e in self.AST.eMessageType.values:
            if None != self.replacePrefix:
                stream.write(self.getIndent() + 'case %s_%s:\n'
                             % (creator[0],
                             e.name.replace(self.replacePrefix[0], '',
                             1)))
            else:
                stream.write(self.getIndent() + 'case %s_%s:\n'
                             % (creator[0], e.name))

            self.indent()

            if None == e.type:

        # we throw here the first exception of the list

                stream.write(self.getIndent()
                             + 'throw %s("%s message type should not be used!!");\n'
                              % (self.exception[0], e.name))
            else:
                stream.write(self.getIndent() + 'msg = new %s();\n'
                             % e.type)
            stream.write(self.getIndent() + 'break;\n')
            self.unIndent()
        self.unIndent()
        stream.write(self.getIndent() + '} '
                     + self.commentLineBeginWith
                     + ' end if switch (type)\n')
        stream.write(self.getIndent() + 'return msg;\n')
        self.unIndent()
        stream.write(self.getIndent() + '''} /* end of %s_%s */

'''
                     % (creator[1], creator[2]))

    def writeFactoryReceiver(self, stream):
        receiver = (self.AST.factory.receiver[0],
                    self.AST.factory.name) \
            + self.AST.factory.receiver[1:]
        stream.write(self.getIndent() + '%s* %s_%s(%s stream) throw ('
                     % receiver)
        stream.write('%s' % self.exception[0])
        for exception in self.exception[1:]:
            stream.write(' ,%s' % exception)
        stream.write(') { \n')

        self.indent()
        stream.write(self.getIndent() + self.commentLineBeginWith
                     + ' FIXME This is not thread safe\n')
        stream.write(self.getIndent() + 'static %s msgBuffer;\n'
                     % self.serializeBufferType)
        stream.write(self.getIndent() + '%s  msgGen;\n' % receiver[0])
        stream.write(self.getIndent() + '''%s* msg;

''' % receiver[0])
        stream.write(self.getIndent() + self.commentLineBeginWith
                     + ' receive generic message \n')
        stream.write(self.getIndent()
                     + 'msgGen.receive(stream,msgBuffer);\n')
        stream.write(self.getIndent() + self.commentLineBeginWith
                     + ' create specific message from type \n')

        stream.write(self.getIndent() + 'msg = ')
        stream.write(self.AST.factory.name + '_'
                     + self.AST.factory.creator[1] + '(msgGen.%s);\n'
                     % self.messageTypeGetter)

        stream.write(self.getIndent()
                     + 'MB_assumeSizeFromReservedBytes();\n')
        stream.write(self.getIndent() + 'msg->deserialize(msgBuffer);\n'
                     )
        stream.write(self.getIndent() + 'return msg;\n')
        self.unIndent()
        stream.write(self.getIndent() + '''} /* end of %s_constructor */ 

'''
                     % (receiver[1], receiver[2]))

    def applyToFields(
        self,
        stream,
        fields,
        applyObject,
        ):
        for field in fields:
            if isinstance(field, GenMsgAST.MessageType.CombinedField):
                for cfield in field.fields:
                    applyObject(stream, cfield)
            else:
                applyObject(stream, field)

    def generateBody(self, stream, factoryOnly=False):
        """
        Generate the body.
        """

        # add necessary standard includes

        stream.write('#include <vector>\n')
        stream.write('#include <string>\n')

        # [Try to] add corresponding header include

        supposedHeaderName = stream.name
        if supposedHeaderName != '<stdout>':
            supposedHeaderName = os.path.basename(supposedHeaderName)
            supposedHeaderName = os.path.splitext(supposedHeaderName)[0]
            stream.write('#include "' + supposedHeaderName + '.h"\n')

        # Generate namespace for specified package package
        # we may have nested namespace

        self.openNamespaces(stream)
        if not factoryOnly:

            # Generate message type

            for msg in self.AST.messages:

                # Generate Constructor

                stream.write(self.getIndent() + '%s_create() {\n'
                             % (msg.name))
                self.indent()
                if msg.hasMerge():

                    # Assign my name.

                    stream.write(self.getIndent()
                                 + 'this->messageName = "' + msg.name
                                 + '";\n')
                    if None != self.replacePrefix:
                        stream.write(self.getIndent() + 'this->type = '
                                + msg.name.upper().replace(self.replacePrefix[0],
                                self.replacePrefix[1], 1) + ';\n')
                    else:
                        stream.write(self.getIndent() + 'this->type = '
                                + msg.name.upper() + ';\n')

                # Write init value if any was provided

                if len(msg.fields) > 0:
                    self.applyToFields(stream, msg.fields,
                            self.writeInitFieldStatement)
                self.unIndent()
                stream.write(self.getIndent() + '''}

''')

                # Generate Destructor

                stream.write(self.getIndent() + '%s_~%s() {\n'
                             % (msg.name, msg.name))
                self.indent()
                self.unIndent()
                stream.write(self.getIndent() + '''}

''')

                # write virtual serialize and deserialize
                # if we have some specific field

                if len(msg.fields) > 0:

                    # begin serialize method

                    stream.write(self.getIndent()
                                 + 'void %s_serialize(%s* msgBuffer) {\n'
                                  % (msg.name,
                                 self.serializeBufferType))
                    self.indent()
                    if msg.hasMerge():
                        stream.write(self.getIndent()
                                + self.commentLineBeginWith)
                        stream.write('Call mother class\n')
                        stream.write(self.getIndent()
                                + 'Super_serialize(msgBuffer);\n')
                    stream.write(self.getIndent()
                                 + self.commentLineBeginWith)
                    stream.write('Specific serialization code\n')
                    self.applyToFields(stream, msg.fields,
                            self.writeSerializeFieldStatement)
                    self.unIndent()
                    stream.write(self.getIndent() + '''}

''')

                    # end serialize method

                    # begin deserialize method

                    stream.write(self.getIndent()
                                 + 'void %s_deserialize(%s* msgBuffer) {\n'
                                  % (msg.name,
                                 self.serializeBufferType))
                    self.indent()
                    if msg.hasMerge():
                        stream.write(self.getIndent()
                                + self.commentLineBeginWith)
                        stream.write('Call mother class\n')
                        stream.write(self.getIndent()
                                + 'Super_deserialize(msgBuffer);\n')
                    stream.write(self.getIndent()
                                 + self.commentLineBeginWith)
                    stream.write('Specific deserialization code\n')
                    self.applyToFields(stream, msg.fields,
                            self.writeDeSerializeFieldStatement)
                    self.unIndent()
                    stream.write(self.getIndent() + '''}

''')

                    # end deserialize method
                    # begin show method

                    stream.write(self.getIndent()
                                 + 'FILE* %s_show(FILE* out) {\n'
                                  % msg.name)
                    self.indent()
                    stream.write(self.getIndent()
                                 + 'fprintf(out,"[%s -Begin] \\n");\n'
                                 % msg.name)
                    stream.write(self.getIndent()
                                 + 'fflush(out);')
                    if msg.hasMerge():
                        stream.write(self.getIndent()
                                + self.commentLineBeginWith)
                        stream.write('Call mother class\n')
                        stream.write(self.getIndent()
                                + 'Super_show(out);\n')
                    stream.write(self.getIndent()
                                 + self.commentLineBeginWith)
                    stream.write('Specific show code\n')
                    self.applyToFields(stream, msg.fields,
                            self.writeShowFieldStatement)
                    stream.write(self.getIndent()
                                 + 'fprintf(out,"[%s -End] \\n");\n'
                                 % msg.name)
                    stream.write(self.getIndent()
                                 + 'fflush(out);\n')
                    stream.write(self.getIndent() + 'return out;\n')
                    self.unIndent()
                    stream.write(self.getIndent() + '''}

''')

                    # end show method

        # Generate Factory (if any)
        # @todo

        if self.AST.hasFactory():

            # begin creator

            self.writeFactoryCreator(stream)

            # begin receiver

            if self.AST.factory.hasFactoryReceiver():
                self.writeFactoryReceiver(stream)

        self.closeNamespaces(stream)


class CCERTIMessageGenerator(CCERTIGenerator):

    """
    This is a C generator for C{MessageAST}.
    
    """

    def generatorName(cls):
        return 'CCERTIMessage'

    generatorName = classmethod(generatorName)

    def __init__(self, MessageAST):
        super(CCERTIMessageGenerator, self).__init__(MessageAST)
        self.replacePrefix = list()
        self.replacePrefix.append('M_')
        self.replacePrefix.append('Message::')
        self.exportPrefix = 'CERTI_EXPORT'
        self.serializeBufferType = 'MsgBuffer_t'
        self.messageTypeGetter = 'getMessageType()'
        self.exception = ['NetworkError', 'NetworkSignal']


class CCERTINetworkMessageGenerator(CCERTIGenerator):

    """
    This is a C generator for C{MessageAST}.
    
    """

    def generatorName(cls):
        return 'CCERTINetworkMessage'

    generatorName = classmethod(generatorName)

    def __init__(self, MessageAST):
        super(CCERTINetworkMessageGenerator,
              self).__init__(MessageAST)
        self.replacePrefix = list()
        self.replacePrefix.append('NM_')
        self.replacePrefix.append('NetworkMessage::')
        self.exportPrefix = 'CERTI_EXPORT'
        self.serializeBufferType = 'MsgBuffer_t'
        self.messageTypeGetter = 'getMessageType()'
        self.exception = ['NetworkError', 'NetworkSignal']


