// cflow2cflow v1.2
//
// Copyright (C) 2002, by
// 
// Carlo Wood, Run on IRC <carlo@alinoe.com>
// RSA-1024 0x624ACAD5 1997-01-26                    Sign & Encrypt
// Fingerprint16 = 32 EC A7 B6 AC DB 65 A6  F6 F6 55 DD 1C DC FF 61
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

#ifndef PARSE_H
#define PARSE_H
//--------------------------------------------------------------------------------
class function_ct;
class functions_ct;
class function_references_ct;
//--------------------------------------------------------------------------------

#include <string>
#include <iosfwd>
#include <set>
#include <list>
#include <map>

class functions_ct {
  public:
    typedef std::map<std::string, function_ct> functionmap_type;
    typedef functionmap_type::iterator iterator_type;
    typedef functionmap_type::const_iterator const_iterator_type;
  private:
    functionmap_type M_functions;
  public:
    iterator_type add(function_ct const& function);
    iterator_type find(unsigned int refnum);
    iterator_type end(void) { return M_functions.end(); }

  void print_on(std::ostream& os, unsigned int depth, unsigned int max_depth, bool inversed) const;
};

class function_references_ct {
  private:
    typedef std::set<functions_ct::iterator_type> reference_list_t;
    reference_list_t M_functions;
  public:
    void add(functions_ct::iterator_type const& iter);
    size_t size(void) const { return M_functions.size(); }

    void print_on(std::ostream& os, unsigned int depth, unsigned int max_depth, bool inversed) const;
};

class function_ct {
  private:
    std::string M_name;
    std::string M_filename;
    unsigned int M_line;
    static unsigned int S_refnum;
    unsigned int M_refnum;
    mutable unsigned int M_refout;
    mutable int M_already_printed;
    bool M_stop;
    bool M_external;
    static bool S_sun_format;
    function_references_ct M_functions_calls;
    function_references_ct M_functions_called_by;
  public:
    function_ct(unsigned int refnum, std::string const& name, std::string const& filename, unsigned int line) :
        M_name(name), M_filename(filename), M_line(line), M_refnum(refnum), M_refout(0), M_already_printed(0), M_stop(false), M_external(false) { }
    function_ct(std::string const& name) : M_name(name), M_refout(0), M_external(true) { }
    bool operator<(function_ct const& function) const;
    void calls(functions_ct::iterator_type const& iter) { M_functions_calls.add(iter); }
    void called_by(functions_ct::iterator_type const& iter) { M_functions_called_by.add(iter); }
    unsigned int get_refnum(void) const { return M_refnum; }
    bool already_printed(void) const { return M_already_printed; }
    void print_on(std::ostream& os, unsigned int depth, unsigned int max_depth, bool inversed) const;
    void stop(void) { M_stop = true; }
    static void sun_format(bool what) { S_sun_format = what; }
    std::string const& name(void) const { return M_name; }
};

class cflow_ct {
  public:
    functions_ct all_functions;
    functions_ct external_functions;
  protected:
    functions_ct::iterator_type last_func[256];
  protected:
    virtual void new_function(int depth, std::string const& funcname, std::string const& filename, unsigned int line) { }
    virtual void new_external_function(int depth, std::string const& funcname) { }
  public:
    virtual ~cflow_ct() { }
    void read_from(std::istream* input);

  friend std::istream& operator>>(std::istream& is, cflow_ct& calltree);
};

//--------------------------------------------------------------------------------
#include "definition.h"
#ifdef DEFINITION_PARSE
#undef DEFINITION_PARSE
//--------------------------------------------------------------------------------

#include <iostream>

void function_ct::print_on(std::ostream& os, unsigned int depth, unsigned int max_depth, bool inversed) const
{
  ++S_refnum;
  os << S_refnum;
  for (unsigned int d = 0; d <= depth; ++d)
    os << '\t';
  if (M_external)
  {
    os << M_name << (S_sun_format ? ": <>" : " {}") << '\n';
  }
  else
  {
    if (M_already_printed == 1 && !S_sun_format)
      os << "... ";
    os << M_name;
    if (M_already_printed)
      os << (S_sun_format ? ": " : " ... {") << M_refout << (S_sun_format ? "" : "}");
    else
    {
      os << (S_sun_format ? ": ..., <" : " {") << M_filename << ' ' << M_line << (S_sun_format ? ">" : "}");
      M_refout = S_refnum;
    }
    if (M_stop)
    {
      os << " [STOP]";
      M_already_printed = 2;
    }
    else if (depth >= max_depth)
    {
      os << " [MAX-DEPTH]";
      M_already_printed = 2;
    }
    os << '\n';
    if (!M_already_printed)
    {
      M_already_printed = 1;
      if (inversed)
	M_functions_called_by.print_on(os, depth + 1, max_depth, true);
      else
	M_functions_calls.print_on(os, depth + 1, max_depth, false);
      M_already_printed = 2;
    }
  }
}

bool function_ct::operator<(function_ct const& function) const
{
  return M_name < function.M_name;
}

unsigned int function_ct::S_refnum;
bool function_ct::S_sun_format;

functions_ct::iterator_type functions_ct::add(function_ct const& function)
{
  return M_functions.insert(functionmap_type::value_type(function.name(), function)).first;
}

functions_ct::iterator_type functions_ct::find(unsigned int refnum)
{
  for (iterator_type iter(M_functions.begin()); iter != M_functions.end(); ++iter)
    if ((*iter).second.get_refnum() == refnum)
      return iter;
  return M_functions.end();
}

void functions_ct::print_on(std::ostream& os, unsigned int depth, unsigned int max_depth, bool inverse) const
{
  for (const_iterator_type i(M_functions.begin()); i != M_functions.end(); ++i)
  {
    if (!(*i).second.already_printed())
      (*i).second.print_on(os, depth, max_depth, inverse);
  }
}

void function_references_ct::add(functions_ct::iterator_type const& iter)
{
  M_functions.insert(iter);
}

void function_references_ct::print_on(std::ostream& os, unsigned int depth, unsigned int max_depth, bool inverse) const
{
  for (function_references_ct::reference_list_t::const_iterator i(M_functions.begin()); i != M_functions.end(); ++i)
    (*i)->second.print_on(os, depth, max_depth, inverse);
}

bool operator<(functions_ct::iterator_type const& iter1, functions_ct::iterator_type const& iter2)
{
  return (*iter1) < (*iter2);
}

std::istream& operator>>(std::istream& is, cflow_ct& calltree)
{
  calltree.read_from(&is);
  return is;
}

void cflow_ct::read_from(std::istream* input)
{
  for (int curref = 1;; ++curref)
  {
    unsigned int line;
    *input >> line;
    if (!*input)
      break;
    int depth = 0;
    bool recursive_call = false;
    char c = input->get();
    while (c == '\t' || c == ' ' || c == '<' || c == '.')
    {
      if (c == '.')
	recursive_call = true;
      else if (c == '<')
	depth += 14;
      else if (c == '\t')
	++depth;
      c = input->get();
    }
    input->putback(c);
    std::string funcname;
    *input >> funcname;
    *input >> c;
    while (c == '.')
      *input >> c;
    c = input->get();
    bool saw_new_function = false;
    bool reference = false;
    unsigned int refnum = 0;
    std::string filename;
    if (c != '}')
    {
      input->putback(c);
      if (!isdigit(c))
      {
	saw_new_function = true;
	*input >> filename;
	*input >> line;
      }
      else
      {
	reference = true;
	*input >> refnum;
      }
      *input >> c;
      if (c != '}')
      {
	std::cerr << "Parse error in line " << curref << '\n';
	exit(2);
      }
    }
    assert( depth > 1 || saw_new_function );
    if (saw_new_function)
    {
      last_func[depth] = all_functions.add(function_ct(curref, funcname, filename, line));
      new_function(depth, funcname, filename, line);
    }
    else if (reference)
    {
      last_func[depth] = all_functions.find(refnum);
      if (last_func[depth] == all_functions.end())
      {
	std::cerr << "Couldn't find reference number " << refnum << '\n';
	exit(3);
      }
    }
    else
    {
      external_functions.add(function_ct(funcname));
      new_external_function(depth, funcname);
      last_func[depth] = all_functions.add(function_ct(funcname));
    }
    if (depth > 1)
    {
      last_func[depth - 1]->second.calls(last_func[depth]);
      last_func[depth]->second.called_by(last_func[depth - 1]);
    }
  }
}

//--------------------------------------------------------------------------------
#endif // DEFINITION_PARSE
#endif // PARSE_H
