"""basic cfvers classes

This module implements the basic cfvers objects, in a
repository-independent way.

"""

# Copyright 2003 Iustin Pop
#
# This file is part of cfvers.
#
# cfvers is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# cfvers is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with cfvers; if not, write to the Free Software Foundation,
# Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

import os, struct, stat, os.path, re, commands, sys
import base64
import types
import random
import difflib
import time
import bz2
import sha
import errno
import string
from cStringIO import StringIO
from mx import DateTime

__all__ = ["Area", "Item", "RevEntry", "AreaRev", "forcejoin", "rexists"]

def forcejoin(a, *p):
    """Join two or more pathname components, considering them all relative"""
    path = a
    for b in p:
        if path[-1:] == '/' and b[:1] == '/':
            path = path + b[1:]
        elif path[-1:] != '/' and b[:1] != '/':
            path = path + '/' + b
        else:
            path = path + b
    return path

def rexists(filename):
    """Checks if a file exists, even if is a broken symlink"""
    try:
        os.lstat(filename)
    except OSError, e:
        if e.errno == errno.ENOENT:
            return False
    return True
            
class Area(object):
    """Class implementing an area object.

    An area is a independent group of versioned items in a
    repository. You will usually use more than one area for different
    servers in a networked repository, and/or for multiple chroot
    jails on one server.
    
    """
    __slots__ = ["name", "revno", "_ctime", "root",
                 "description", "numitems", "revno"]
    
    def __init__(self, name=None,
                 description="", root="/", ctime = None,
                 numitems=0, revno=0):
        """Constructor for the Area class"""
        self.name = name
        self.ctime = ctime or DateTime.utc()
        self.description = description
        self.root = root
        self.numitems = int(numitems)
        self.revno = revno
        return

    def _set_ctime(self, val):
        if isinstance(val, types.StringTypes):
            val = DateTime.ISO.ParseDateTimeUTC(val)
            val = val.localtime()
        self._ctime = val

    def _get_ctime(self):
        return self._ctime

    ctime = property(_get_ctime, _set_ctime, None,
                     "The creation time of this area")
    

class Item(object):
    __slots__ = ["id", "area", "name", "ctime", "dirname"]
    
    def __init__(self, id=-1, area=None, name=None,
                 ctime=None, dirname=None):
        if name is None or not name.startswith("/"):
            raise ValueError, "Invalid name '%s'" % name
        self.id = id
        self.area = area
        self.name = name
        if dirname is None:
            self.dirname = os.path.dirname(name)
        else:
            self.dirname = dirname
        if ctime is None:
            self.ctime = DateTime.utc()
        elif not isinstance(ctime, DateTime.DateTimeType):
            self.ctime = DateTime.ISO.ParseDateTimeUTC(ctime)
        else:
            self.ctime = ctime
        return

    def __str__(self):
        return "Item: id #%d, name %s" % (self.id, self.name)

class RevEntry(object):
    __slots__ = ["item", "revno",
                 "filename", "filetype",
                 "filecontents",
                 "mode",
                 "mtime", "atime", "ctime",
                 "inode", "device", "nlink",
                 "uid", "gid",
                 "sha1sum", "size",
                 "blocks", "rdev", "blksize",
                 ]
    
    modemap = {
        stat.S_IFDIR: 'directory',
        stat.S_IFREG: 'regular file',
        stat.S_IFLNK: 'symbolic link',
        stat.S_IFBLK: 'block device',
        stat.S_IFCHR: 'character device',
        stat.S_IFIFO: 'pipe',
        stat.S_IFSOCK: 'socket',
        }
                 
    def __init__(self, vi = None, revno = None):
        if vi is None:
            # the init will be done manually
            # ugly....
            return
        source = forcejoin(vi.area.root, vi.name)
        self.filename = vi.name
        self.item = vi.id
        self.revno = revno
        st = os.lstat(source)

        # Read mandatory stat attributes
        self.filetype = stat.S_IFMT(st.st_mode)
        # Read filecontents
        if self.filetype == stat.S_IFREG:
            self.filecontents = file(source, "r").read()
        elif self.filetype == stat.S_IFDIR:
            self.filecontents = "\n".join(os.listdir(source))
        elif self.filetype == stat.S_IFLNK:
            self.filecontents = os.readlink(source)
        else:
            self.filecontents = ""
        psha = sha.new(self.filecontents)
        self.sha1sum = psha.hexdigest()
        self.inode = st.st_ino
        self.device = st.st_dev
        self.nlink = st.st_nlink
        self.mode = st.st_mode
        self.mtime = st.st_mtime
        self.atime = st.st_atime
        self.ctime = st.st_ctime
        self.uid = st.st_uid
        self.gid = st.st_gid
        self.size = st.st_size

        # Try to read optional stat components
        self.blocks = getattr(st, 'st_blocks', None)
        self.blksize = getattr(st, 'st_blksize', None)
        if self.filetype == stat.S_IFBLK or self.filetype == stat.S_IFCHR:
            self.rdev = getattr(st, 'st_rdev', None)
        else:
            self.rdev = None
        return

    def _diffdata(older, newer, ofname=None, nfname=None, oftime=None, nftime=None):
        a = older.splitlines(True)
        b = newer.splitlines(True)
        differ = difflib.unified_diff(
            a, b,
            ofname, nfname,
            oftime, nftime,
            )
        data = "".join(differ)
        return data
    
    _diffdata = staticmethod(_diffdata)
    
    def to_filesys(self, destdir=None, use_dirs=1):
        """Writes the revision entry to the filesystem.

        This is one of the most important functions in the whole
        software. It tries to restore a given version to the
        filesystem, with almost all the attributes intact (ctime can't
        be restored, as far as I know).

        """

        self._check_sum()
        if destdir is None:
            target = self.filename
        else:
            if use_dirs:
                target = forcejoin(destdir, self.filename)
            else:
                target = os.path.join(destdir, os.path.basename(self.filename))

        if self.filetype not in (stat.S_IFREG, stat.S_IFLNK, stat.S_IFCHR,
                                 stat.S_IFBLK, stat.S_IFIFO, stat.S_IFDIR):
            print >>sys.stderr, "Skipping: file type %s (%s) cannot be restored." \
                  % (self.modemap[self.filetype], self.filename)
            return

        if (self.filetype == stat.S_IFBLK or self.filetype == stat.S_IFCHR) \
           and self.rdev is None:
            print >>sys.stderr, "Skipping: st_rdev information not available, can't restore device file!"
            return

        do_create  = True
        do_payload = True
        do_attrs   = True
        do_rename  = True
        
        # check for parent existence
        targetbase = os.path.dirname(target)
        if not rexists(targetbase):
            try:
                os.makedirs(targetbase)
            except OSError, e:
                print >>sys.stderr, "Warning: While making intermediate directories: %s" % e

        if self.filetype == stat.S_IFDIR and rexists(target):
            if not os.path.islink(target) and not os.path.isdir(target):
                print >>sys.stderr, "Error: Can't overwrite non-directory with directory!"
                return
            else: # The directory already exists; we must restore ownership and attrs
                do_create  = False
                do_payload = False
                do_rename  = False
                newname = target
        else:
            if not os.path.islink(target) and os.path.isdir(target):
                print >>sys.stderr, "Can't overwrite directory with non-directory!"
                return
            newname = "%s.%07d" % (target, random.randint(0, 999999))
            retries = 0
            while rexists(newname) and retries < 1000:
                newname = "%s.%07d" % (target, random.randint(0, 999999))
                retries += 1
            if rexists(newname):
                print >>sys.stderr, "Error: Can't create a temporary filename! Programmer error or race attack?"
                return

        oldumask = os.umask(0777)
        # try...finally for umask restoration
        try:
            must_remove_new = False
            # The ideea is the operation is done in five steps:
            # 1. creation of item; can fail; fatal
            # 2. if item is file, write contents; can fail; fatal
            # 3. change ownership; can fail; non-fatal
            # 4. change permissions and timestamps; shouldn't fail
            #    except for symlinks; fatal
            # 5. rename to target; can fail; fatal

            # Step 1
            if do_create:
                try:
                    if self.filetype == stat.S_IFIFO:
                        os.mkfifo(newname, 0)
                    elif self.filetype == stat.S_IFREG:
                        fd = os.open(newname, os.O_WRONLY|os.O_CREAT|os.O_EXCL|os.O_NOCTTY)
                    elif self.filetype == stat.S_IFLNK:
                        os.symlink(self.filecontents, newname)
                    elif self.filetype == stat.S_IFCHR or self.filetype == stat.S_IFBLK:
                        os.mknod(newname, self.mode, self.rdev)
                    elif self.filetype == stat.S_IFDIR:
                        os.mkdir(newname, 0)
                    else:
                        raise TypeError, "Programming error: shouldn't have to handle file type %s!" % self.modemap[self.filetype]
                        return
                except OSError, e:
                    print >>sys.stderr, "Error: error '%s' while creating temporary file." % e
                    return
                else:
                    must_remove_new = True
            # From now on, we must cleanup on exit (done via ...finally)
            # Step 2
            if do_payload:
                try:
                    if self.filetype == stat.S_IFREG:
                        os.write(fd, self.filecontents)
                        os.close(fd)
                except OSError, e:
                    print >>sys.stderr, "Error: error '%s' while writing file contents." % e
                    return
            # Step 3
            if do_attrs:
                try:
                    os.lchown(newname, self.uid, self.gid)
                except OSError, e:
                    print >>sys.stderr, "Warning: error '%s' while modifying temporary file ownership." % e

                # WARNING: don't chmod for symlinks, it acts on the target!!!
                if not self.filetype == stat.S_IFLNK:
                    # Step 4
                    try:
                        os.chmod(newname, self.mode)
                        os.utime(newname, (self.atime, self.mtime))
                    except OSError, e:
                        print >>sys.stderr, "Error: error '%s' while modifying temporary file attributes." % e
                        return
            # Step 5
            if do_rename:
                try:
                    os.rename(newname, target)
                except OSError, e:
                    print >>sys.stderr, "Error: error '%s' while renaming" % e
                    raise
                else:
                    must_remove_new = False # We managed to finish!
        finally:
            os.umask(oldumask)
            if must_remove_new:
                try:
                    os.unlink(newname)
                except OSError, e:
                    print >>sys.stderr, "Error: while cleaning-up: %s" % e
        return
        
    def diff(self, older, options=None):
        obuff = StringIO()
        if not isinstance(older, RevEntry):
            raise TypeError("Invalid diff!")
        if self == older:
            return ""
        if self.revno is None:
            newrev = 'current'
        else:
            newrev = "rev %s" % self.revno
        orev = "rev %s" % older.revno
        if self.filetype != older.filetype:
            return "File type has changed: from %s to %s" % \
                   (older.modemap[older.filetype],
                    self.modemap[self.filetype])
        if self.filetype == stat.S_IFREG or self.filetype == stat.S_IFDIR:
            if self.filecontents != older.filecontents:
                if self.printablepayload() and older.printablepayload():
                    obuff.write("== File contents diff: ==\n")
                    data = self._diffdata(
                        older.filecontents, self.filecontents,
                        older.filename, self.filename,
                        "%s (%s)" % (time.ctime(older.mtime), orev),
                        "%s (%s)" % (time.ctime(self.mtime), newrev)
                        )
                    obuff.write(data)
                    obuff.write("\n")
                else:
                    obuff.write("Binary files %s and %s differ\n" % (older.filename, self.filename))
        elif self.filetype == stat.S_IFLNK:
            if self.filecontents != older.filecontents:
                obuff.write("Symlink target changed from '%s' to '%s'" % (older.filecontents, self.filecontents))
        else:
            obuff.write("Don't know how to diff")
        obuff.write("== File metadata diff ==\n")
        for i in ('ctime', 'mtime', 'uid', 'gid', 'mode'):
            oval = getattr(older, i)
            nval = getattr(self, i)
            if oval != nval:
                obuff.write("- %s: %s\n" % (i, oval))
                obuff.write("+ %s: %s\n" % (i, nval))
        return obuff.getvalue()

    def __eq__(self, other):
        if not isinstance(other, RevEntry):
            return NotImplemented
        # Do not test atime, it's irrelevant
        for i in ('filename', 'filetype', 'mode',
                  'mtime', 'uid', 'gid',
                  'filecontents', 'rdev'):
            if getattr(self, i) != getattr(other, i):
                return False
        return True

    def printablepayload(self):
        for i in self.filecontents:
            if i not in string.printable:
                return False
        return True
    
    def isdir(self):
        return self.filetype == stat.S_IFDIR

    def isreg(self):
        return self.filetype == stat.S_IFREG

    def islnk(self):
        return self.filetype == stat.S_IFLNK

    def isblk(self):
        return self.filetype == stat.S_ISBLK

    def ischr(self):
        return self.filetype == stat.S_ISCHR

    def ififo(self):
        return self.filetype == stat.S_IFIFO

    def ifsock(self):
        return self.filetype == stat.S_IFSOCK

    def mode2str(self):
        def mapbit(mode, bit, y):
            if mode & bit:
                return y
            else:
                return '-'
            
        modemap = {
            stat.S_IFDIR: 'd',
            stat.S_IFREG: '-',
            stat.S_IFLNK: 'l',
            stat.S_IFBLK: 'b',
            stat.S_IFCHR: 'c',
            stat.S_IFIFO: 'p',
            stat.S_IFSOCK: 's',
            }
        tchar = modemap.get(self.filetype, '?')
        tchar += mapbit(self.mode, stat.S_IRUSR, 'r')
        tchar += mapbit(self.mode, stat.S_IWUSR, 'w')
        if self.mode & stat.S_ISUID:
            tchar += 'S'
        else:
            tchar += mapbit(self.mode, stat.S_IXUSR, 'x')
        tchar += mapbit(self.mode, stat.S_IRGRP, 'r')
        tchar += mapbit(self.mode, stat.S_IWGRP, 'w')
        if self.mode & stat.S_ISGID:
            tchar += 'S'
        else:
            tchar += mapbit(self.mode, stat.S_IXGRP, 'x')
        tchar += mapbit(self.mode, stat.S_IROTH, 'r')
        tchar += mapbit(self.mode, stat.S_IWOTH, 'w')
        tchar += mapbit(self.mode, stat.S_IXOTH, 'x')
        return tchar

    def _check_sum(self):
        psha = sha.new(self.filecontents)
        psum = psha.hexdigest()
        if psum != self.sha1sum:
            raise ValueError, "Invalid checksum in file contents (%s != %s)!" % (psum, self.sha1sum)
        return

    def stat(self):
        obuf=StringIO()
        obuf.write("  File: `%s'\n" % self.filename)
        obuf.write("  Size: %-15d Blocks: " % self.size)
        if self.blocks is None:
            obuf.write("%-10s" % "N/A")
        else:
            obuf.write("%-10d" % self.blocks)
        obuf.write("IO Block: ")
        if self.blksize is None:
            obuf.write("%-6s" % "N/A")
        else:
            obuf.write("%-6d " % self.blksize)
        obuf.write(self.modemap[self.filetype])
        obuf.write("\n")
        obuf.write("Device: %3xh/%3dd       Inode: %-11d Links: %d\n" %
                   (self.device, self.device, self.inode, self.nlink))
        obuf.write("Access: (%04o/%s) Uid: (%5d/%8s)   Gid: (%5d/%8s)\n" %
                   (stat.S_IMODE(self.mode), self.mode2str(),
                    self.uid, "",
                    self.gid, "")
                   )
        for i in (('Access', self.atime),
                  ('Modify', self.mtime),
                  ('Change', self.ctime)):
            ovar = DateTime.localtime(i[1])
            obuf.write("%s: %s %s\n" % (i[0], ovar.strftime("%Y-%m-%d %T.000000000"), ovar.tz))
        return obuf.getvalue()
            
class AreaRev(object):
    __slots__ = ["area", "revno", "logmsg", "uid", "gid",
                 "commiter", "_ctime", "itemids", "server"]

    def __init__(self, area=None, logmsg=None, commiter=None,
                 server=None):
        if area is None and logmsg is None:
            # manual initialization
            return
        self.area = area.name
        if area is None or area.revno is None:
            self.revno = 1
        else:
            self.revno = area.revno + 1
        if server is None:
            self.server = os.uname()[1]
        else:
            self.server = server
        self.logmsg = logmsg
        self._ctime = DateTime.utc()
        self.uid = os.getuid()
        self.gid = os.getgid()
        if commiter is None:
            try:
                self.commiter = os.getlogin()
            except OSError:
                self.commiter = '<unknown>'
        else:
            self.commiter = commiter
        self.itemids = []

    def _set_ctime(self, val):
        if isinstance(val, types.StringTypes):
            val = DateTime.ISO.ParseDateTimeUTC(val)
        self._ctime = val

    def _get_ctime(self):
        return self._ctime

    ctime = property(_get_ctime, _set_ctime, None,
                     "The creation time of this revision")
