"""Module implementing a SQLite based repository for cfvers"""

# Copyright 2003 Iustin Pop
#
# This file is part of cfvers.
#
# cfvers is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# cfvers is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with cfvers; if not, write to the Free Software Foundation,
# Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

import sqlite

import cfvers.repository.sql
import cfvers

class RSqlite(cfvers.repository.sql.RSql):
    def __init__(self, create=False, cnxargs=None, createopts=None):
        self.conn = sqlite.connect(cnxargs)
        self.backend = "sqlite"
        self.cnxargs = cnxargs
        if create:
            self._create(createopts=createopts)
        self._check_schema(self.conn.cursor())
        return

    def areas(self):
        cursor = self.conn.cursor()
        cursor.execute("select name, root, ctime, description from areas")
        c2 = self.conn.cursor()
        areas = []
        for row in cursor.fetchall():
            c2.execute("select count(*) from items where items.area = %s", (row[0],))
            nitems = int(c2.fetchone()[0])
            c2.execute("select max(revno) from arearevs where area = %s", (row[0],))
            revno = c2.fetchone()[0]
            if revno is not None:
                revno = int(revno)
            areas.append(cfvers.Area(name=row[0], root=row[1], ctime=row[2],
                                     numitems=nitems, revno=revno,
                                     description=row[3]))
        return areas
    
    def getArea(self, name):
        cursor = self.conn.cursor()
        cursor.execute("select name, root, ctime, description from areas where name = %s", (name,))
        row = cursor.fetchone()
        if row is None:
            return None
        c2 = self.conn.cursor()
        c2.execute("select count(*) from items where items.area = %s", (name,))
        nitems = int(c2.fetchone()[0])
        c2.execute("select max(revno) from arearevs where area = %s", (name,))
        revno = c2.fetchone()[0]
        if revno is not None:
            revno = int(revno)
        area = cfvers.Area(name=row[0], root=row[1], ctime=row[2],
                           numitems=nitems, revno=revno,
                           description=row[3])
        return area
    
    def getEntries(self, area, revno, do_payload=True, strict_ver=False):
        cursor = self.conn.cursor()
        if revno is None:
            rcond = ""
        else:
            if strict_ver:
                rcond = " and r.revno = %d" % revno
            else:
                rcond = " and r.revno <= %d " % revno
        if do_payload:
            pfield = 'r.filecontents'
        else:
            pfield = 'Null'
        cursor.execute("select distinct on (r.item) r.item, r.revno, r.filename, r.filetype, %s, r.mode, r.mtime, r.atime, r.uid, r.gid, r.rdev, r.encoding, r.sha1sum, r.size, r.ctime, r.inode, r.device, r.nlink, r.blocks, r.blksize from revisions r, items i where i.area = %%s and i.id = r.item %s order by r.item asc, r.revno desc" % (pfield, rcond), (area.name,))
        while True:
            row = cursor.fetchone()
            if row is None:
                break
            rev = RevEntry()
            (rev.item, rev.revno, rev.filename, rev.filetype,
             payload, rev.mode, rev.mtime, rev.atime,
             rev.uid, rev.gid, rev.rdev, encoding, rev.sha1sum,
             rev.size, rev.ctime, rev.inode, rev.device, rev.nlink,
             rev.blocks, rev.blksize,
             ) = row
            rev.filecontents = self._decode_payload(payload, encoding)
            yield rev
        return
