#include <iostream>
#include <iomanip>
#include "vector.h"
#include "chebyshev.h"
#include "flowfield.h"
#include "nsintegrator.h"

int main() {

  // Define gridsize
  const int Nx=24;
  const int Ny=49;
  const int Nz=24;

  // Define box size
  const Real Lx=1.75*pi;
  const Real a= -1.0;
  const Real b=  1.0;
  const Real Lz=1.2*pi;

  // Define flow parameters
  const Real Reynolds = 400.0;
  const Real nu = 1.0/Reynolds;

  // Define integration parameters
  const int n = 10;     // take ten steps between printouts
  const Real dt = 0.10;
  const Real T  = 100.0;

  // Define size and smoothness of initial disturbance
  Real decay = 0.6;   
  Real magnitude  = 0.1;   
  int kxmax = 3;
  int kzmax = 3;

  // Construct base flow for plane Couette: U(y) = y
  ChebyCoeff U(Ny,a,b,Physical);
  Vector y = chebypoints(Ny, a,b);
  for (int ny=0; ny<Ny; ++ny) 
    U[ny] = y[ny];
  U.save("U");
  y.save("y");
 
  // Construct data fields: 3d velocity and 1d pressure
  FlowField u(Nx,Ny,Nz,3,Lx,Lz,a,b);
  FlowField q(Nx,Ny,Nz,1,Lx,Lz,a,b);

  // Perturb velocity field
  u.addPerturbations(kxmax,kzmax,magnitude,decay);
    
  // Construct Navier-Stoke Integrator
  DNSFlags flags; 
  flags.timestepping = RK3;             // use 3rd-order Runge-Kutta method
  flags.constraint  = PressureGradient; // enforce constant pressure gradient

  NSIntegrator dns(u, U, nu, dt, flags);

  for (Real t=0; t<T; t += n*dt) {
    cout << "===============================================" << endl;
    cout << "         t == " << t << endl;
    cout << "       CFL == " << dns.CFL() << endl;
    cout << "L2Norm2(u) == " << L2Norm2(u) << endl;

    // Save the kx=1,kz=2 Fourier profile and the velocity field
    ComplexChebyCoeff uprofile12 = u.profile(1,2,0);
    uprofile12.makePhysical();
    uprofile12.save("uprofile12_"+i2s(int(t))); 

    // Take n steps of length dt
    dns.advance(u, q, n);
  }
  u.binarySave("u"+i2s(int(T)));
  q.binarySave("q"+i2s(int(T)));
}

