/* basisfunc.cpp: a Complex-Vector-valued Spectral expansion class.
 * Channelflow-0.9
 *
 * Copyright (C) 2001  John F. Gibson  
 *  
 * jgibson@mail.sjcsf.edu  
 * John F. Gibson 
 * St. John's College
 * 1160 Camino de la Cruz Blanca
 * Santa Fe, NM 87501
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, U
 */

#include <fstream>
#include <iomanip>
#include "basisfunc.h"

BasisFunc::BasisFunc()
  :
  Ny_(0),
  kx_(0),
  kz_(0),
  Lx_(0),
  Lz_(0),
  a_(0),
  b_(0),
  state_(Spectral),
  lambda_(0),
  u_(),
  v_(),
  w_()
{}

BasisFunc::BasisFunc(int Ny, const BasisFunc& f)
  :
  Ny_(Ny),
  kx_(f.kx_),
  kz_(f.kz_),
  Lx_(f.Lx_),
  Lz_(f.Lz_),
  a_(f.a_),
  b_(f.b_),
  state_(f.state_),
  lambda_(f.lambda_),
  u_(Ny, f.u_),
  v_(Ny, f.v_),
  w_(Ny, f.w_)
{}

BasisFunc::BasisFunc(int Ny, int kx, int kz, Real Lx, Real Lz, Real a, Real b, 
	    fieldstate s)
  :
  Ny_(Ny),
  kx_(kx),
  kz_(kz),
  Lx_(Lx),
  Lz_(Lz),
  a_(a),
  b_(b),
  state_(s),
  lambda_(0.0),
  u_(Ny, a, b, s),
  v_(Ny, a, b, s),
  w_(Ny, a, b, s)
{}



BasisFunc::BasisFunc(const string& filebase) 
:
  Ny_(0),
  kx_(0),
  kz_(0),
  Lx_(0),
  Lz_(0),
  a_(0),
  b_(0),
  state_(Spectral),
  lambda_(0),
  u_(),
  v_(),
  w_()
{
  string filename = filebase + string(".bf");
  ifstream is(filename.c_str());
  if (!is.good()) {
    cerr << "BasisFunc::BasisFunc(filebase) : can't open file " << filename << '\n';
    abort();
  }
  char c;
  is >> c;
  if (c != '%') {
    cerr << "BasisFunc::BasisFunc(filebase): bad header in file " << filename << endl;
    assert(false);
  }
  int newNy;
  is >> newNy
     >> kx_
     >> kz_
     >> Lx_
     >> Lz_
     >> a_
     >> b_
     >> state_ 
     >> lambda_;
  u_.setBounds(a_,b_);
  v_.setBounds(a_,b_);
  w_.setBounds(a_,b_);
  u_.setState(state_);
  v_.setState(state_);
  w_.setState(state_);

  resize(newNy);
  Real r;
  Real i;
  for (int n=0; n<Ny_; ++n) {
    is >> r >> i; u_.set(n,r+I*i);
    is >> r >> i; v_.set(n,r+I*i);
    is >> r >> i; w_.set(n,r+I*i);
  }
}


/***************
BasisFunc::BasisFunc(istream& is)
  :
  Ny_(0),
  kx_(0),
  kz_(0),
  Lx_(0),
  Lz_(0),
  a_(0),
  b_(0)
  state_(Spectral),
  lambda_(0)
  u_(),
  v_(),
  w_()
{
  binaryLoad(is);
}
****************/


BasisFunc::BasisFunc(const ComplexChebyCoeff& u,
		     const ComplexChebyCoeff& v,
		     const ComplexChebyCoeff& w,
		     Real lam, int kx, int kz, Real Lx, Real Lz) 
:
  Ny_(u.numModes()),
  kx_(kx),
  kz_(kz),
  Lx_(Lx),
  Lz_(Lz),
  a_(u.a()),
  b_(u.b()),
  state_(u.state()),
  lambda_(lam),
  u_(u),
  v_(v),
  w_(w)
{
  assert(u.congruent(v) && u.congruent(w));
}

void BasisFunc::save(const string& filebase) const {
  string filename(filebase);
  filename += string(".bf");
  ofstream os(filename.c_str());
  os << setprecision(REAL_DIGITS);
  char s=' ';
  //if (MATLABVERSION > 4)
    os << '%'
     <<s<< Ny_
     <<s<< kx_
     <<s<< kz_
     <<s<< Lx_
     <<s<< Lz_
     <<s<< a_
     <<s<< b_
     <<s<< state_ 
     <<s<< lambda_
     << '\n';

  for (int i=0; i<Ny_; ++i) 
    os << Re(u_[i]) <<s<< Im(u_[i]) <<s
       << Re(v_[i]) <<s<< Im(v_[i]) <<s
       << Re(w_[i]) <<s<< Im(w_[i]) <<'\n';
  os.close();
}

void BasisFunc::m4save(const string& filebase) const {
  string filename = filebase + string(".asc");
  ofstream os(filename.c_str());
  os << setprecision(REAL_DIGITS);
  char s=' ';
  for (int i=0; i<Ny_; ++i) 
    os << Re(u_[i]) <<s<< Im(u_[i]) <<s
       << Re(v_[i]) <<s<< Im(v_[i]) <<s
       << Re(w_[i]) <<s<< Im(w_[i]) <<'\n';
  os.close();
}

void BasisFunc::binaryDump(ostream& os) const {
  write(os, Ny_);
  write(os, kx_);
  write(os, kz_);
  write(os, Lx_);
  write(os, Lz_);
  write(os, a_);
  write(os, b_);
  write(os, state_);
  write(os, lambda_);
  u_.binaryDump(os);
  v_.binaryDump(os);
  w_.binaryDump(os);
}

void BasisFunc::binaryLoad(istream& is) {
  if (!is.good()) {
    cerr << "BasisFunc::binaryLoad(istream) : input error" << endl;
    abort();
  }
  read(is, Ny_);
  read(is, kx_);
  read(is, kz_);
  read(is, Lx_);
  read(is, Lz_);
  read(is, a_);
  read(is, b_);
  read(is, state_);
  read(is, lambda_);
  u_.binaryLoad(is);
  v_.binaryLoad(is);
  w_.binaryLoad(is);
}
  
void BasisFunc::randomize(bool adiri, bool bdiri, Real decay) {
  
  setToZero();
  setState(Spectral);

  ComplexChebyCoeff u(Ny_, a_, b_, Spectral);
  ComplexChebyCoeff v(Ny_, a_, b_, Spectral);
  ComplexChebyCoeff w(Ny_, a_, b_, Spectral);
  ComplexChebyCoeff vy(Ny_, a_, b_, Spectral);

  
  // Make a u,v part
  if (kx_ == 0) {
    u.randomize(decay);
    ubcFix(u, adiri, bdiri);
  }
  else {
    v.randomize(decay);
    vbcFix(v, adiri, bdiri);
    diff(v, u);
    u *= I*Lx_/(2*pi*kx_);
  }
  u_ += u;
  v_ += v;

  u.setToZero();
  v.setToZero();
  w.setToZero();

  // Make a v,w, part
  if (kz_ == 0) {
    w.randomize(decay);
    ubcFix(w, adiri, bdiri);
  }
  else {
    v.randomize(decay);
    vbcFix(v, adiri, bdiri);
    diff(v, w);
    w *= I*Lz_/(2*pi*kz_);
  }
  w_ += w;
  v_ += v;
  
  if (kx_ == 0 && kz_ == 0) {
    u_.im.setToZero();
    w_.im.setToZero();
    v_.re.setToZero(); // should already be zero
  }
}
void BasisFunc::interpolate(const BasisFunc& phi) {
  assert(a_ >= phi.a_ && b_ <= phi.b_);
  assert(phi.state_ == Spectral);
  u_.interpolate(phi.u_);
  v_.interpolate(phi.v_);
  w_.interpolate(phi.w_);
  state_ = Physical;
}

void BasisFunc::reflect(const BasisFunc& phi) {
  assert((a_+b_)/2 == phi.a() && b_ <= phi.b() && b_ > phi.a());
  assert(phi.state_ == Spectral);
  u_.reflect(phi.u_, Odd);
  v_.reflect(phi.v_, Even);
  w_.reflect(phi.w_, Odd);
  //ChebyCoeff tmp = v_.re;
  //v_.re = v_.im;
  //v_.im = tmp;
  state_ = Physical;
}
  
void BasisFunc::resize(int N) {
  if (N != Ny_) {
    Ny_ = N;
    u_.resize(Ny_);
    v_.resize(Ny_);
    w_.resize(Ny_);
  }
}

void BasisFunc::setBounds(Real Lx, Real Lz, Real a, Real b) {
  Lx_ = Lx;
  Lz_ = Lz;
  u_.setBounds(a,b);
  v_.setBounds(a,b);
  w_.setBounds(a,b);
}
void BasisFunc::setkxkz(int kx, int kz) {
  kx_=kx;
  kz_=kz;
}
void BasisFunc::setState(fieldstate s) {
  assert(s == Spectral || s == Physical);
  u_.setState(s);
  v_.setState(s);
  w_.setState(s);
  state_ = s;
}

void BasisFunc::setRmsval(Real lambda) {
  lambda_ = lambda;
}

void BasisFunc::setToZero() {
   u_.setToZero();
   v_.setToZero();
   w_.setToZero();
}

void BasisFunc::fill(const BasisFunc& f) {
   u_.fill(f.u_);
   v_.fill(f.v_);
   w_.fill(f.w_);
}
void BasisFunc::conjugate() {
  u_.conjugate();
  v_.conjugate();
  w_.conjugate();
  kx_ *= -1;
  kz_ *= -1; // Note that no neg vals of kz appear in FlowFields.
}

Real BasisFunc::bcNorm(bool a_dirichlet, bool b_dirichlet) const {
  Real err=0.0;
  if (b_dirichlet) 
    err += norm2(u_.eval_b()) + norm2(v_.eval_b()) + norm2(w_.eval_b());
  if (a_dirichlet)
    err += norm2(u_.eval_a()) + norm2(v_.eval_a()) + norm2(w_.eval_a());
  return sqrt(err);
}

Real BasisFunc::divNorm() const {
  alwaysAssert(state_ == Spectral);
  ComplexChebyCoeff div = u_;
  div *= (2*pi*kx_/Lx_)*I;
  ComplexChebyCoeff tmp = w_;
  tmp *= (2*pi*kz_/Lz_)*I;
  div += tmp;
  diff(v_, tmp);
  div += tmp;
  return L2Norm(div);
}

bool BasisFunc::geometryCongruent(const BasisFunc& Phi) const {
  return (Ny_==Phi.Ny_ && Lx_==Phi.Lx_ && Lz_==Phi.Lz_ &&
	  a_==Phi.a_ && b_==Phi.b_);
}

bool BasisFunc::congruent(const BasisFunc& Phi) const {
  return (geometryCongruent(Phi) && kx_==Phi.kx_ && kz_==Phi.kz_
	  && state_ == Phi.state_);
}

void BasisFunc::chebyfft() {
  assert(state_ == Physical);
  ChebyTransform t(Ny_);
  t.chebyfft(u_);
  t.chebyfft(v_);
  t.chebyfft(w_);
  state_ = Spectral;
}

void BasisFunc::ichebyfft() {
  assert(state_ == Spectral);
  ChebyTransform t(Ny_);
  t.ichebyfft(u_);
  t.ichebyfft(v_);
  t.ichebyfft(w_);
  state_ = Physical;
}
void BasisFunc::makeSpectral()  {
  ChebyTransform t(Ny_);
  t.makeSpectral(u_);
  t.makeSpectral(v_);
  t.makeSpectral(w_);
  state_ = Spectral;
}
void BasisFunc::makePhysical() {
  ChebyTransform t(Ny_);
  t.makePhysical(u_);
  t.makePhysical(v_);
  t.makePhysical(w_);
  state_ = Physical;
}
void BasisFunc::makeState(fieldstate s) {
  if (state_ != s) {
    ChebyTransform t(Ny_);
    if (s == Physical) {
      t.makePhysical(u_);
      t.makePhysical(v_);
      t.makePhysical(w_);
      state_ = Physical;
    }
    else {
      t.makeSpectral(u_);
      t.makeSpectral(v_);
      t.makeSpectral(w_);
      state_ = Spectral;
    }
  }
}

void BasisFunc::chebyfft(const ChebyTransform& t) {
  assert(state_ == Physical);
  t.chebyfft(u_);
  t.chebyfft(v_);
  t.chebyfft(w_);
  state_ = Spectral;
}

void BasisFunc::ichebyfft(const ChebyTransform& t) {
  assert(state_ == Spectral);
  t.ichebyfft(u_);
  t.ichebyfft(v_);
  t.ichebyfft(w_);
  state_ = Physical;
}
void BasisFunc::makeSpectral(const ChebyTransform& t)  {
  t.makeSpectral(u_);
  t.makeSpectral(v_);
  t.makeSpectral(w_);
  state_ = Spectral;
}
void BasisFunc::makePhysical(const ChebyTransform& t) {
  t.makePhysical(u_);
  t.makePhysical(v_);
  t.makePhysical(w_);
  state_ = Physical;
}
void BasisFunc::makeState(fieldstate s, const ChebyTransform& t) {
  if (s == Physical) {
    t.makePhysical(u_);
    t.makePhysical(v_);
    t.makePhysical(w_);
    state_ = Physical;
  }
  else {
    t.makeSpectral(u_);
    t.makeSpectral(v_);
    t.makeSpectral(w_);
    state_ = Spectral;
  }
}


const ComplexChebyCoeff& BasisFunc::u() const {return u_;}
const ComplexChebyCoeff& BasisFunc::v() const {return v_;}
const ComplexChebyCoeff& BasisFunc::w() const {return w_;}
ComplexChebyCoeff& BasisFunc::u() {return u_;}
ComplexChebyCoeff& BasisFunc::v() {return v_;}
ComplexChebyCoeff& BasisFunc::w() {return w_;}

const ComplexChebyCoeff& BasisFunc::component(int i) const {
  assert(i>=0 && i<=2);
  switch(i) {
  case (0): return u_;
  case (1): return v_;
  case (2): return w_;
  default: 
    exit(-1); // so the compiler doesn't complain
    return w_;
  }
}
ComplexChebyCoeff& BasisFunc::component(int i) {
  assert(i>=0 && i<=2);
  switch(i) {
  case (0): return u_;
  case (1): return v_;
  case (2): return w_;
  default: 
    exit(-1);
    return w_; // so the compiler doesn't complain
  }
}
const ComplexChebyCoeff& BasisFunc::operator[](int i) const {
  assert(i>=0 && i<=2);
  switch(i) {
  case (0): return u_;
  case (1): return v_;
  case (2): return w_;
  default: 
    exit(-1); // so the compiler doesn't complain
    return w_;
  }
}
ComplexChebyCoeff& BasisFunc::operator[](int i) {
  assert(i>=0 && i<=2);
  switch(i) {
  case (0): return u_;
  case (1): return v_;
  case (2): return w_;
  default: 
    exit(-1);
    return w_; // so the compiler doesn't complain
  }
}
/**************
bool BasisFunc::operator==(const BasisFunc& Phi) const {
  return !(*this != Phi);
}
bool BasisFunc::operator!=(const BasisFunc& Phi) const {
  return (!congruent(Phi) || u_ != Phi.u_ || v_ != Phi.v_ || w_ != Phi.w_);
}
****************/

bool BasisFunc::operator==(const BasisFunc& Phi) const {
  return (congruent(Phi) && u_==Phi.u_ && v_==Phi.v_ && w_==Phi.w_) 
    ? true : false;
}
bool BasisFunc::operator!=(const BasisFunc& Phi) const {
  return (*this == Phi) ? false : true;
}

BasisFunc& BasisFunc::operator *= (const BasisFunc& phi) {
  u_ *= phi.u_;
  v_ *= phi.v_;
  w_ *= phi.w_;
  return *this;
}

BasisFunc& BasisFunc::operator += (const BasisFunc& phi) {
  u_ += phi.u_;
  v_ += phi.v_;
  w_ += phi.w_;
  return *this;
}
BasisFunc& BasisFunc::operator -= (const BasisFunc& phi) {
  u_ -= phi.u_;
  v_ -= phi.v_;
  w_ -= phi.w_;
  return *this;
}
BasisFunc& BasisFunc::operator *= (Real c) {
  u_ *= c;
  v_ *= c;
  w_ *= c;
  return *this;
}
BasisFunc& BasisFunc::operator *= (Complex c) {
  u_ *= c;
  v_ *= c;
  w_ *= c;
  return *this;
}
   
// ==================================================================
// L2 norms
Real L2Norm(const BasisFunc& phi, bool normalize) {
  Real rtn2 = 
    L2Norm2(phi.u(), normalize) +
    L2Norm2(phi.v(), normalize) + 
    L2Norm2(phi.w(), normalize);
  if (!normalize)
    rtn2 *= phi.Lx()*phi.Lz();
  return sqrt(rtn2);
}

Real L2Norm2(const BasisFunc& phi, bool normalize) {
  Real rtn = 
    L2Norm2(phi.u(), normalize) +
    L2Norm2(phi.v(), normalize) + 
    L2Norm2(phi.w(), normalize);
  if (!normalize)
    rtn *= phi.Lx()*phi.Lz();
  return rtn;
}

Real L2Dist2(const BasisFunc& f, const BasisFunc& g, bool normalize) {
  assert(f.geometryCongruent(g));
  Real rtn=0.0;
  if (f.kx()==g.kx() && f.kz()==g.kz())
    rtn = 
      L2Dist2(f.u(), g.u(), normalize) +
      L2Dist2(f.v(), g.v(), normalize) +
      L2Dist2(f.w(), g.w(), normalize);
  if (!normalize)
    rtn *= f.Lx()*f.Lz();
  return rtn;

}

Real L2Dist(const BasisFunc& f, const BasisFunc& g, bool normalize) {
    return sqrt(L2Dist2(f,g,normalize));
}

Complex L2InnerProduct(const BasisFunc& f, const BasisFunc& g,
		       bool normalize) {
  assert(f.geometryCongruent(g));
  Complex rtn(0.0, 0.0);
  if (f.kx()==g.kx() && f.kz()==g.kz())
    rtn = 
      L2InnerProduct(f.u(), g.u(), normalize) +
      L2InnerProduct(f.v(), g.v(), normalize) +
      L2InnerProduct(f.w(), g.w(), normalize);
  if (!normalize)
    rtn *= f.Lx() * f.Lz();
  return rtn;
}

// ==================================================================
// cheby norms
Real chebyNorm(const BasisFunc& phi, bool normalize) {
  Real rtn2 = 
    chebyNorm2(phi.u(), normalize) +
    chebyNorm2(phi.v(), normalize) + 
    chebyNorm2(phi.w(), normalize);
  if (!normalize)
    rtn2 *= phi.Lx()*phi.Lz();
  return sqrt(rtn2);
}

Real chebyNorm2(const BasisFunc& phi, bool normalize) {
  Real rtn = 
    chebyNorm2(phi.u(), normalize) +
    chebyNorm2(phi.v(), normalize) + 
    chebyNorm2(phi.w(), normalize);
  if (!normalize)
    rtn *= phi.Lx()*phi.Lz();
  return rtn;
}

Real chebyDist2(const BasisFunc& f, const BasisFunc& g, bool normalize) {
  assert(f.geometryCongruent(g));
  Real rtn=0.0;
  if (f.kx()==g.kx() && f.kz()==g.kz())
    rtn = 
      chebyDist2(f.u(), g.u(), normalize) +
      chebyDist2(f.v(), g.v(), normalize) +
      chebyDist2(f.w(), g.w(), normalize);
  if (!normalize)
    rtn *= f.Lx()*f.Lz();
  return rtn;

}

Real chebyDist(const BasisFunc& f, const BasisFunc& g, bool normalize) {
    return sqrt(chebyDist2(f,g,normalize));
}

Complex chebyInnerProduct(const BasisFunc& f, const BasisFunc& g,
			  bool normalize) {
  assert(f.geometryCongruent(g));
  Complex rtn(0.0, 0.0);
  if (f.kx()==g.kx() && f.kz()==g.kz())
    rtn = 
      chebyInnerProduct(f.u(), g.u(), normalize) +
      chebyInnerProduct(f.v(), g.v(), normalize) +
      chebyInnerProduct(f.w(), g.w(), normalize);
  if (!normalize)
    rtn *= f.Lx() * f.Lz();
  return rtn;
}

// ==================================================================
// switchable norms
Real norm(const BasisFunc& phi, normType n, bool normalize) {
  return (n==L2NORM) ? L2Norm(phi,normalize) : chebyNorm(phi,normalize);
}

Real norm2(const BasisFunc& phi, normType n, bool normalize) {
  return (n==L2NORM) ? L2Norm2(phi,normalize) : chebyNorm2(phi,normalize);
}

Real dist2(const BasisFunc& f, const BasisFunc& g, normType n, bool nrmlz) {
  return (n==L2NORM) ? L2Dist2(f,g,nrmlz) : chebyDist2(f,g,nrmlz);
}

Real dist(const BasisFunc& f, const BasisFunc& g,  normType n, bool nrmlz) {
  return (n==L2NORM) ? L2Dist(f,g,nrmlz) : chebyDist(f,g,nrmlz);
}

Complex innerProduct(const BasisFunc& f, const BasisFunc& g, normType n,
		       bool nrmlz) {
  return (n==L2NORM) ? L2InnerProduct(f,g,nrmlz) : chebyInnerProduct(f,g,nrmlz);
}

BasisFunc xdiff(const BasisFunc& phi) {
  BasisFunc phix(phi);
  phix *= (2*pi*phi.kx()/phi.Lx())*I;
  return phix;
}

  
BasisFunc ydiff(const BasisFunc& phi) {
  assert(phi.state() == Spectral);
  BasisFunc phiy(phi.Ny(), phi.kx(), phi.kz(), phi.Lx(), phi.Lz(), phi.a(), phi.b(), Spectral);
  diff(phi.u(), phiy.u());
  diff(phi.v(), phiy.v());
  diff(phi.w(), phiy.w());
  return phiy;
}

BasisFunc zdiff(const BasisFunc& phi) {
  BasisFunc phiz(phi);
  phiz *= (2*pi*phi.kz()/phi.Lz())*I;
  return phiz;
}

ComplexChebyCoeff divergence(const BasisFunc& phi) {
  assert(phi.state() == Spectral);
  ComplexChebyCoeff div(phi.Ny(), phi.a(), phi.b(), Spectral);
  ComplexChebyCoeff tmp(phi.Ny(), phi.a(), phi.b(), Spectral);
  tmp = phi[0];
  tmp *= (2*pi*phi.kx()/phi.Lx())*I;
  div += tmp;

  diff(phi[1], tmp);
  div += tmp;

  tmp = phi[2];
  tmp *= (2*pi*phi.kz()/phi.Lz())*I;
  div += tmp;
  
  return div;
}


BasisFunc laplacian(const BasisFunc& phi) {
  BasisFunc lapl(phi);
  Real c = -4*square(pi)*(square(phi.kx()/phi.Lx())+square(phi.kz()/phi.Lz()));
  lapl.u() *= c;
  lapl.v() *= c;
  lapl.w() *= c;

  ComplexChebyCoeff tmp;
  diff2(phi.u(), tmp);
  lapl.u() += tmp;
  diff2(phi.v(), tmp);
  lapl.v() += tmp;
  diff2(phi.w(), tmp);
  lapl.w() += tmp;
  return lapl;
}

BasisFunc gradient(const ComplexChebyCoeff& f, int kx,int kz, Real Lx,Real Lz){
  assert(f.state() == Spectral);
  BasisFunc grad(f.numModes(), kx, kz, Lx, Lz, f.a(), f.b(), f.state());
  grad.u() = f;
  grad.u() *= (2*pi*kx/Lx)*I;
  diff(f, grad.v());
  grad.w() = f;
  grad.w() *= (2*pi*kz/Lz)*I;
  return grad;
}

BasisFunc gradient(const BasisFunc& phi, int i) {
  assert(phi.state() == Spectral);

  BasisFunc grad(phi.Ny(), phi.kx(), phi.kz(), phi.Lx(), phi.Lz(), phi.a(), phi.b(), phi.state());
  const ComplexChebyCoeff& phi_i = phi[i];

  grad.u() = phi_i;
  grad.u() *= (2*pi*phi.kx()/phi.Lx())*I;

  diff(phi_i, grad.v());

  grad.w() = phi_i;
  grad.w() *= (2*pi*phi.kz()/phi.Lz())*I;
  return grad;
}


ComplexChebyCoeff dot(const BasisFunc& f_, const BasisFunc& g_) { 
  BasisFunc& f = (BasisFunc&) f_;
  BasisFunc& g = (BasisFunc&) g_;
  assert(f.geometryCongruent(g));
  int Ny= f.Ny();
  fieldstate entrystate = f.state();
  ChebyTransform t(Ny);
  f.makePhysical(t);
  g.makePhysical(t);

  ComplexChebyCoeff fg(Ny, f.a(), f.b(), Physical);
  for (int ny=0; ny<Ny; ++ny)
    fg.set(ny, f[0][ny]*g[0][ny] + f[1][ny]*g[1][ny] + f[2][ny]*g[2][ny]);

  if (entrystate==Spectral) {
    f.makeSpectral(t);
    g.makeSpectral(t);
    fg.makeSpectral(t);
  }
  return fg;
}
	   

BasisFunc cross(const BasisFunc& f_, const BasisFunc& g_) {
  BasisFunc& f = (BasisFunc&) f_;
  BasisFunc& g = (BasisFunc&) g_;
  assert(f.geometryCongruent(g)); 
  int Ny= f.Ny();
  
  ChebyTransform t(Ny);
  fieldstate entrystate = f.state();
  f.makePhysical(t);
  g.makePhysical(t);


  BasisFunc rtn(Ny, f.kx()+g.kx(), f.kz()+g.kz(), f.Lx(), f.Lz(), f.a(), f.b(),
		Physical);
  for (int ny=0; ny<Ny; ++ny) {
    Complex fu = f[0][ny];
    Complex fv = f[1][ny];
    Complex fw = f[2][ny];
    Complex gu = g[0][ny];
    Complex gv = g[1][ny];
    Complex gw = g[2][ny];
    
    rtn[0].set(ny, fv*gw - fw*gv);
    rtn[1].set(ny, fw*gu - fu*gw);
    rtn[2].set(ny, fu*gv - fv*gu);
  }
    
  if (entrystate==Spectral) {
    f.makeSpectral(t);
    g.makeSpectral(t);
    rtn.makeSpectral(t);
  }
  return rtn;
}

BasisFunc curl(const BasisFunc& f_) {

  BasisFunc& f = (BasisFunc&) f_;
  fieldstate entrystate = f.state();
  int Ny= f.Ny();
  ChebyTransform t(Ny);
  f.makeSpectral(t);

  BasisFunc f_y = ydiff(f);
  Complex d_dx = (2*pi*f.kx()/f.Lx())*I;
  Complex d_dz = (2*pi*f.kz()/f.Lz())*I;

  BasisFunc rtn(f);
  
  rtn.setToZero();
  for (int ny=0; ny<Ny; ++ny) {
    Complex u  = f[0][ny];
    Complex v  = f[1][ny];
    Complex w  = f[2][ny];
    Complex u_y = f_y[0][ny];
    Complex w_y = f_y[2][ny];
    
    rtn[0].set(ny, w_y - d_dz*v);
    rtn[1].set(ny, d_dz*u - d_dx*w);
    rtn[2].set(ny, d_dx*v - u_y);
  }
    
  if (entrystate==Spectral) {
    f.makeSpectral(t);
    rtn.makeSpectral(t);
  }
  return rtn;
}

// rtn = phi_n dot grad phi_p
void dotgrad(const BasisFunc& phi_n, const BasisFunc& grad_phi_pu,
	     const BasisFunc& grad_phi_pv, const BasisFunc& grad_phi_pw,
	     ComplexChebyCoeff& tmp, BasisFunc& rtn) 
{
  assert(phi_n.state() == Physical);
  assert(grad_phi_pu.state() == Physical);
  assert(grad_phi_pv.state() == Physical);
  assert(grad_phi_pw.state() == Physical);
  assert(phi_n.geometryCongruent(grad_phi_pu));
  assert(phi_n.geometryCongruent(grad_phi_pv));
  assert(phi_n.geometryCongruent(grad_phi_pw));
  rtn.setToZero();
  rtn.setkxkz(phi_n.kx()+grad_phi_pu.kx(), phi_n.kz()+grad_phi_pu.kz());
  rtn.setBounds(phi_n.Lx(), phi_n.Lz(), phi_n.a(), phi_n.b());
  rtn.setState(Physical);

  tmp.setToZero();
  tmp.setBounds(phi_n.a(), phi_n.b());
  tmp.setState(Physical);

  // =====================================
  // Calculate rtn.u =   phi_n.u d/dx phi_p.u
  //                   + phi_n.v d/dy phi_p.u
  //                   + phi_n.w d/dz phi_p.u
  int i; // MSVC++ FOR-SCOPE BUG
  for (i=0; i<3; ++i) {
    tmp = phi_n[i];
    tmp *= grad_phi_pu[i];
    rtn.u() += tmp;
  }
  // =====================================
  // Calculate rtn.v =   phi_n.u d/dx phi_p.v
  //                   + phi_n.v d/dy phi_p.v
  //                   + phi_n.w d/dz phi_p.v
  for (i=0; i<3; ++i) {
    tmp = phi_n[i];
    tmp *= grad_phi_pv[i];
    rtn.v() += tmp;
  }
  // =====================================
  // Calculate rtn.w =   phi_n.u d/dx phi_p.w
  //                   + phi_n.v d/dy phi_p.w
  //                   + phi_n.w d/dz phi_p.w
  for (i=0; i<3; ++i) {
    tmp = phi_n[i];
    tmp *= grad_phi_pw[i];
    rtn.w() += tmp;
  }
  return;
}

BasisFunc dotgrad(const BasisFunc& phi_n_arg, const BasisFunc& phi_p_arg) {
  BasisFunc& phi_n = (BasisFunc&) phi_n_arg;
  BasisFunc& phi_p = (BasisFunc&) phi_p_arg;
  
  fieldstate original_n_state = phi_n.state();
  fieldstate original_p_state = phi_p.state();

  int Ny=phi_n.Ny();
  ChebyTransform trans(Ny);
  phi_p.makeSpectral(trans);   // Now phi_p is spectral

  BasisFunc rtn(Ny, phi_n.kx()+phi_p.kx(), phi_n.kz()+phi_p.kz(), 
		phi_n.Lx(), phi_n.Lz(), phi_n.a(), phi_n.b(), Spectral);

  BasisFunc grad_phi_pu = gradient(phi_p, 0);
  BasisFunc grad_phi_pv = gradient(phi_p, 1);
  BasisFunc grad_phi_pw = gradient(phi_p, 2);
  ComplexChebyCoeff tmp(Ny, phi_n.a(), phi_n.b(), Physical);
  
  phi_n.makePhysical(trans);  // Now phi_n is physical
  grad_phi_pu.makePhysical(trans);
  grad_phi_pv.makePhysical(trans);
  grad_phi_pw.makePhysical(trans);

  dotgrad(phi_n, grad_phi_pu, grad_phi_pv, grad_phi_pw, tmp, rtn);
  
  // phi_n is physical, 
  if (original_n_state == Spectral) {
    phi_n.makeSpectral(trans);
    rtn.makeSpectral(trans);
  }
  // phi_p is spectral.
  if (original_p_state == Physical)  
    phi_p.makePhysical(trans);
  
  return rtn;
}

void ubcFix(ChebyCoeff& u, bool a_dirichlet, bool b_dirichlet) {
  Real ua = u.a();
  Real ub = u.b();
  u.setBounds(-1,1);
  
  if (a_dirichlet && b_dirichlet) {
    u[0] -= 0.5*(u.eval_b() + u.eval_a());
    u[1] -= 0.5*(u.eval_b() - u.eval_a());
  }
  else if (a_dirichlet) 
    u[0] -= u.eval_a();
  else if (a_dirichlet) 
    u[0] -= u.eval_b();

  u.setBounds(ua, ub);
}

void ubcFix(ComplexChebyCoeff& u, bool a_dirichlet, bool b_dirichlet) {
  Real ua = u.a();
  Real ub = u.b();
  u.setBounds(-1,1);
  
  if (a_dirichlet && b_dirichlet) {
    u.sub(0, 0.5*(u.eval_b() + u.eval_a()));
    u.sub(1, 0.5*(u.eval_b() - u.eval_a()));
  }
  else if (a_dirichlet) 
    u.sub(0, u.eval_a());
  else if (a_dirichlet) 
    u.sub(0, u.eval_b());

  u.setBounds(ua, ub);
}

void vbcFix(ChebyCoeff& v, bool a_dirichlet, bool b_dirichlet) {

  Real va = v.a();
  Real vb = v.b();
  v.setBounds(-1,1);
  
  if (a_dirichlet && b_dirichlet) {
    // Adjust v to match v(+-1)=v'(+-1)=0 BCs
    ChebyCoeff vy = diff(v);
    Real a = v.eval_a();
    Real b = v.eval_b(); 
    Real c = vy.eval_a();
    Real d = vy.eval_b(); 

    v[0] -= 0.5*(a + b) + 0.125*(c - d);
    v[1] -= 0.5625*(b - a) - 0.0625*(c + d);
    v[2] -= 0.125*(d - c);
    v[3] -= 0.0625*(a - b + c + d);
  }
  else if (a_dirichlet) {
    ChebyCoeff vy = diff(v);
    v[1] -= vy.eval_a();
    v[0] -= v.eval_a();
  }
  else if (b_dirichlet) {
    ChebyCoeff vy = diff(v);
    v[1] -= vy.eval_b();
    v[0] -= v.eval_b();
  }
  v.setBounds(va,vb);
}
  
void vbcFix(ComplexChebyCoeff& v, bool a_dirichlet, bool b_dirichlet) {

  Real va = v.a();
  Real vb = v.b();
  v.setBounds(-1,1);
  
  if (a_dirichlet && b_dirichlet) {
    // Adjust v to match v(+-1)=v'(+-1)=0 BCs
    ComplexChebyCoeff vy = diff(v);
    Complex a = v.eval_a();
    Complex b = v.eval_b(); 
    Complex c = vy.eval_a();
    Complex d = vy.eval_b(); 

    v.sub(0, 0.5*(a + b) + 0.125*(c - d));
    v.sub(1, 0.5625*(b - a) - 0.0625*(c + d));
    v.sub(2, 0.125*(d - c));
    v.sub(3, 0.0625*(a - b + c + d));
  }
  else if (a_dirichlet) {
    ComplexChebyCoeff vy = diff(v);
    v.sub(1, vy.eval_a());
    v.sub(0,  v.eval_a());
  }
  else if (b_dirichlet) {
    ComplexChebyCoeff vy = diff(v);
    v.sub(1, vy.eval_b());
    v.sub(0,  v.eval_b());
  }
  v.setBounds(va,vb);
}
  

