#include <iostream>
#include <iomanip>
#include "vector.h"
#include "chebyshev.h"
#include "flowfield.h"
#include "nsintegrator.h"

// This program produces data for figures like the User's Manual Fig 1.
// You might need to change how the initial conditions are set to start
// it properly (i.e. read different saved fiels, or start afresh with 
// perturbations).

int main() {

  // Define gridsize
  const int Nx=24;
  const int Ny=49;
  const int Nz=24;

  // Define box size
  const Real Lx=1.75*pi;
  const Real a= -1.0;
  const Real b=  1.0;
  const Real Lz=1.2*pi;

  // Define flow parameters
  const Real Reynolds = 600.0;
  const Real nu = 1.0/Reynolds;

  // Define integration parameters
  const Real dtmax = 0.15;
  const Real dtmin = 0.05;
  const Real CFLmax = 0.90;
  const Real CFLmin = 0.5;
  const Real dT  = 1.0;     // plot interval
  const Real T0  = 000.0;   // start time
  const Real T1  = 300.0;   // stop time

  // Define size and smoothness of initial disturbance
  Real decay = 0.4;   
  Real magnitude  = 0.01;   
  int kxmax = 4;
  int kzmax = 4;

  // Construct base flow for plane Couette: U(y) = y
  ChebyCoeff U(Ny,a,b,Physical);
  Vector x = periodicpoints(Nx,Lx);
  Vector y = chebypoints(Ny, a,b);
  Vector z = periodicpoints(Nz,Lz);
  for (int ny=0; ny<Ny; ++ny) 
    U[ny] = y[ny];
  U.save("U");
  x.save("x");
  y.save("y");
  z.save("z");

  // Construct data fields: 3d velocity and 1d pressure
  //FlowField u("u100");
  //FlowField q("q100");
  // Construct data fields: 3d velocity and 1d pressure
  FlowField u(Nx,Ny,Nz,3,Lx,Lz,a,b);
  FlowField q(Nx,Ny,Nz,1,Lx,Lz,a,b);
  u.addPerturbations(kxmax,kzmax,magnitude,decay);

  FlowField omega(u); 

  // Construct Navier-Stoke Integrator
  DNSFlags flags; 
  flags.timestepping = RK3;             // use 3rd-order Runge-Kutta method
  flags.constraint   = BulkVelocity;     

  TimeStep dt((dtmax+dtmin)/2, dtmin, dtmax, dT, CFLmin, CFLmax);
  NSIntegrator dns(u, U, nu, dt, flags, T0);
  dns.resetUbulk(0.0);

  for (Real t=T0; t<=T1; t += dT) {
    cout << "===============================================" << endl;
    cout << "         t == " << t << endl;
    cout << "       CFL == " << dns.CFL() << endl;
    cout << "L2Norm2(u) == " << L2Norm2(u) << endl;

    // Save the kx=1,kz=2 Fourier profile and the velocity field
    ComplexChebyCoeff uprofile12 = u.profile(1,2,0);
    uprofile12.makePhysical();
    //uprofile12.save("uprofile12_"+i2s(int(t))); 
    
    if (int(t) % 20 == 0 && int(t) != T0) {
      u.binarySave("u"+i2s(int(t)));
      q.binarySave("q"+i2s(int(t)));
    }

    int nz0 = 12;
    u.saveSliceXY(nz0,0,"uside");
    u.saveSliceXY(nz0,1,"vside");
    u.saveSliceXY(nz0,2,"wside");
    u.saveSliceXY(nz0,2,"wside");
    u.saveCrossSection(0,0,"usec");
    u.saveCrossSection(0,1,"vsec");
    u.saveCrossSection(0,2,"wsec");

    u.vorticity2(omega);
    omega.saveCrossSection(0,0,"omsec0");
    omega.saveCrossSection(0,1,"omsec1");
    omega.saveCrossSection(0,2,"omsec2");

    // Take n steps of length dt
    dns.advance(u, q, dt.n());

    if (dt.adjust(dns.CFL())) {
      cout << "adjusting timestep" << endl;
      dns.reset(nu, dt);
    }
  }
}

