/* diffops.h: Differential operators for FlowFields
 * Channelflow-0.9
 *
 * Copyright (C) 2001-2005  John F. Gibson  
 *  
 * jgibson@mail.sjcsf.edu  
 * John F. Gibson 
 * St. John's College
 * 1160 Camino de la Cruz Blanca
 * Santa Fe, NM 87501
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, U
 */

#ifndef DIFFOPS_H
#define DIFFOPS_H

#include "mathdefs.h"
#include "cfvector.h"
#include "chebyshev.h"
#include "flowfield.h"

// helper function for emulating 2-tensors with 9d fields
inline int i3j(int i, int j) {return 3*i + j;}

// In general, these fucntions take input fields in any state and return 
// output fields in Spectral,Spectral leaving inputs unchanged. 
// But it's most efficient to send input fields as Spectral,Spectral.
// Input fields are logically constant, but they may be modified and 
// unmodified (transforms and inverses) during the course of the computation.

//  lapl(f,laplf) computes laplf(nx,ny,nz,i) = d^2 f(nx,ny,nz,i)/dx_j^2
//  grad(f,gradf) computes 
//                gradf(nx,ny,nz,i)        = df(nx,ny,nz,0)/dx_i for 1d f
//                gradf(nx,ny,nz,i3j(i,j)) = df(nx,ny,nz,i)/dx_j for 3d f


// Warning: these functions are convenient but inefficient due to 
// construction and copying of returned FlowFields. Prefer the 
// void-returning versions below for repeated calculations.
FlowField xdiff(const FlowField& f, int n=1);   // d^nf/dx^n
FlowField ydiff(const FlowField& f, int n=1);   // d^nf/dy^n
FlowField zdiff(const FlowField& f, int n=1);   // d^nf/dz^n
FlowField diff(const FlowField& f, int nx, int ny, int nz); // d^(nx+ny+nz)/(dx^nx dy^ny dz^nz) f

FlowField grad(const FlowField& f);   // grad f
FlowField lapl(const FlowField& f);   // lapl f
FlowField curl(const FlowField& f);   // del cross f
FlowField norm(const FlowField& f);   // ||f||
FlowField norm2(const FlowField& f);  // ||f||^2
FlowField div(const FlowField& f);    // del dot f 

FlowField cross(const FlowField& f, const FlowField& g);   
FlowField outer(const FlowField& f, const FlowField& g);//fi gj ei ej
FlowField dot(const FlowField& f, const FlowField& g);   
FlowField energy(const FlowField& u);
FlowField energy(const FlowField& u, ChebyCoeff& U); 


// These functions are preferred 
void xdiff(const FlowField& f, FlowField& dfdx, int n=1);
void ydiff(const FlowField& f, FlowField& dfdy, int n=1);
void zdiff(const FlowField& f, FlowField& dfdz, int n=1);
void diff(const FlowField& f, FlowField& df, int nx, int ny, int nz); 

void grad(const FlowField& f, FlowField& grad_f);   // grad f, 1d->3d or 3d->9d
void lapl(const FlowField& f, FlowField& lapl_f);   // lapl f
void curl(const FlowField& f, FlowField& curl_f);   // del cross f
void norm(const FlowField& f, FlowField& norm_f);   // ||f||
void norm2(const FlowField& f, FlowField& norm2_f); // ||f||^2
void div(const FlowField& f,  FlowField& divf);     // del dot f 

void cross(const FlowField& f, const FlowField& g, FlowField& f_cross_g);   
void outer(const FlowField& f, const FlowField& g, FlowField& fg);//fi gj ei ej
void dot(const FlowField& f, const FlowField& g, FlowField& f_dot_g);   
void energy(const FlowField& u, FlowField& e); 
void energy(const FlowField& u, const ChebyCoeff& U, FlowField& e); 


// Different ways to calculate nonlinear term of Navier-Stokes equation.
// Let u' = u + U ex
// Rotational:  f = (curl u)  cross u + U du/dx + v dUdy ex, tmp = curl u
// Rotational2: f = (curl u') cross u'                       tmp = curl u'
// Convection:  f = u' dot grad u',                          tmp = grad u'
// Divergence:  f = div (u' u'),                             tmp = u' u'
// Skewsymm  :  f = 1/2 [u' dot (grad u') + div (u' u')],    tmp = grad u'
// Alternating:  alternate btwn divergence and convection 
// Alternating_: alternate btwn convection and divergence
// Linearized:  f = u dot grad (U ex) + (U ex) dot grad u
//                = v dU/dy ex + U du/dx

// Note: these functions return f in Spectral, Spectral and u,U in entry states
// The functions are most efficient if u,U enter in Spectral, Spectral.
// If you want dealiased nonlinearities, call u.setAliasedModesToZero() 
// beforehand and possibly f.setAliasedModesToZero() afterwards.

enum NonlinearMethod {Rotational, Convection, Divergence, SkewSymmetric, 
		      Alternating, Alternating_, Linearized}; 

void    rotationalNL(const FlowField& u, const ChebyCoeff& U, FlowField& f, 
		     FlowField& tmp);

void    convectionNL(const FlowField& u, const ChebyCoeff& U, FlowField& f, 
		     FlowField& tmp); 

void    divergenceNL(const FlowField& u, const ChebyCoeff& U, FlowField& f, 
		     FlowField& tmp);

void skewsymmetricNL(const FlowField& u, const ChebyCoeff& U, FlowField& f, 
		     FlowField& tmp);

void    linearizedNL(const FlowField& u, const ChebyCoeff& U, FlowField& f);

void  navierstokesNL(const FlowField& u, const ChebyCoeff& U,  FlowField& f, 
		     FlowField& tmp, NonlinearMethod& method);

ostream& operator<<(ostream& os, NonlinearMethod nonl);

// Convert back and forth from modified pressure q to true pressure field p.
// For rotational compuation of nonlinearity,  q = p + 1/2 u dot u. 
// For other compuations of nonlinearities ,   q = p
void uq2p(const FlowField& u, const FlowField& q, FlowField& p,
	  NonlinearMethod nonlin_method);

void up2q(const FlowField& u, const FlowField& p, FlowField& q,
	  NonlinearMethod nonlin_method);

void matlabSave(FlowField& f, int i, string& filename);

#endif
