/* nsintegrator.h: time-integration class for spectral Navier-Stokes DNS
 * Channelflow-0.9
 *
 * Copyright (C) 2001  John F. Gibson  
 *  
 * jgibson@mail.sjcsf.edu  
 * John F. Gibson 
 * St. John's College
 * 1160 Camino de la Cruz Blanca
 * Santa Fe, NM 87501
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, U
 */

// A C++ class for integrating Navier-Stokes equation.

#ifndef NSINTEGRATOR_H
#define NSINTEGRATOR_H

#include "mathdefs.h"
#include "flowfield.h"
#include "cfvector.h"
#include "chebyshev.h"
#include "tausolver.h"
#include "diffops.h"

enum Dealiasing      {NoDealiasing, DealiasXZ, DealiasY, DealiasXYZ};
enum MeanConstraint  {PressureGradient, BulkVelocity};
enum TimeStepMethod  {CNAB2, RK3};
enum Verbosity       {Silent, PrintTicks, PrintTime, VerifyTauSolve, PrintAll};
//enum NonlinearMethod {Rotational, SkewSymmetric, Convection, Divergence, Alternating, Linearized}; // declared in diffops.h

ostream& operator<<(ostream& os, Dealiasing d);
ostream& operator<<(ostream& os, MeanConstraint m);
ostream& operator<<(ostream& os, TimeStepMethod t);
ostream& operator<<(ostream& os, Verbosity v);

class DNSFlags {
public: 
  //       Option                         Default value
  DNSFlags(MeanConstraint  constraint   = BulkVelocity,
	   TimeStepMethod  timestepping = RK3,
	   NonlinearMethod nonlinearity = Rotational, 
	   Dealiasing      dealiasing   = DealiasXZ,
	   bool            taucorrection = true,
	   Verbosity       verbosity    = PrintTicks);

  MeanConstraint  constraint;
  TimeStepMethod  timestepping;
  NonlinearMethod nonlinearity;
  Dealiasing      dealiasing;
  Verbosity       verbosity;
  bool taucorrection; 

  bool dealias_xz() const;
  bool dealias_y() const;
  
  int numSubsteps() const;
};
ostream& operator<<(ostream& os, DNSFlags& flags);


// TimeStep keeps dt between dtmin and dtmax, and CFL between CFLminand CFLmax,
// in such a way that dt*n = dT for some integer n. That's useful if you
// want to plot/save data at regular dT intervals, but use a variable timestep 
// dt for efficiency. For example of use, see example codes. 

class TimeStep {
public:
  TimeStep(Real dt, Real dtmin, Real dtmax, Real dT, Real CFLmin, Real CFLmax);
  bool adjust(Real CFL); // return true if dt changes, keep CFLmin<CFL<CFLmax
  Real CFL() const;
  int n() const;         // n*dt == dT
  Real dt() const;       // integration timestep
  Real dT() const;       // plot interval
  operator Real() const; // same as dt()

private:

  int n_;     // n_    == int(dT/dt), number of dt steps per plot interval
  int nmin_;  // nmin_ == int(dT/dtmax)
  int nmax_;  // nmax_ == int(dT/dtmin)
  Real dT_;    // plot interval
  Real CFLmin_;
  Real CFL_;
  Real CFLmax_;
};

class NSIntegrator {
public:
  NSIntegrator();
  NSIntegrator(FlowField& u, const ChebyCoeff& Ubase, Real nu, Real dt, 
	       const DNSFlags& flags, Real T0=0);
  ~NSIntegrator();

  NSIntegrator& operator=(const NSIntegrator& ns);
  
  void advance(FlowField& u, FlowField& q, int nSteps=1); 
  void advance(FlowField& u, FlowField& q, Real constraint);

  void resettime(Real t);
  void resetdPdx(Real dPdx);    // change dPdx and enforce const dPdx
  void resetUbulk(Real Ubulk);  // change Ubulk and enforce const Ubulk
  void reset(Real nu, Real dt); // expensive! don't do it every timestep

  Real dt() const;
  Real CFL() const;
  Real time() const;
  Real dPdx() const;      // the mean pressure gradient at the current time
  Real Ubulk() const;     // the actual bulk velocity at the current time

  Real dPdxRef() const;   // the mean press grad enforced during integration
  Real UbulkRef() const;  // the bulk velocity enforced during integ.

protected:
  // Parameters of integration
  int Nx_;      // number of X gridpoints
  int Ny_;      // number of Chebyshev T(y) modes
  int Nyd_;     // number of dealiased Chebyshev T(y) modes 
  int Nz_;      // number of Z gridpoints
  int kxd_max_; // maximum value of kx among dealiased modes
  int kzd_max_; // maximum value of kz among dealiased modes
  int Nsubsteps_; // 3 for implicit RK3, 1 for CNAB.
  Real Lx_;
  Real Lz_;
  Real a_;
  Real b_;
  Real nu_;
  Real dt_;

  // Time-stepping constants
  Real alpha_[3];
  Real beta_[3];
  Real gamma_[3];
  Real zeta_[3];

  DNSFlags flags_;

  // These variables used to keep mean dPdx or mass flux const.
  Real dPdxRef_;   // Enforced mean pressure gradient (0.0 if unused).
  Real dPdxAct_;   // Actual mean pressure gradient at previous timestep.
  Real UbulkRef_;  // Enforced total bulk velocity (0.0 if unused).
  Real UbulkAct_;  // Actual total bulk velocity bulk obtained.
  Real UbulkBase_; // Bulk velocity of base flow

  Real t_;        // time in convective units
  Real cfl_;      // CFL number

  ChebyCoeff Ubase_;   // baseflow physical
  ChebyCoeff Ubasey_;  // baseflow' physical
  ChebyCoeff UbaseyT_; // baseflow' chebyshev
  ChebyCoeff Ubaseyy_; // baseflow'' physical

  FlowField vortn_;  // vorticity field at t = n dt
  FlowField fn_;     // nonlinear term  at t = n dt
  FlowField fn1_;    // nonlinear term  at t = (n-1) dt
  TauSolver*** tausolver_;  // 3d array of tausolvers, index [rk][nx][nz]

  // These variables are used as temp storage when solving indpt tau problems.
  ComplexChebyCoeff uk_;   // profile of u_{kx,kz} (y) at t = n dt
  ComplexChebyCoeff vk_;
  ComplexChebyCoeff wk_;
  ComplexChebyCoeff Rxk_;
  ComplexChebyCoeff Ryk_;
  ComplexChebyCoeff Rzk_;
  ComplexChebyCoeff fxk_;  // stores x comp of nonlinear term
  ComplexChebyCoeff fyk_;  
  ComplexChebyCoeff fzk_;  
  ComplexChebyCoeff Pk_;   // profile of P_{kx,kz} (y)
  ComplexChebyCoeff Pyk_;  // profile of dP_{kx,kz}/dy (y)
  
  void init(const FlowField& u); // shared ctor work.
  void debugfunc(const FlowField& u) const; // see note in nsintegrator.cpp

  int kxmaxDealiased() const;
  int kzmaxDealiased() const;
  bool isAliasedMode(int kx, int kz) const;
};


// Given a baseflow, fluctation, modified pressure triple (U0,u0,q0) and
// a new baseflow U1, compute new fluctuation u1 and modified pressure q1.
//     utot == U0 + u0 == U1 + u1
// pressure == q0 - 1/2 || u0 ||^2 == q1 - 1/2 || u1 ||^2 
void changeBaseFlow(const ChebyCoeff& U0, const FlowField& u0, const FlowField& q0, 
		    const ChebyCoeff& U1, FlowField& u1, FlowField& q1);

#endif 
