/* utilfuncs.cpp: an assortment of mathematical functions convenient for
 * my research problems but probably irrelevant to everyone else
 *
 * Channelflow-0.9
 *
 * Copyright (C) 2001-2005  John F. Gibson
 *
 * John F. Gibson
 * Center for Nonlinear Sciences
 * School of Physics
 * Georgia Institute of Technology
 * Atlanta, GA 30332-0430
 *
 * gibson@cns.physics.gatech.edu
 * jfg@member.fsf.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, U
 */

#include "utilfuncs.h"

// Algorithm from Numerical Recipes in C pg 110;
Real polyInterp(const array<Real>& fa, const array<Real>& xa, Real x) {
  assert(fa.N() == xa.N());
  int N = fa.N();
  array<Real> C(N);
  array<Real> D(N);
  
  Real xdiff = abs(x-xa[0]);
  Real xdiff2 = 0.0;
  Real df;

  int i_closest = 0;
  for (int i=0; i<N; ++i) {
    if ((xdiff2 = abs(x-xa[i])) < xdiff) {
      i_closest = i;
      xdiff = xdiff2;
    }
    C[i] = fa[i];
    D[i] = fa[i];
  }
  
  // Initial approx to f(x)
  Real fx = fa[i_closest--];

  for (int m=1; m<N; ++m) {
    for (int i=0; i<N-m; ++i) {
      Real C_dx = xa[i]-x;
      Real D_dx = xa[i+m] - x;
      Real C_D = C[i+1]-D[i];
      Real denom = C_dx - D_dx;
      if (denom == 0.0)
	cferror("polyinterp(fa,xa,x) : two values of xa has are equal!");
      denom = C_D/denom;
      C[i] = C_dx*denom;
      D[i] = D_dx*denom;
    }
    fx += (df=(2*(i_closest+1) < (N-m)? C[i_closest+1] : D[i_closest--]));
  }
  return fx;
}

Real secantSearch(Real a, Real b, array<Real>& fn, const array<Real>& xn, 
		  Real feps, int maxsteps) {
  Real fa = polyInterp(fn, xn, a);
  Real fb = polyInterp(fn, xn, b);
  if (fa*fb > 0) 
    cferror("secantSearch(a,b,fn,xn) : a and b don't bracket a zero");

  Real c = a - fa * (b-a)/(fb-fa);
  Real fc;

  for (int n=0; n<maxsteps; ++n) {
    c = a - fa * (b-a)/(fb-fa);
    fc = polyInterp(fn,xn,c);
    if (abs(fc) < feps) 
      break;
    if (fc*fa > 0) {
      a = c;
      fa = fc;
    }
    else {
      b = c;
      fb = fc;
    }
  }
  return c;
}


Real bisectSearch(Real a, Real b, array<Real>& fn, const array<Real>& xn, 
		  Real feps, int maxsteps) {
  Real c = 0.5*(a+b)/2; // GCC-3.3.5 complains if not explicitly init'ed
  Real fc;
  Real fa = polyInterp(fn, xn, a);
  Real fb = polyInterp(fn, xn, b);
  if (fa*fb > 0) 
    cferror("bisectSearch(a,b,fn,xn) : a and b don't bracket a zero");

  for (int n=0; n<maxsteps; ++n) {
    c = 0.5*(a+b);
    fc = polyInterp(fn,xn,c);
    if (abs(fc) < feps) 
      break;
    if (fc*fa > 0) {
      a = c;
      fa = fc;
    }
    else {
      b = c;
      fb = fc;
    }
  }
  return c;
}


FieldSeries::FieldSeries() 
  :
  t_(0),
  f_(0)
{}

FieldSeries::FieldSeries(int N) 
  :
  t_(N),
  f_(N)
{}

/*************************************************************
Still working on first draft of these functions

void FieldSeries::push(const FlowField& f, Real t) {
  for (int n=N-1; n>0; --n) {
    if (f_[n].congruent(f_[n-1]))
      swap(f_[n], f_[n-1]);
    else
      f_[n] = f_[n-1];
    t_[n] = t_[n-1];
  }
  if (N>0) {
    f_[0] = f;
    t_[0] = t;
  }
}

  
void FieldSeries::interpolate(Real t, FlowField& u) {
  if (!(this->full())) 
    cferror("FieldSeries::interpolate(Real t, FlowField& f) : FieldSeries is not completely initialized. Take more time steps before interpolating");

  if (u_[0].xzstate() == Spectral) {
    int Mx = u_[0].Mx();
    int Mz = u_[0].Mz();
    int Ny = u_[0].Ny();
    int Nd = u_[0].Nd();
    int N = u_.N();
    array<Real> f(N);
    for (int i=0; i<Nd; ++i) 
      for (int ny=0; ny<Ny; ++ny) 
	for (int mx=0; mx<Mx; ++mx) 
	  for (int mz=0; mz<Mz; ++mz) {
	    for (int n=0; n<N; ++n)
	      f[n] = Re(u_[n].cmplx(mx,ny,mz,i));
	    Real a = polyIntepr(f,t_,t);
	    
	    for (int n=0; n<N; ++n)
	      f[n] = Im(u_[n].cmplx(mx,ny,mz,i));
	    Real b = polyIntepr(f,t_,t);

	    u.cmplx(mx,ny,mz,i) = Complex(a,b);
	  }
  }
  else {
    int Nx = u_[0].Nx();
    int Ny = u_[0].Ny();
    int Nz = u_[0].Nz();
    int Nd = u_[0].Nd();
    int N = u_.N();
    array<Real> f(N);
    for (int i=0; i<Nd; ++i) 
      for (int ny=0; ny<Ny; ++ny) 
	for (int nx=0; nx<Nx; ++nx) 
	  for (int nz=0; nz<Nz; ++nz) {
	    for (int n=0; n<N; ++n)
	      f[n] = u_[n](nx,ny,nz,i);
	    u(nx,ny,nz,i) = polyInterp(f,t_,);
	  }
		 

}
      
**********************************/

