/* basisfunc.cpp: a Complex-Vector-valued Spectral expansion class.
 * Channelflow-0.9
 *
 * Copyright (C) 2001-2005  John F. Gibson
 *
 * John F. Gibson
 * Center for Nonlinear Sciences
 * School of Physics
 * Georgia Institute of Technology
 * Atlanta, GA 30332-0430
 *
 * gibson@cns.physics.gatech.edu
 * jfg@member.fsf.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, U
 */

#include <fstream>
#include <iomanip>
#include "basisfunc.h"

BasisFunc::BasisFunc()
  :
  Nd_(0),
  Ny_(0),
  kx_(0),
  kz_(0),
  Lx_(0),
  Lz_(0),
  a_(0),
  b_(0),
  state_(Spectral),
  lambda_(0),
  u_(0)
{}

BasisFunc::BasisFunc(int Ny, const BasisFunc& f)
  :
  Nd_(f.Nd_),
  Ny_(Ny),
  kx_(f.kx_),
  kz_(f.kz_),
  Lx_(f.Lx_),
  Lz_(f.Lz_),
  a_(f.a_),
  b_(f.b_),
  state_(f.state_),
  lambda_(f.lambda_),
  u_(f.Nd())
{}

BasisFunc::BasisFunc(int Nd, int Ny, int kx, int kz, Real Lx, Real Lz, 
		     Real a, Real b, fieldstate s)
  :
  Nd_(Nd),
  Ny_(Ny),
  kx_(kx),
  kz_(kz),
  Lx_(Lx),
  Lz_(Lz),
  a_(a),
  b_(b),
  state_(s),
  lambda_(0.0),
  u_(Nd)
{
  for (int n=0; n<Nd_; ++n) 
    u_[n] = ComplexChebyCoeff(Ny_,a_,b_,state_);
}

BasisFunc::BasisFunc(int Ny, int kx, int kz, Real Lx, Real Lz, 
		     Real a, Real b, fieldstate s)
  :
  Nd_(3),
  Ny_(Ny),
  kx_(kx),
  kz_(kz),
  Lx_(Lx),
  Lz_(Lz),
  a_(a),
  b_(b),
  state_(s),
  lambda_(0.0),
  u_(3)
{
  for (int n=0; n<Nd_; ++n) 
    u_[n] = ComplexChebyCoeff(Ny_,a_,b_,state_);
}

BasisFunc::BasisFunc(const string& filebase) 
:
  Nd_(0),
  Ny_(0),
  kx_(0),
  kz_(0),
  Lx_(0),
  Lz_(0),
  a_(0),
  b_(0),
  state_(Spectral),
  lambda_(0),
  u_(0)
{
  string filename = filebase + string(".bf");
  ifstream is(filename.c_str());
  if (!is.good()) {
    cerr << "BasisFunc::BasisFunc(filebase) : can't open file " << filename << '\n';
    abort();
  }
  char c;
  is >> c;
  if (c != '%') {
    cerr << "BasisFunc::BasisFunc(filebase): bad header in file " << filename << endl;
    assert(false);
  }
  is >> Nd_
     >> Ny_
     >> kx_
     >> kz_
     >> Lx_
     >> Lz_
     >> a_
     >> b_
     >> state_ 
     >> lambda_;
  u_.resize(Nd_);
  for (int n=0; n<Nd_; ++n) {
    u_[n].resize(Ny_);
    u_[n].setBounds(a_,b_);
    u_[n].setState(state_);
  }

  Real r;
  Real i;
  for (int ny=0; ny<Ny_; ++ny) 
    for (int n=0; n<Nd_; ++n) {
      is >> r >> i; 
      u_[n].set(ny,Complex(r,i));
    }
}


/***************
BasisFunc::BasisFunc(istream& is)
  :
  Ny_(0),
  kx_(0),
  kz_(0),
  Lx_(0),
  Lz_(0),
  a_(0),
  b_(0)
  state_(Spectral),
  lambda_(0)
  u_(),
  v_(),
  w_()
{
  binaryLoad(is);
}
****************/


BasisFunc::BasisFunc(const ComplexChebyCoeff& u,
		     const ComplexChebyCoeff& v,
		     const ComplexChebyCoeff& w,
		     Real lam, int kx, int kz, Real Lx, Real Lz) 
:
  Nd_(3),
  Ny_(u.numModes()),
  kx_(kx),
  kz_(kz),
  Lx_(Lx),
  Lz_(Lz),
  a_(u.a()),
  b_(u.b()),
  state_(u.state()),
  lambda_(lam),
  u_(3)
{
  u_[0] = u;
  u_[1] = v;
  u_[2] = w;
  for (int n=1; n<Nd_; ++n) {
    assert(u_[0].congruent(u_[n]));
    ;  // this statement needed for optimized compilation, #define assert(x) ;
  }
}

void BasisFunc::save(const string& filebase, fieldstate savestate) const {
  fieldstate origstate = state_;
  ((BasisFunc&) *this).makeState(savestate);

  string filename(filebase);
  filename += string(".bf");
  ofstream os(filename.c_str());
  os << scientific << setprecision(REAL_DIGITS);
  char sp=' ';
  os << '%'
     <<sp<< Nd_
     <<sp<< Ny_
     <<sp<< kx_
     <<sp<< kz_
     <<sp<< Lx_
     <<sp<< Lz_
     <<sp<< a_
     <<sp<< b_
     <<sp<< state_ 
     <<sp<< lambda_
     << '\n';

  for (int ny=0; ny<Ny_; ++ny) {
    for (int n=0; n<Nd_; ++n) 
      os << Re(u_[n][ny]) <<sp<< Im(u_[n][ny]) <<sp;
    os << '\n';
  }
  os.close();

  ((BasisFunc&) *this).makeState(origstate);
}

void BasisFunc::binaryDump(ostream& os) const {
  write(os, Nd_);
  write(os, Ny_);
  write(os, kx_);
  write(os, kz_);
  write(os, Lx_);
  write(os, Lz_);
  write(os, a_);
  write(os, b_);
  write(os, state_);
  write(os, lambda_);
  for (int n=0; n<Nd_; ++n)
    u_[n].binaryDump(os);
}

void BasisFunc::binaryLoad(istream& is) {
  if (!is.good()) {
    cerr << "BasisFunc::binaryLoad(istream) : input error" << endl;
    abort();
  }
  read(is, Ny_);
  read(is, kx_);
  read(is, kz_);
  read(is, Lx_);
  read(is, Lz_);
  read(is, a_);
  read(is, b_);
  read(is, state_);
  read(is, lambda_);
  for (int n=0; n<Nd_; ++n)
    u_[n].binaryLoad(is);
}
  
void BasisFunc::randomize(bool adiri, bool bdiri, Real decay) {
  assert(Nd_ == 3);

  setToZero();
  setState(Spectral);

  ComplexChebyCoeff u(Ny_, a_, b_, Spectral);
  ComplexChebyCoeff v(Ny_, a_, b_, Spectral);
  ComplexChebyCoeff w(Ny_, a_, b_, Spectral);
  ComplexChebyCoeff vy(Ny_, a_, b_, Spectral);

  
  // Make a u,v part
  if (kx_ == 0) {
    u.randomize(decay);
    ubcFix(u, adiri, bdiri);
  }
  else {
    v.randomize(decay);
    vbcFix(v, adiri, bdiri);
    diff(v, u);
    u *= I*Lx_/(2*pi*kx_);
  }
  u_[0] += u;
  u_[1] += v;

  u.setToZero();
  v.setToZero();
  w.setToZero();

  // Make a v,w, part
  if (kz_ == 0) {
    w.randomize(decay);
    ubcFix(w, adiri, bdiri);
  }
  else {
    v.randomize(decay);
    vbcFix(v, adiri, bdiri);
    diff(v, w);
    w *= I*Lz_/(2*pi*kz_);
  }
  u_[2] += w;
  u_[1] += v;
  
  if (kx_ == 0 && kz_ == 0) {
    u_[0].im.setToZero();
    u_[1].re.setToZero(); // should already be zero
    u_[2].im.setToZero();
  }
}

void BasisFunc::interpolate(const BasisFunc& phi) {
  assert(phi.state_ == Spectral);
  assert(a_ >= phi.a_ && b_ <= phi.b_);
  assert(Nd_ == phi.Nd_);
  for (int n=0; n<Nd_; ++n)
    u_[n].interpolate(phi.u_[n]);
  state_ = Physical;
}

void BasisFunc::reflect(const BasisFunc& phi) {
  assert((a_+b_)/2 == phi.a() && b_ <= phi.b() && b_ > phi.a());
  assert(phi.state_ == Spectral);
  for (int n=0; n<Nd_; n += 2)
    u_[n].reflect(phi.u_[n], Odd);
  for (int n=1; n<Nd_; n += 2)
    u_[n].reflect(phi.u_[n], Even);
  state_ = Physical;
}
  
void BasisFunc::resize(int N) {
  if (N != Ny_) {
    Ny_ = N;
    for (int n=0; n<Nd_; ++n)
      u_[n].resize(Ny_);
  }
}

void BasisFunc::setBounds(Real Lx, Real Lz, Real a, Real b) {
  Lx_ = Lx;
  Lz_ = Lz;
  for (int n=0; n<Nd_; ++n)
    u_[n].setBounds(a,b);
}
void BasisFunc::setkxkz(int kx, int kz) {
  kx_=kx;
  kz_=kz;
}
void BasisFunc::setState(fieldstate s) {
  assert(s == Spectral || s == Physical);
  for (int n=0; n<Nd_; ++n)
    u_[n].setState(s);
  state_ = s;
}

void BasisFunc::setRmsval(Real lambda) {
  lambda_ = lambda;
}

void BasisFunc::setToZero() {
  for (int n=0; n<Nd_; ++n)
    u_[n].setToZero();
}

void BasisFunc::fill(const BasisFunc& f) {
  for (int n=0; n<Nd_; ++n)
    u_[n].fill(f.u_[n]);
}
void BasisFunc::conjugate() {
  for (int n=0; n<Nd_; ++n)
    u_[n].conjugate();
  kx_ *= -1;
  kz_ *= -1; // Note that no neg vals of kz appear in FlowFields.
}

Real BasisFunc::bcNorm(bool a_dirichlet, bool b_dirichlet) const {
  Real err=0.0;
  if (a_dirichlet)
    for (int n=0; n<Nd_; ++n)
      err += abs2(u_[n].eval_a());
  if (b_dirichlet) 
    for (int n=0; n<Nd_; ++n)
      err += abs2(u_[n].eval_b());
  return sqrt(err);
}

bool BasisFunc::geometryCongruent(const BasisFunc& Phi) const {
  return (Nd_ == Phi.Nd_ && Ny_==Phi.Ny_ && Lx_==Phi.Lx_ && Lz_==Phi.Lz_ &&
	  a_==Phi.a_ && b_==Phi.b_);
}

bool BasisFunc::congruent(const BasisFunc& Phi) const {
  return (geometryCongruent(Phi) && kx_==Phi.kx_ && kz_==Phi.kz_
	  && state_ == Phi.state_);
}

void BasisFunc::chebyfft() {
  assert(state_ == Physical);
  ChebyTransform t(Ny_);
  for (int n=0; n<Nd_; ++n)
    u_[n].chebyfft(t);
  state_ = Spectral;
}

void BasisFunc::ichebyfft() {
  assert(state_ == Spectral);
  ChebyTransform t(Ny_);
  for (int n=0; n<Nd_; ++n)
    u_[n].ichebyfft(t);
  state_ = Physical;
}
void BasisFunc::makeSpectral()  {
  ChebyTransform t(Ny_);
  for (int n=0; n<Nd_; ++n)
    u_[n].makeSpectral(t);
  state_ = Spectral;
}
void BasisFunc::makePhysical() {
  ChebyTransform t(Ny_);
  for (int n=0; n<Nd_; ++n)
    u_[n].makePhysical(t);
  state_ = Physical;
}
void BasisFunc::makeState(fieldstate s) {
  ChebyTransform t(Ny_);
  for (int n=0; n<Nd_; ++n)
    u_[n].makeState(s,t);
  state_ = s;
}

void BasisFunc::chebyfft(const ChebyTransform& t) {
  assert(state_ == Physical);
  for (int n=0; n<Nd_; ++n)
    u_[n].chebyfft(t);
  state_ = Spectral;
}

void BasisFunc::ichebyfft(const ChebyTransform& t) {
  assert(state_ == Spectral);
  for (int n=0; n<Nd_; ++n)
    u_[n].ichebyfft(t);
  state_ = Physical;
}
void BasisFunc::makeSpectral(const ChebyTransform& t)  {
  for (int n=0; n<Nd_; ++n)
    u_[n].makeSpectral(t);
  state_ = Spectral;
}
void BasisFunc::makePhysical(const ChebyTransform& t) {
  for (int n=0; n<Nd_; ++n)
    u_[n].makePhysical(t);
  state_ = Physical;
}
void BasisFunc::makeState(fieldstate s, const ChebyTransform& t) {
  for (int n=0; n<Nd_; ++n)
    u_[n].makeState(s,t);
  state_ = s;
}


const ComplexChebyCoeff& BasisFunc::u() const {return u_[0];}
const ComplexChebyCoeff& BasisFunc::v() const {return u_[1];}
const ComplexChebyCoeff& BasisFunc::w() const {return u_[2];}
ComplexChebyCoeff& BasisFunc::u() {return u_[0];}
ComplexChebyCoeff& BasisFunc::v() {return u_[1];}
ComplexChebyCoeff& BasisFunc::w() {return u_[2];}

const ComplexChebyCoeff& BasisFunc::component(int i) const {
  assert(i>=0 && i<Nd_);
  return u_[i];
}
ComplexChebyCoeff& BasisFunc::component(int i) {
  assert(i>=0 && i<Nd_);
  return u_[i];
}
const ComplexChebyCoeff& BasisFunc::operator[](int i) const {
  assert(i>=0 && i<Nd_);
  return u_[i];
}
ComplexChebyCoeff& BasisFunc::operator[](int i) {
  assert(i>=0 && i<Nd_);
  return u_[i];
}
/**************
bool BasisFunc::operator==(const BasisFunc& Phi) const {
  return !(*this != Phi);
}
bool BasisFunc::operator!=(const BasisFunc& Phi) const {
  return (!congruent(Phi) || u_ != Phi.u_ || v_ != Phi.v_ || w_ != Phi.w_);
}
****************/

BasisFunc& BasisFunc::operator *= (const BasisFunc& phi) {
  for (int n=0; n<Nd_; ++n)
    u_[n] *= phi.u_[n];
  return *this;
}

BasisFunc& BasisFunc::operator += (const BasisFunc& phi) {
  for (int n=0; n<Nd_; ++n)
    u_[n] += phi.u_[n];
  return *this;
}
BasisFunc& BasisFunc::operator -= (const BasisFunc& phi) {
  for (int n=0; n<Nd_; ++n)
    u_[n] -= phi.u_[n];
  return *this;
}
BasisFunc& BasisFunc::operator *= (Real c) {
  for (int n=0; n<Nd_; ++n)
    u_[n] *= c;
  return *this;
}
BasisFunc& BasisFunc::operator *= (Complex c) {
  for (int n=0; n<Nd_; ++n)
    u_[n] *= c;
  return *this;
}

bool operator!=(const BasisFunc& f, const BasisFunc& g) {
  return (f == g) ? false : true;
}

bool operator==(const BasisFunc& f, const BasisFunc& g) {
  if (!f.congruent(g))
    return false;

  for (int n=0; n<f.Nd(); ++n)
    if (f[n] != g[n])
      return false;

  return true;
}

   
// ==================================================================
// L2 norms
Real L2Norm(const BasisFunc& phi, bool normalize) {
  Real rtn2 = 0.0;
  for (int n=0; n<phi.Nd(); ++n)
    rtn2 += L2Norm2(phi[n], normalize);
  if (!normalize)
    rtn2 *= phi.Lx()*phi.Lz();
  return sqrt(rtn2);
}

Real L2Norm2(const BasisFunc& phi, bool normalize) {
  Real rtn = 0.0;
  for (int n=0; n<phi.Nd(); ++n)
    rtn += L2Norm2(phi[n], normalize);
  if (!normalize)
    rtn *= phi.Lx()*phi.Lz();
  return rtn;
}

Real L2Dist2(const BasisFunc& f, const BasisFunc& g, bool normalize) {
  assert(f.geometryCongruent(g));
  Real rtn=0.0;
  if (f.kx()==g.kx() && f.kz()==g.kz())
    for (int n=0; n<f.Nd(); ++n)
      rtn += L2Dist2(f[n], g[n], normalize);
  if (!normalize)
    rtn *= f.Lx()*f.Lz();
  return rtn;

}

Real L2Dist(const BasisFunc& f, const BasisFunc& g, bool normalize) {
    return sqrt(L2Dist2(f,g,normalize));
}

Complex L2InnerProduct(const BasisFunc& f, const BasisFunc& g,
		       bool normalize) {
  assert(f.geometryCongruent(g));
  Complex rtn(0.0, 0.0);
  if (f.kx()==g.kx() && f.kz()==g.kz())
    for (int n=0; n<f.Nd(); ++n)
      rtn += L2InnerProduct(f[n], g[n], normalize);
  if (!normalize)
    rtn *= f.Lx() * f.Lz();
  return rtn;
}

// ==================================================================
// cheby norms
Real chebyNorm(const BasisFunc& phi, bool normalize) {
  Real rtn2 = 0.0;
  for (int n=0; n<phi.Nd(); ++n)
    rtn2 += chebyNorm2(phi[n], normalize);
  if (!normalize)
    rtn2 *= phi.Lx()*phi.Lz();
  return sqrt(rtn2);
}

Real chebyNorm2(const BasisFunc& phi, bool normalize) {
  Real rtn = 0.0;
  for (int n=0; n<phi.Nd(); ++n)
    rtn += chebyNorm2(phi[n], normalize);
  if (!normalize)
    rtn *= phi.Lx()*phi.Lz();
  return rtn;
}

Real chebyDist2(const BasisFunc& f, const BasisFunc& g, bool normalize) {
  assert(f.geometryCongruent(g));
  Real rtn=0.0;
  if (f.kx()==g.kx() && f.kz()==g.kz())
    for (int n=0; n<f.Nd(); ++n)
      rtn += chebyDist2(f[n], g[n], normalize);
  if (!normalize)
    rtn *= f.Lx()*f.Lz();
  return rtn;

}

Real chebyDist(const BasisFunc& f, const BasisFunc& g, bool normalize) {
    return sqrt(chebyDist2(f,g,normalize));
}

Complex chebyInnerProduct(const BasisFunc& f, const BasisFunc& g,
			  bool normalize) {
  assert(f.geometryCongruent(g));
  Complex rtn(0.0, 0.0);
  if (f.kx()==g.kx() && f.kz()==g.kz())
    for (int n=0; n<f.Nd(); ++n)
      rtn += chebyInnerProduct(f[n], g[n], normalize);
  if (!normalize)
    rtn *= f.Lx() * f.Lz();
  return rtn;
}

// ==================================================================
// switchable norms
Real norm(const BasisFunc& phi, NormType n, bool normalize) {
  return (n==L2NORM) ? L2Norm(phi,normalize) : chebyNorm(phi,normalize);
}

Real norm2(const BasisFunc& phi, NormType n, bool normalize) {
  return (n==L2NORM) ? L2Norm2(phi,normalize) : chebyNorm2(phi,normalize);
}

Real dist2(const BasisFunc& f, const BasisFunc& g, NormType n, bool nrmlz) {
  return (n==L2NORM) ? L2Dist2(f,g,nrmlz) : chebyDist2(f,g,nrmlz);
}

Real dist(const BasisFunc& f, const BasisFunc& g,  NormType n, bool nrmlz) {
  return (n==L2NORM) ? L2Dist(f,g,nrmlz) : chebyDist(f,g,nrmlz);
}

Complex innerProduct(const BasisFunc& f, const BasisFunc& g, NormType n,
		       bool nrmlz) {
  return (n==L2NORM) ? L2InnerProduct(f,g,nrmlz) : chebyInnerProduct(f,g,nrmlz);
}

// =================================================================
// boundary and divergence norms

Real divNorm2(const BasisFunc& f, bool normalize) {
  assert(f.state() == Spectral);
  ComplexChebyCoeff div = f.u();
  div *= (2*pi*f.kx()/f.Lx())*I;
  ComplexChebyCoeff tmp = f.w();
  tmp *= (2*pi*f.kz()/f.Lz())*I;
  div += tmp;
  diff(f.v(), tmp);
  div += tmp;
  return L2Norm2(div, normalize);
}

Real divNorm(const BasisFunc& f, bool normalize)  {
  return sqrt(divNorm2(f, normalize));
}

Real divDist2(const BasisFunc& f, const BasisFunc& g, bool normalize) {
  assert(f.state() == Spectral && g.state() == Spectral);
  assert(f.congruent(g));
  ComplexChebyCoeff div = f.u();
  div -= g.u();
  div *= (2*pi*f.kx()/f.Lx())*I;
  ComplexChebyCoeff tmp = f.w();
  tmp -= g.w();
  tmp *= (2*pi*f.kz()/f.Lz())*I;
  div += tmp;
  diff(f.v(), tmp);
  div += tmp;
  diff(g.v(), tmp);
  div -= tmp;
  return L2Norm2(div, normalize);
}

Real divDist(const BasisFunc& f, const BasisFunc& g, bool normalize)  {
  return sqrt(divDist2(f,g, normalize));
}

Real bcNorm2(const BasisFunc& f, bool normalize) {
  Real bc2 = 
    abs2(f.u().eval_a()) + abs2(f.u().eval_b()) +
    abs2(f.v().eval_a()) + abs2(f.v().eval_b()) +
    abs2(f.w().eval_a()) + abs2(f.w().eval_b());
  if (!normalize)
    bc2 *= f.Lx()*f.Lz();
  return bc2;
}

Real bcNorm(const BasisFunc& f, bool normalize)  {
  return sqrt(bcNorm2(f, normalize));
}

Real bcDist2(const BasisFunc& f, const BasisFunc& g, bool normalize) {
  Real bc2 = 
    abs2(f.u().eval_a() - g.u().eval_a()) + 
    abs2(f.u().eval_b() - g.u().eval_b()) + 
    abs2(f.v().eval_a() - g.v().eval_a()) + 
    abs2(f.v().eval_b() - g.v().eval_b()) + 
    abs2(f.w().eval_a() - g.w().eval_a()) + 
    abs2(f.w().eval_b() - g.w().eval_b());
  if (!normalize)
    bc2 *= f.Lx()*f.Lz();
  return bc2;
}

Real bcNorm(const BasisFunc& f, const BasisFunc& g, bool normalize)  {
  return sqrt(bcNorm2(f, normalize));
}

BasisFunc xdiff(const BasisFunc& phi) {
  BasisFunc phix(phi);
  phix *= (2*pi*phi.kx()/phi.Lx())*I;
  return phix;
}

  
BasisFunc ydiff(const BasisFunc& phi) {
  assert(phi.state() == Spectral);
  BasisFunc phiy(phi.Ny(), phi.kx(), phi.kz(), phi.Lx(), phi.Lz(), phi.a(), phi.b(), Spectral);
  diff(phi.u(), phiy.u());
  diff(phi.v(), phiy.v());
  diff(phi.w(), phiy.w());
  return phiy;
}

BasisFunc zdiff(const BasisFunc& phi) {
  BasisFunc phiz(phi);
  phiz *= (2*pi*phi.kz()/phi.Lz())*I;
  return phiz;
}

ComplexChebyCoeff divergence(const BasisFunc& phi) {
  assert(phi.state() == Spectral);
  ComplexChebyCoeff div(phi.Ny(), phi.a(), phi.b(), Spectral);
  ComplexChebyCoeff tmp(phi.Ny(), phi.a(), phi.b(), Spectral);
  tmp = phi[0];
  tmp *= (2*pi*phi.kx()/phi.Lx())*I;
  div += tmp;

  diff(phi[1], tmp);
  div += tmp;

  tmp = phi[2];
  tmp *= (2*pi*phi.kz()/phi.Lz())*I;
  div += tmp;
  
  return div;
}


BasisFunc laplacian(const BasisFunc& phi) {
  BasisFunc lapl(phi);
  Real c = -4*square(pi)*(square(phi.kx()/phi.Lx())+square(phi.kz()/phi.Lz()));
  lapl.u() *= c;
  lapl.v() *= c;
  lapl.w() *= c;

  ComplexChebyCoeff tmp;
  diff2(phi.u(), tmp);
  lapl.u() += tmp;
  diff2(phi.v(), tmp);
  lapl.v() += tmp;
  diff2(phi.w(), tmp);
  lapl.w() += tmp;
  return lapl;
}

BasisFunc gradient(const ComplexChebyCoeff& f, int kx,int kz, Real Lx,Real Lz){
  assert(f.state() == Spectral);
  BasisFunc grad(f.numModes(), kx, kz, Lx, Lz, f.a(), f.b(), f.state());
  grad.u() = f;
  grad.u() *= (2*pi*kx/Lx)*I;
  diff(f, grad.v());
  grad.w() = f;
  grad.w() *= (2*pi*kz/Lz)*I;
  return grad;
}

BasisFunc gradient(const BasisFunc& phi, int i) {
  assert(phi.state() == Spectral);

  BasisFunc grad(phi.Ny(), phi.kx(), phi.kz(), phi.Lx(), phi.Lz(), phi.a(), phi.b(), phi.state());
  const ComplexChebyCoeff& phi_i = phi[i];

  grad.u() = phi_i;
  grad.u() *= (2*pi*phi.kx()/phi.Lx())*I;

  diff(phi_i, grad.v());

  grad.w() = phi_i;
  grad.w() *= (2*pi*phi.kz()/phi.Lz())*I;
  return grad;
}


ComplexChebyCoeff dot(const BasisFunc& f_, const BasisFunc& g_) { 
  BasisFunc& f = (BasisFunc&) f_;
  BasisFunc& g = (BasisFunc&) g_;
  assert(f.geometryCongruent(g));
  int Ny= f.Ny();
  fieldstate fstate = f.state();
  fieldstate gstate = g.state();
  ChebyTransform t(Ny);
  f.makePhysical(t);
  g.makePhysical(t);

  ComplexChebyCoeff fg(Ny, f.a(), f.b(), Physical);
  for (int ny=0; ny<Ny; ++ny)
    fg.set(ny, f[0][ny]*g[0][ny] + f[1][ny]*g[1][ny] + f[2][ny]*g[2][ny]);

  
  f.makeState(fstate,t);
  g.makeState(gstate,t);
  fg.makeState(fstate, t);
  return fg;
}
	   

BasisFunc cross(const BasisFunc& f_, const BasisFunc& g_) {
  BasisFunc& f = (BasisFunc&) f_;
  BasisFunc& g = (BasisFunc&) g_;
  assert(f.geometryCongruent(g)); 
  int Ny= f.Ny();
  
  ChebyTransform t(Ny);
  fieldstate entrystate = f.state();
  f.makePhysical(t);
  g.makePhysical(t);


  BasisFunc rtn(Ny, f.kx()+g.kx(), f.kz()+g.kz(), f.Lx(), f.Lz(), f.a(), f.b(),
		Physical);
  for (int ny=0; ny<Ny; ++ny) {
    Complex fu = f[0][ny];
    Complex fv = f[1][ny];
    Complex fw = f[2][ny];
    Complex gu = g[0][ny];
    Complex gv = g[1][ny];
    Complex gw = g[2][ny];
    
    rtn[0].set(ny, fv*gw - fw*gv);
    rtn[1].set(ny, fw*gu - fu*gw);
    rtn[2].set(ny, fu*gv - fv*gu);
  }
    
  if (entrystate==Spectral) {
    f.makeSpectral(t);
    g.makeSpectral(t);
    rtn.makeSpectral(t);
  }
  return rtn;
}

BasisFunc curl(const BasisFunc& f_) {

  BasisFunc& f = (BasisFunc&) f_;
  fieldstate entrystate = f.state();
  int Ny= f.Ny();
  ChebyTransform t(Ny);
  f.makeSpectral(t);

  BasisFunc f_y = ydiff(f);
  Complex d_dx = (2*pi*f.kx()/f.Lx())*I;
  Complex d_dz = (2*pi*f.kz()/f.Lz())*I;

  BasisFunc rtn(f);
  
  rtn.setToZero();
  for (int ny=0; ny<Ny; ++ny) {
    Complex u  = f[0][ny];
    Complex v  = f[1][ny];
    Complex w  = f[2][ny];
    Complex u_y = f_y[0][ny];
    Complex w_y = f_y[2][ny];
    
    rtn[0].set(ny, w_y - d_dz*v);
    rtn[1].set(ny, d_dz*u - d_dx*w);
    rtn[2].set(ny, d_dx*v - u_y);
  }
    
  if (entrystate==Spectral) {
    f.makeSpectral(t);
    rtn.makeSpectral(t);
  }
  return rtn;
}

// rtn = phi_n dot grad phi_p
void dotgrad(const BasisFunc& phi_n, const BasisFunc& grad_phi_pu,
	     const BasisFunc& grad_phi_pv, const BasisFunc& grad_phi_pw,
	     ComplexChebyCoeff& tmp, BasisFunc& rtn) 
{
  assert(phi_n.state() == Physical);
  assert(grad_phi_pu.state() == Physical);
  assert(grad_phi_pv.state() == Physical);
  assert(grad_phi_pw.state() == Physical);
  assert(phi_n.geometryCongruent(grad_phi_pu));
  assert(phi_n.geometryCongruent(grad_phi_pv));
  assert(phi_n.geometryCongruent(grad_phi_pw));
  rtn.setToZero();
  rtn.setkxkz(phi_n.kx()+grad_phi_pu.kx(), phi_n.kz()+grad_phi_pu.kz());
  rtn.setBounds(phi_n.Lx(), phi_n.Lz(), phi_n.a(), phi_n.b());
  rtn.setState(Physical);

  tmp.setToZero();
  tmp.setBounds(phi_n.a(), phi_n.b());
  tmp.setState(Physical);

  // =====================================
  // Calculate rtn.u =   phi_n.u d/dx phi_p.u
  //                   + phi_n.v d/dy phi_p.u
  //                   + phi_n.w d/dz phi_p.u
  int i; // MSVC++ FOR-SCOPE BUG
  for (i=0; i<3; ++i) {
    tmp = phi_n[i];
    tmp *= grad_phi_pu[i];
    rtn.u() += tmp;
  }
  // =====================================
  // Calculate rtn.v =   phi_n.u d/dx phi_p.v
  //                   + phi_n.v d/dy phi_p.v
  //                   + phi_n.w d/dz phi_p.v
  for (i=0; i<3; ++i) {
    tmp = phi_n[i];
    tmp *= grad_phi_pv[i];
    rtn.v() += tmp;
  }
  // =====================================
  // Calculate rtn.w =   phi_n.u d/dx phi_p.w
  //                   + phi_n.v d/dy phi_p.w
  //                   + phi_n.w d/dz phi_p.w
  for (i=0; i<3; ++i) {
    tmp = phi_n[i];
    tmp *= grad_phi_pw[i];
    rtn.w() += tmp;
  }
  return;
}

BasisFunc dotgrad(const BasisFunc& phi_n_arg, const BasisFunc& phi_p_arg) {
  BasisFunc& phi_n = (BasisFunc&) phi_n_arg;
  BasisFunc& phi_p = (BasisFunc&) phi_p_arg;
  
  fieldstate original_n_state = phi_n.state();
  fieldstate original_p_state = phi_p.state();

  int Ny=phi_n.Ny();
  ChebyTransform trans(Ny);
  phi_p.makeSpectral(trans);   // Now phi_p is spectral

  BasisFunc rtn(Ny, phi_n.kx()+phi_p.kx(), phi_n.kz()+phi_p.kz(), 
		phi_n.Lx(), phi_n.Lz(), phi_n.a(), phi_n.b(), Spectral);

  BasisFunc grad_phi_pu = gradient(phi_p, 0);
  BasisFunc grad_phi_pv = gradient(phi_p, 1);
  BasisFunc grad_phi_pw = gradient(phi_p, 2);
  ComplexChebyCoeff tmp(Ny, phi_n.a(), phi_n.b(), Physical);
  
  phi_n.makePhysical(trans);  // Now phi_n is physical
  grad_phi_pu.makePhysical(trans);
  grad_phi_pv.makePhysical(trans);
  grad_phi_pw.makePhysical(trans);

  dotgrad(phi_n, grad_phi_pu, grad_phi_pv, grad_phi_pw, tmp, rtn);
  
  // phi_n is physical, 
  if (original_n_state == Spectral) {
    phi_n.makeSpectral(trans);
    rtn.makeSpectral(trans);
  }
  // phi_p is spectral.
  if (original_p_state == Physical)  
    phi_p.makePhysical(trans);
  
  return rtn;
}

void ubcFix(ChebyCoeff& u, bool a_dirichlet, bool b_dirichlet) {
  Real ua = u.a();
  Real ub = u.b();
  u.setBounds(-1,1);
  
  if (a_dirichlet && b_dirichlet) {
    u[0] -= 0.5*(u.eval_b() + u.eval_a());
    u[1] -= 0.5*(u.eval_b() - u.eval_a());
  }
  else if (a_dirichlet) 
    u[0] -= u.eval_a();
  else if (a_dirichlet) 
    u[0] -= u.eval_b();

  u.setBounds(ua, ub);
}

void ubcFix(ComplexChebyCoeff& u, bool a_dirichlet, bool b_dirichlet) {
  Real ua = u.a();
  Real ub = u.b();
  u.setBounds(-1,1);
  
  if (a_dirichlet && b_dirichlet) {
    u.sub(0, 0.5*(u.eval_b() + u.eval_a()));
    u.sub(1, 0.5*(u.eval_b() - u.eval_a()));
  }
  else if (a_dirichlet) 
    u.sub(0, u.eval_a());
  else if (a_dirichlet) 
    u.sub(0, u.eval_b());

  u.setBounds(ua, ub);
}

void vbcFix(ChebyCoeff& v, bool a_dirichlet, bool b_dirichlet) {

  Real va = v.a();
  Real vb = v.b();
  v.setBounds(-1,1);
  
  if (a_dirichlet && b_dirichlet) {
    // Adjust v to match v(+-1)=v'(+-1)=0 BCs
    ChebyCoeff vy = diff(v);
    Real a = v.eval_a();
    Real b = v.eval_b(); 
    Real c = vy.eval_a();
    Real d = vy.eval_b(); 

    v[0] -= 0.5*(a + b) + 0.125*(c - d);
    v[1] -= 0.5625*(b - a) - 0.0625*(c + d);
    v[2] -= 0.125*(d - c);
    v[3] -= 0.0625*(a - b + c + d);
  }
  else if (a_dirichlet) {
    ChebyCoeff vy = diff(v);
    v[1] -= vy.eval_a();
    v[0] -= v.eval_a();
  }
  else if (b_dirichlet) {
    ChebyCoeff vy = diff(v);
    v[1] -= vy.eval_b();
    v[0] -= v.eval_b();
  }
  v.setBounds(va,vb);
}
  
void vbcFix(ComplexChebyCoeff& v, bool a_dirichlet, bool b_dirichlet) {

  Real va = v.a();
  Real vb = v.b();
  v.setBounds(-1,1);
  
  if (a_dirichlet && b_dirichlet) {
    // Adjust v to match v(+-1)=v'(+-1)=0 BCs
    ComplexChebyCoeff vy = diff(v);
    Complex a = v.eval_a();
    Complex b = v.eval_b(); 
    Complex c = vy.eval_a();
    Complex d = vy.eval_b(); 

    v.sub(0, 0.5*(a + b) + 0.125*(c - d));
    v.sub(1, 0.5625*(b - a) - 0.0625*(c + d));
    v.sub(2, 0.125*(d - c));
    v.sub(3, 0.0625*(a - b + c + d));
  }
  else if (a_dirichlet) {
    ComplexChebyCoeff vy = diff(v);
    v.sub(1, vy.eval_a());
    v.sub(0,  v.eval_a());
  }
  else if (b_dirichlet) {
    ComplexChebyCoeff vy = diff(v);
    v.sub(1, vy.eval_b());
    v.sub(0,  v.eval_b());
  }
  v.setBounds(va,vb);
}
  

