/* diffops.cpp: Differential operators for FlowFields
 * Channelflow-0.9
 *
 * Copyright (C) 2001-2005  John F. Gibson
 *
 * John F. Gibson
 * Center for Nonlinear Sciences
 * School of Physics
 * Georgia Institute of Technology
 * Atlanta, GA 30332-0430
 *
 * gibson@cns.physics.gatech.edu
 * jfg@member.fsf.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, U
 */

#include <fstream>
#include <iomanip>
#include "channelflow/diffops.h"

using namespace std;

void xdiff(const FlowField& f_, FlowField& dfdx, int n) {
  if (n==0)
    return;
  
  FlowField& f = (FlowField&) f_;
  fieldstate sxz = f.xzstate();
  fieldstate sy  = f.ystate();
  f.makeSpectral_xz();

  Real Lx = f.Lx();

  if (!f.congruent(dfdx))
    dfdx.resize(f.Nx(), f.Ny(), f.Nz(), f.Nd(), f.Lx(), f.Lz(), f.a(), f.b());
  else
    dfdx.setToZero();
  dfdx.setState(Spectral, sy);

  Complex rot(0.0, 0.0);
  switch(n % 4) {
  case 0: rot = Complex( 1.0, 0.0); break;
  case 1: rot = Complex( 0.0, 1.0); break;
  case 2: rot = Complex(-1.0, 0.0); break;
  case 3: rot = Complex( 0.0,-1.0); break;
  default: cferror("xdiff(f, dfdx, n) : impossible: n % 4 > 4 !!"); break;
  }

  int Nd = f.Nd();
  int Mx = f.Mx();
  int My = f.My();
  int Mz = f.Mz();
  int kxmax = f.kxmax();

  for (int i=0; i<Nd; ++i)
    for (int my=0; my<My; ++my)
      for (int mx=0; mx<Mx; ++mx) {
	int kx = f.kx(mx);
	Complex cx = rot*(pow(2*pi*kx/Lx, n)*zero_last_mode(kx,kxmax,n));
	for (int mz=0; mz<Mz; ++mz) 
	  dfdx.cmplx(mx,my,mz,i) = cx*f.cmplx(mx,my,mz,i);
      }

  f.makeState(sxz,sy);
  dfdx.makeSpectral();
  return;
}
  
void ydiff(const FlowField& f_, FlowField& dfdy, int n) {
  if (n==0)
    return;

  FlowField& f = (FlowField&) f_;
  fieldstate sxz = f.xzstate();
  fieldstate sy  = f.ystate();
  f.makeSpectral_y();

  if (!f.congruent(dfdy))
    dfdy.resize(f.Nx(), f.Ny(), f.Nz(), f.Nd(), f.Lx(), f.Lz(), f.a(), f.b());
  else
    dfdy.setToZero();
  dfdy.setState(sxz, Spectral);

  int Nd = f.Nd();
  if (sxz == Spectral) {
    int Mx = f.Mx();
    int My = f.My();
    int Mz = f.Mz();

    ComplexChebyCoeff  g(f.Ny(), f.a(), f.b(), Spectral);
    ComplexChebyCoeff gy(f.Ny(), f.a(), f.b(), Spectral);

    for (int i=0; i<Nd; ++i)
      for (int mx=0; mx<Mx; ++mx)
	for (int mz=0; mz<Mz; ++mz) {
	  for (int my=0; my<My; ++my)
	    g.set(my, f.cmplx(mx,my,mz,i));
	  
	  // differentiate n times and leave derivative in *g
	  for (int k=0; k<n; ++k) {
	    diff(g,gy);
	    swap(g,gy);
	  }
	  for (int my=0; my<My; ++my)
	    dfdy.cmplx(mx,my,mz,i) = g[my];
	}
  }
  else {
    int Nx = f.Nx();
    int Ny = f.Ny();
    int Nz = f.Nz();
    ChebyCoeff g(Ny,  f.a(), f.b(), Spectral);
    ChebyCoeff gy(Ny, f.a(), f.b(), Spectral);
  
    for (int i=0; i<Nd; ++i)
      for (int nx=0; nx<Nx; ++nx)
	for (int nz=0; nz<Nz; ++nz) {
	  for (int ny=0; ny<Ny; ++ny)
	    g[ny] = f(nx,ny,nz,i);
	  
	  // differentiate n times and leave derivative in g
	  for (int k=0; k<n; ++k) {
	    diff(g,gy);
	    swap(g,gy);
	  }
	  for (int ny=0; ny<Ny; ++ny)
	    dfdy(nx,ny,nz,i) = g[ny];
	}
    }
  
  f.makeState(sxz, sy);
  dfdy.makeSpectral();
}
  
  
void zdiff(const FlowField& f_, FlowField& dfdz, int n) {
  if (n==0)
    return;

  FlowField& f = (FlowField&) f_;
  fieldstate sxz = f.xzstate();
  fieldstate sy  = f.ystate();
  f.makeSpectral_xz();

  // compute gradf(nx,ny,nz,i) = df(nx,ny,nz,0)/dx_i for scalar-valued f
  if (!f.congruent(dfdz))
    dfdz.resize(f.Nx(), f.Ny(), f.Nz(), f.Nd(), f.Lx(), f.Lz(), f.a(), f.b());
  else
    dfdz.setToZero();
  dfdz.setState(Spectral, sy);

  Complex rot(0.0, 0.0);
  switch(n % 4) {
  case 0: rot = Complex( 1.0, 0.0); break;
  case 1: rot = Complex( 0.0, 1.0); break;
  case 2: rot = Complex(-1.0, 0.0); break;
  case 3: rot = Complex( 0.0,-1.0); break;
  default: cferror("zdiff(f, dfdz, n) : impossible: n % 4 > 4 !!"); break;
  }

  int Nd = f.Nd();
  int Mx = f.Mx();
  int My = f.My();
  int Mz = f.Mz();
  int kzmax = f.kzmax();
  Real Lz = f.Lz();

  Real az = 2*pi/Lz;
  for (int i=0; i<Nd; ++i)
    for (int my=0; my<My; ++my)
      for (int mx=0; mx<Mx; ++mx) 
	for (int mz=0; mz<Mz; ++mz) {
	  int kz = f.kz(mz);
	  Complex cz = rot*(pow(az*kz, n)*zero_last_mode(kz,kzmax,n));
	  dfdz.cmplx(mx,my,mz,i) = cz*f.cmplx(mx,my,mz,i);
	}

  f.makeState(sxz,sy);
  dfdz.makeSpectral();
  return;
}
  
void diff(const FlowField& f_, FlowField& df, int nx, int ny, int nz) {
  FlowField& f = (FlowField&) f_;
  fieldstate sxz = f.xzstate();
  fieldstate sy  = f.ystate();
  f.makeSpectral();

  Real Lx = f.Lx();
  Real Lz = f.Lz();

  if (!f.congruent(df))
    df.resize(f.Nx(), f.Ny(), f.Nz(), f.Nd(), f.Lx(), f.Lz(), f.a(), f.b());
  else
    df.setToZero();
  df.setState(Spectral, Spectral);

  Complex xrot(0.0, 0.0);
  switch(nx % 4) {
  case 0: xrot = Complex( 1.0, 0.0); break;
  case 1: xrot = Complex( 0.0, 1.0); break;
  case 2: xrot = Complex(-1.0, 0.0); break;
  case 3: xrot = Complex( 0.0,-1.0); break;
  default: cferror("diff(f, df,nx,ny,nz) : impossible: nx % 4 > 4 !!"); break;
  }
  Complex zrot(0.0, 0.0);
  switch(nz % 4) {
  case 0: zrot = Complex( 1.0, 0.0); break;
  case 1: zrot = Complex( 0.0, 1.0); break;
  case 2: zrot = Complex(-1.0, 0.0); break;
  case 3: zrot = Complex( 0.0,-1.0); break;
  default: cferror("diff(f, df,nx,ny,nz) : impossible: nz % 4 > 4 !!"); break;
  }

  int Nd = f.Nd();
  int Mx = f.Mx();
  int My = f.My();
  int Mz = f.Mz();
  int kxmax = f.kxmax();
  int kzmax = f.kzmax();

  // Do the x and z differentiation
  Real ax = 2*pi/Lx;
  Real az = 2*pi/Lz;
  for (int i=0; i<Nd; ++i)
    for (int my=0; my<My; ++my)
      for (int mx=0; mx<Mx; ++mx) {
	int kx = f.kx(mx);
	Complex cx = xrot*(pow(ax*kx, nx)*zero_last_mode(kx,kxmax,nx));
	for (int mz=0; mz<Mz; ++mz) {
	  int kz = f.kz(mz);
	  Complex cz = zrot*(pow(az*kz, nz)*zero_last_mode(kz,kzmax,nz));
	  df.cmplx(mx,my,mz,i) = cx*cz*f.cmplx(mx,my,mz,i);
	}
      }
  
  // Do the y differentiation
  ComplexChebyCoeff  g(f.Ny(), f.a(), f.b(), Spectral);
  ComplexChebyCoeff gy(f.Ny(), f.a(), f.b(), Spectral);

  for (int i=0; i<Nd; ++i)
    for (int mx=0; mx<Mx; ++mx)
      for (int mz=0; mz<Mz; ++mz) {
	for (int my=0; my<My; ++my)
	  g.set(my, df.cmplx(mx,my,mz,i));

	// differentiate n times and leave derivative in *g
	for (int k=0; k<ny; ++k) {
	  diff(g,gy);
	  swap(g,gy);
	}
	for (int my=0; My<My; ++my)
	  df.cmplx(mx,my,mz,i) = g[my];
      }
  
  f.makeState(sxz,sy);
  df.makeSpectral();
  return;
}

//  gradf(nx,ny,nz,i3j(i,j)) = df(nx,ny,nz,i)/dx_j for vector-valued f
//  gradf(nx,ny,nz,i) = df(nx,ny,nz,0)/dx_i for scalar-valued f

void grad(const FlowField& f_, FlowField& gradf) {

  FlowField& f = (FlowField&) f_;
  fieldstate sxz = f.xzstate();
  fieldstate sy  = f.ystate();
  f.makeSpectral();

  Real Lx = f.Lx();
  Real Lz = f.Lz();

  // compute gradf(nx,ny,nz,i) = df(nx,ny,nz,0)/dx_i for scalar-valued f
  if (f.Nd() == 1) {
    if (!f.geomCongruent(gradf) || gradf.vectorDim() != 3)
      gradf.resize(f.Nx(), f.Ny(), f.Nz(), 3, f.Lx(), f.Lz(), f.a(), f.b());
    else
      gradf.setToZero();
    gradf.setState(Spectral, Spectral);

    int Mx = f.Mx();
    int My = f.My();
    int Mz = f.Mz();
    int kxmax = f.kxmax();
    int kzmax = f.kzmax();

    // Assign df/dx to Df(i) and df/dz to Df(i)
    for (int my=0; my<My; ++my)
      for (int mx=0; mx<Mx; ++mx) {
	int kx = f.kx(mx);
	Complex cxkx(0.0, 2.0*pi*kx/Lx*zero_last_mode(kx,kxmax,1));
	for (int mz=0; mz<Mz; ++mz) {
	  int kz = f.kz(mz);
	  Complex czkz(0.0, 2.0*pi*kz/Lz*zero_last_mode(kz,kzmax,1));
	  Complex fval = f.cmplx(mx,my,mz,0);
	  gradf.cmplx(mx,my,mz,0) = fval*cxkx;
	  gradf.cmplx(mx,my,mz,2) = fval*czkz;
	}
      }

    // Assign df/dy to Df(i,1) 
    ComplexChebyCoeff g(f.Ny(),  f.a(), f.b(), Spectral);
    ComplexChebyCoeff gy(f.Ny(), f.a(), f.b(), Spectral);
    for (int mx=0; mx<Mx; ++mx)
      for (int mz=0; mz<Mz; ++mz) {
	for (int my=0; my<My; ++my)
	  g.set(my, f.cmplx(mx,my,mz,0));
	diff(g,gy);
	for (int my=0; my<My; ++my)
	  gradf.cmplx(mx,my,mz,1) = gy[my];
      }
  }
  else if (f.Nd() == 3) {
    if (!f.geomCongruent(gradf) || gradf.vectorDim() != 9)
      gradf.resize(f.Nx(), f.Ny(), f.Nz(), 9, f.Lx(), f.Lz(), f.a(), f.b());
    else
      gradf.setToZero();
    gradf.setState(Spectral, Spectral);

    int Mx = f.Mx();
    int My = f.My();
    int Mz = f.Mz();
    int Nd = 3;
    int kxmax = f.kxmax();
    int kzmax = f.kzmax();
  
    // Assign df_i/dx to Df(i,0) and df_i/dz to Df(i,2)
    for (int i=0; i<Nd; ++i)
      for (int my=0; my<My; ++my)
	for (int mx=0; mx<Mx; ++mx) {
	  int kx = f.kx(mx);
	  Complex cxkx(0.0, 2.0*pi*kx/Lx*zero_last_mode(kx,kxmax,1));
	  for (int mz=0; mz<Mz; ++mz) {
	    int kz = f.kz(mz);
	    Complex czkz(0.0, 2.0*pi*kz/Lz*zero_last_mode(kz,kzmax,1));
	    Complex fval = f.cmplx(mx,my,mz,i);
	    gradf.cmplx(mx,my,mz,i3j(i,0)) = fval*cxkx;
	    gradf.cmplx(mx,my,mz,i3j(i,2)) = fval*czkz;
	  }
	}

    // Assign df_i/dy to Df(i,1) 
    ComplexChebyCoeff g(f.Ny(),  f.a(), f.b(), Spectral);
    ComplexChebyCoeff gy(f.Ny(), f.a(), f.b(), Spectral);
    for (int i=0; i<Nd; ++i)
      for (int mx=0; mx<Mx; ++mx)
	for (int mz=0; mz<Mz; ++mz) {
	  for (int my=0; my<My; ++my)
	    g.set(my, f.cmplx(mx,my,mz,i));
	  diff(g,gy);
	  for (int my=0; my<My; ++my)
	    gradf.cmplx(mx,my,mz,i3j(i,1)) = gy[my];
	}
  }
  else 
    cferror("grad(FlowField f, FlowField grad_f) : f must be 1d or 3d");

  f.makeState(sxz, sy);
  return;
}

void lapl(const FlowField& f_, FlowField& laplf) {
  FlowField& f = (FlowField&) f_;
  fieldstate sxz = f.xzstate();
  fieldstate sy  = f.ystate();
  f.makeSpectral();

  if (!f.congruent(laplf))
    laplf.resize(f.Nx(), f.Ny(), f.Nz(), f.Nd(), f.Lx(), f.Lz(), f.a(), f.b());
  else
    laplf.setToZero();
  laplf.setState(Spectral, Spectral);

  // compute gradf(nx,ny,nz,i) = df(nx,ny,nz,0)/dx_i for scalar-valued f
  Real Lx = f.Lx();
  Real Lz = f.Lz();
  int Mx = f.Mx();
  int My = f.My();
  int Mz = f.Mz();
  int Nd = f.Nd();
    
  ComplexChebyCoeff g(f.Ny(),  f.a(), f.b(), Spectral);
  ComplexChebyCoeff gyy(f.Ny(), f.a(), f.b(), Spectral);
  
  Real az = -1.0*square(2.0*pi/Lz);
  for (int i=0; i<Nd; ++i) {
    // Assign lapl f_i = (d^2/dx^2 + d^2/dz^2) f_i 
    for (int my=0; my<My; ++my)
      for (int mx=0; mx<Mx; ++mx) {
	int kx = f.kx(mx);
	Complex cx = -1.0*square(2.0*pi*kx/Lx);
	for (int mz=0; mz<Mz; ++mz) 
	  laplf.cmplx(mx,my,mz,i) = 
	    (cx + az*square(f.kz(mz)))*f.cmplx(mx,my,mz,i);
      }

    // Add d^2/dy^2 f_i on to previous result 
    for (int mx=0; mx<Mx; ++mx)
      for (int mz=0; mz<Mz; ++mz) {
	for (int my=0; my<My; ++my)
	  g.set(my, f.cmplx(mx,my,mz,i));
	diff2(g,gyy);
	for (int my=0; my<My; ++my)
	  laplf.cmplx(mx,my,mz,i) += gyy[my];
      }
  }
  f.makeState(sxz,sy);
}

void norm2(const FlowField& f_, FlowField& norm2f) {
  FlowField& f = (FlowField&) f_;
  fieldstate sxz = f.xzstate();
  fieldstate sy  = f.ystate();
  f.makeState(Spectral, Spectral);

  if (!f.geomCongruent(norm2f) || norm2f.vectorDim() != 1)
    norm2f.resize(f.Nx(), f.Ny(), f.Nz(), 1, f.Lx(), f.Lz(), f.a(), f.b());
  else
    norm2f.setToZero();
  norm2f.setState(Physical, Physical);

  f.makePhysical();

  int Nx = f.Nx();
  int Ny = f.Ny();
  int Nz = f.Nz();
  int Nd = f.Nd();
  
  for (int i=0; i<Nd; ++i)
    for (int ny=0; ny<Ny; ++ny)
      for (int nx=0; nx<Nx; ++nx)
	for (int nz=0; nz<Nz; ++nz)
	  norm2f(nx,ny,nz,0) += square(f(nx,ny,nz,i));

  f.makeState(sxz,sy);
  norm2f.makeSpectral();
  return;
}
 
void norm(const FlowField& f_, FlowField& normf) {
  FlowField& f = (FlowField&) f_;
  fieldstate sxz = f.xzstate();
  fieldstate sy  = f.ystate();
  f.makePhysical();

  if (!f.geomCongruent(normf) || normf.vectorDim() != 1)
    normf.resize(f.Nx(), f.Ny(), f.Nz(), 1, f.Lx(), f.Lz(), f.a(), f.b());
  else
    normf.setToZero();
  normf.setState(Physical, Physical);

  // compute gradf(nx,ny,nz,i) = df(nx,ny,nz,0)/dx_i for scalar-valued f
  int Nx = f.Nx();
  int Ny = f.Ny();
  int Nz = f.Nz();
  int Nd = f.Nd();
  
  for (int ny=0; ny<Ny; ++ny)
    for (int nx=0; nx<Nx; ++nx)
      for (int nz=0; nz<Nz; ++nz) {
	Real nrm2 = 0.0;
	for (int i=0; i<Nd; ++i)
	  nrm2 += square(f(nx,ny,nz,i));
	normf(nx,ny,nz,0) = sqrt(nrm2);
      }

  normf.makeSpectral();
  f.makeState(sxz, sy);
  return;
}
 
void curl(const FlowField& f_, FlowField& curlf) {
  FlowField& f = (FlowField&) f_;
  assert(f.Nd() == 3);
  fieldstate sxz = f.xzstate();
  fieldstate sy  = f.ystate();
  f.makeSpectral();

  if (!f.geomCongruent(curlf) || curlf.vectorDim() != 3)
    curlf.resize(f.Nx(), f.Ny(), f.Nz(), 3, f.Lx(), f.Lz(), f.a(), f.b());
  else
    curlf.setToZero();
  curlf.setState(Spectral, Spectral);

  int Mx = f.Mx();
  int My = f.My();
  int Mz = f.Mz();
  int kxmax = f.kxmax();
  int kzmax = f.kzmax();
  Real Lx = f.Lx();
  Real Lz = f.Lz();
  
  // curl_x = w_y - v_z
  // curl_y = u_z - w_x
  // curl_z = v_x - u_y

  // Assign d/dx and d/dz terms to curl
  for (int my=0; my<My; ++my)
    for (int mx=0; mx<Mx; ++mx) {
      int kx = f.kx(mx);
      Complex cx(0.0, 2.0*pi*kx/Lx*zero_last_mode(kx,kxmax,1));
      for (int mz=0; mz<Mz; ++mz) {
	int kz = f.kz(mz);
	Complex cz(0.0, 2.0*pi*kz/Lz*zero_last_mode(kz,kzmax,1));
	Complex u = f.cmplx(mx,my,mz,0);
	Complex v = f.cmplx(mx,my,mz,1);
	Complex w = f.cmplx(mx,my,mz,2);
	curlf.cmplx(mx,my,mz,0) = -cz*v;
	curlf.cmplx(mx,my,mz,1) = cz*u - cx*w;
	curlf.cmplx(mx,my,mz,2) = cx*v;
      }
    }

  // Assign df_i/dy to Df(i,1) 
  ComplexChebyCoeff w(f.Ny(),  f.a(), f.b(), Spectral);
  ComplexChebyCoeff wy(f.Ny(), f.a(), f.b(), Spectral);
  ComplexChebyCoeff u(f.Ny(),  f.a(), f.b(), Spectral);
  ComplexChebyCoeff uy(f.Ny(), f.a(), f.b(), Spectral);
  for (int mx=0; mx<Mx; ++mx)
    for (int mz=0; mz<Mz; ++mz) {
      for (int my=0; my<My; ++my) {
	u.set(my, f.cmplx(mx,my,mz,0));
	w.set(my, f.cmplx(mx,my,mz,2));
      }
      diff(u,uy);
      diff(w,wy);
      for (int my=0; my<My; ++my) {
	curlf.cmplx(mx,my,mz,0) += wy[my];
	curlf.cmplx(mx,my,mz,2) -= uy[my];
      }
    }
  f.makeState(sxz, sy);
}

void outer(const FlowField& f_,  const FlowField& g_, FlowField& fg) {
  FlowField& f = (FlowField&) f_;
  FlowField& g = (FlowField&) g_;
  fieldstate fxz = f.xzstate();
  fieldstate fy  = f.ystate();
  fieldstate gxz = g.xzstate();
  fieldstate gy  = g.ystate();

  assert(f.geomCongruent(g));
  f.makePhysical();
  g.makePhysical();

  if (!f.geomCongruent(fg) || fg.Nd() != f.Nd()*g.Nd())
    fg.resize(f.Nx(), f.Ny(), f.Nz(), f.Nd()*g.Nd(), f.Lx(), f.Lz(), f.a(), f.b());
  else
    fg.setToZero();
  fg.setState(Physical, Physical);

  int Nx = f.Nx();
  int Ny = f.Ny();
  int Nz = f.Nz();
  int fd = f.Nd();
  int gd = g.Nd();
  
  for (int i=0; i<fd; ++i)
    for (int j=0; j<gd; ++j) {
      int ij = i*gd + j;   // generalized form of i3j == i*3 + j
      for (int ny=0; ny<Ny; ++ny)
	for (int nx=0; nx<Nx; ++nx)
	  for (int nz=0; nz<Nz; ++nz)
	    fg(nx,ny,nz,ij) = f(nx,ny,nz,i)*g(nx,ny,nz,j);
    }

  f.makeState(fxz, fy);
  g.makeState(gxz, gy);
  fg.makeSpectral();
  return;
}

void dot(const FlowField& f_,  const FlowField& g_, FlowField& fdotg) {
  FlowField& f = (FlowField&) f_;
  FlowField& g = (FlowField&) g_;
  fieldstate fxz = f.xzstate();
  fieldstate fy  = f.ystate();
  fieldstate gxz = g.xzstate();
  fieldstate gy  = g.ystate();

  assert(f.congruent(g));
  f.makePhysical();
  g.makePhysical();

  if (!f.geomCongruent(fdotg) || fdotg.Nd() != 1)
    fdotg.resize(f.Nx(), f.Ny(), f.Nz(), 1, f.Lx(), f.Lz(), f.a(),f.b());
  else
    fdotg.setToZero();
  fdotg.setState(Physical, Physical);

  int Nx = f.Nx();
  int Ny = f.Ny();
  int Nz = f.Nz();
  int Nd = f.Nd();
  
  for (int i=0; i<Nd; ++i)
    for (int ny=0; ny<Ny; ++ny)
      for (int nx=0; nx<Nx; ++nx)
	for (int nz=0; nz<Nz; ++nz)
	  fdotg(nx,ny,nz,0) += f(nx,ny,nz,i)*g(nx,ny,nz,i);

  f.makeState(fxz, fy);
  g.makeState(gxz, gy);
  fdotg.makeSpectral();
  return;
}

void div(const FlowField& f_,  FlowField& divf) {
  FlowField& f = (FlowField&) f_;
  fieldstate sxz = f.xzstate();
  fieldstate sy  = f.ystate();
  f.makeSpectral();

  int Mx = f.Mx();
  int My = f.My();
  int Mz = f.Mz();
  int kxmax = f.kxmax();
  int kzmax = f.kzmax();
  Real Lx = f.Lx();
  Real Lz = f.Lz();

  if (f.Nd() == 3) {
    if (!f.geomCongruent(divf) || divf.Nd() != 1)
      divf.resize(f.Nx(), f.Ny(), f.Nz(), 1, f.Lx(), f.Lz(), f.a(), f.b());
    else
      divf.setToZero();
    divf.setState(Spectral, Spectral);
    
    // Add df0/dx + df2/dz to divf
    for (int my=0; my<My; ++my)
      for (int mx=0; mx<Mx; ++mx) {
	int kx = f.kx(mx);
	Complex cxkx(0.0, 2.0*pi*kx/Lx*zero_last_mode(kx,kxmax,1));
	for (int mz=0; mz<Mz; ++mz) {
	  int kz = f.kz(mz);
	  Complex czkz(0.0, 2.0*pi*kz/Lz*zero_last_mode(kz,kzmax,1));
	  divf.cmplx(mx,my,mz,0) 
	    = f.cmplx(mx,my,mz,0)*cxkx + f.cmplx(mx,my,mz,2)*czkz;
	}
      }

    // Add df1/dy to to divf
    ComplexChebyCoeff g(f.Ny(),  f.a(), f.b(), Spectral);
    ComplexChebyCoeff gy(f.Ny(), f.a(), f.b(), Spectral);
    for (int mx=0; mx<Mx; ++mx)
      for (int mz=0; mz<Mz; ++mz) {
	for (int my=0; my<My; ++my)
	  g.set(my, f.cmplx(mx,my,mz,1));
	diff(g,gy);
	for (int my=0; my<My; ++my)
	  divf.cmplx(mx,my,mz,0) += gy[my];
      }
  }
  else if (f.Nd() == 9) {
    if (!f.geomCongruent(divf) || divf.Nd() != 3)
      divf.resize(f.Nx(), f.Ny(), f.Nz(), 3, f.Lx(), f.Lz(), f.a(), f.b());
    else
      divf.setToZero();
    divf.setState(Spectral, Spectral);
    
    for (int j=0; j<3; ++j) {

      // Add df0j/dx + df2j/dz to divfj
      for (int my=0; my<My; ++my)
	for (int mx=0; mx<Mx; ++mx) {
	  int kx = f.kx(mx);
	  Complex cxkx(0.0, 2.0*pi*f.kx(mx)/Lx*zero_last_mode(kx,kxmax,1));
	  for (int mz=0; mz<Mz; ++mz) {
	    int kz = f.kz(mz);
	    Complex czkz(0.0, 2.0*pi*kz/Lz*zero_last_mode(kz,kzmax,1));
	      divf.cmplx(mx,my,mz,j) +=
		f.cmplx(mx,my,mz,i3j(0,j))*cxkx + 
		f.cmplx(mx,my,mz,i3j(2,j))*czkz;
	  }
	}

      // Add df1j/dy to to divf
      ComplexChebyCoeff g(f.Ny(),  f.a(), f.b(), Spectral);
      ComplexChebyCoeff gy(f.Ny(), f.a(), f.b(), Spectral);
      for (int mx=0; mx<Mx; ++mx)
	for (int mz=0; mz<Mz; ++mz) {
	  for (int my=0; my<My; ++my)
	    g.set(my, f.cmplx(mx,my,mz,i3j(1,j)));
	  diff(g,gy);
	  for (int my=0; my<My; ++my)
	    divf.cmplx(mx,my,mz,j) += gy[my];
	}
    }
  }
  else 
    cferror("div(FlowField f, FlowField divf): f must be 3d or 9d");

  f.makeState(sxz, sy);
  divf.makeSpectral();
  return;
}

void cross(const FlowField& f_, const FlowField& g_, FlowField& fcg) {
  FlowField& f = (FlowField&) f_;
  FlowField& g = (FlowField&) g_;
  fieldstate fxz = f.xzstate();
  fieldstate fy  = f.ystate();
  fieldstate gxz = g.xzstate();
  fieldstate gy  = g.ystate();
  assert(g.congruent(f));
  assert(f.Nd() == 3 && g.Nd() == 3);

  f.makePhysical();
  g.makePhysical();
  
  if (!f.geomCongruent(fcg) || fcg.Nd() != 3)
    fcg.resize(f.Nx(), f.Ny(), f.Nz(), 3, f.Lx(), f.Lz(), f.a(), f.b());
  else
    fcg.setToZero();
  fcg.setState(Physical,Physical);
    
  int Nx = f.Nx();
  int Ny = f.Ny();
  int Nz = f.Nz();
  int Nd = f.Nd();
  for (int i=0; i<Nd; ++i)
    for (int ny=0; ny<Ny; ++ny)
      for (int nx=0; nx<Nx; ++nx)
	for (int nz=0; nz<Nz; ++nz)
	  fcg(nx,ny,nz,i) = 
	    f(nx,ny,nz,(i+1)%3)*g(nx,ny,nz,(i+2)%3) -
	    f(nx,ny,nz,(i+2)%3)*g(nx,ny,nz,(i+1)%3);

  f.makeState(fxz, fy);
  g.makeState(gxz, gy);
  fcg.makeSpectral();
}

void energy(const FlowField& u_, FlowField& e) {
  FlowField& u = (FlowField&) u_;
  fieldstate sxz = u.xzstate();
  fieldstate sy  = u.ystate();
  u.makePhysical();

  if (!u.geomCongruent(e) || e.Nd() != 1)
    e.resize(u.Nx(), u.Ny(), u.Nz(), 1, u.Lx(), u.Lz(), u.a(), u.b());
  else
    e.setToZero();
  e.setState(Physical,Physical);

  int Nx = u.Nx();
  int Ny = u.Ny();
  int Nz = u.Nz();
  int Nd = u.Nd();

  for (int i=0; i<Nd; ++i) 
    for (int ny=0; ny<Ny; ++ny)
      for (int nx=0; nx<Nx; ++nx)
	for (int nz=0; nz<Nz; ++nz)
	  e(nx,ny,nz,0) += 0.5*square(u(nx,ny,nz,i));
 
  e.makeSpectral();
  u.makeState(sxz, sy);
  return;
}


void energy(const FlowField& u_, const ChebyCoeff& U_, FlowField& e) {

  FlowField& u = (FlowField&) u_;
  ChebyCoeff& U = (ChebyCoeff&) U_;
  fieldstate uxzstate = u.xzstate();
  fieldstate uystate  = u.ystate();
  fieldstate Ustate   = U.state();
  assert(U.numModes() == u.Ny());

  if (!u.geomCongruent(e) || e.Nd() != 1)
    e.resize(u.Nx(), u.Ny(), u.Nz(), 1, u.Lx(), u.Lz(), u.a(), u.b());
  else
    e.setToZero();
  e.setState(Physical,Physical);

  u.makePhysical();
  U.makePhysical();

  e.setToZero();
  e.setState(Physical, Physical);

  int Nx = u.Nx();
  int Ny = u.Ny();
  int Nz = u.Nz();
  int Nd = u.Nd();

  for (int ny=0; ny<Ny; ++ny)
    for (int nx=0; nx<Nx; ++nx)
      for (int nz=0; nz<Nz; ++nz)
	e(nx,ny,nz,0) += 0.5*square(u(nx,ny,nz,0)+U(ny));
 
  for (int i=1; i<Nd; ++i) 
    for (int ny=0; ny<Ny; ++ny)
      for (int nx=0; nx<Nx; ++nx)
	for (int nz=0; nz<Nz; ++nz)
	  e(nx,ny,nz,0) += 0.5*square(u(nx,ny,nz,i));
 
  e.makeSpectral();
  u.makeState(uxzstate,uystate);
  U.makeState(Ustate);

  return;
}

FlowField xdiff(const FlowField& f, int n) {
  FlowField g; 
  xdiff(f,g,n); 
  return g;
}

FlowField ydiff(const FlowField& f, int n) {
  FlowField g; 
  ydiff(f,g,n); 
  return g;
}

FlowField zdiff(const FlowField& f, int n) {
  FlowField g; 
  zdiff(f,g,n); 
  return g;
}
  
FlowField diff(const FlowField& f, int nx, int ny, int nz) {
  FlowField g;
  diff(f,g,nx,ny,nz);
  return g;
}

FlowField grad(const FlowField& f)  {FlowField g;  grad(f,g); return g;}
FlowField lapl(const FlowField& f)  {FlowField g;  lapl(f,g); return g;}
FlowField curl(const FlowField& f)  {FlowField g;  curl(f,g); return g;}
FlowField norm(const FlowField& f)  {FlowField g;  norm(f,g); return g;}
FlowField norm2(const FlowField& f) {FlowField g; norm2(f,g); return g;}
FlowField div(const FlowField& f)   {FlowField g;   div(f,g); return g;}

FlowField outer(const FlowField& f, const FlowField& g) {
  FlowField fg; outer(f,g,fg); return fg;
}
FlowField cross(const FlowField& f, const FlowField& g) {
  FlowField fxg; cross(f,g,fxg); return fxg;
}
FlowField dot(const FlowField& f, const FlowField& g) {
  FlowField fdotg; dot(f,g,fdotg); return fdotg;
}
FlowField energy(const FlowField& u) {
  FlowField e; energy(u,e); return e;
}
FlowField energy(const FlowField& u, ChebyCoeff& U) {
  FlowField e; energy(u,U,e); return e;
}
    
void navierstokesNL(const FlowField& u,const ChebyCoeff& U, FlowField& f, 
		    FlowField& tmp, NonlinearMethod& method) {
  switch (method) {
  case Rotational: 
    rotationalNL(u,U,f,tmp); 
    break;
  case Convection: 
    convectionNL(u,U,f,tmp); 
    break;
  case SkewSymmetric: 
    skewsymmetricNL(u,U,f,tmp); 
    break;
  case Divergence: 
    divergenceNL(u,U,f,tmp); 
    break;
  case Alternating: 
    divergenceNL(u,U,f,tmp); 
    method = Alternating_;
    break;
  case Alternating_: 
    convectionNL(u,U,f,tmp); 
    method = Alternating;
    break;
  case Linearized: 
    linearizedNL(u,U,f); break;
  default:
    cferror("navierstokesNL(method, u,U,f,tmp,parity) : unknown method");
  }
}

// NOTE ON EFFICIENCY: I need to measure the cost of constructing 
// ChebyTransforms, Uy, and Uyy each time. My guess is it's inconsequential
// compared to the u and f transforms (Ny + Ny log Ny  against  
// Nx x Ny x Nz (log Nx + log Ny + log Nz))

// It would be lovely and simple to compute curl(u, vort); cross(u, vort, f); 
// but that's different than the (del cross u) cross u + U du/dx + v dUdy ex
// advertised. I'm sticking with the advertised form since it's been in place 
// and uq2p & up2q rely on it.

void rotationalNL(const FlowField& u_, const ChebyCoeff& U_, FlowField& f, 
		  FlowField& tmp) {

  FlowField& u = (FlowField&) u_;
  ChebyCoeff& U = (ChebyCoeff&) U_;
  FlowField& vort = tmp;

  assert(u.Nd() == 3);
  assert(U.numModes() == u.Ny());
  assert(U.a() == u.a() && U.b() == u.b());
  fieldstate uxzstate = u.xzstate();
  fieldstate uystate  = u.ystate();
  fieldstate Ustate   = U.state();

  if (!u.geomCongruent(f) || f.Nd() != 3)
    f.resize(u.Nx(), u.Ny(), u.Nz(), 3, u.Lx(), u.Lz(), u.a(), u.b());
  else
    f.setToZero();
  f.setState(Physical, Physical);

  if (!u.geomCongruent(vort) || vort.Nd() < 3)
    vort.resize(u.Nx(), u.Ny(), u.Nz(), 3, u.Lx(), u.Lz(), u.a(), u.b());

  ChebyTransform trans(U.N());
  u.makeSpectral();
  U.makeSpectral(trans);
  ChebyCoeff Uy = diff(U);
  
  curl(u, vort);

  u.makeState(Spectral, Physical);;
  vort.makeState(Spectral, Physical);
  f.setState(Spectral, Physical);
  U.makePhysical(trans);
  Uy.makePhysical(trans);

  Complex uc;
  Complex vc;
  Complex wc;

  int Mx = u.Mx();
  int My = u.My();
  int Mz = u.Mz();
  int kxmax = f.kxmax();

  for (int ny=0; ny<My; ++ny) { 
    Real rU  = U[ny];
    Real rUy = Uy[ny];
    for (int mx=0; mx<Mx; ++mx) {
      int kx = u.kx(mx);
      Complex d_dx(0.0, 2*pi*kx/u.Lx()*zero_last_mode(kx,kxmax,1));
      for (int mz=0; mz<Mz; ++mz) {
	uc = u.cmplx(mx,ny,mz, 0);
	vc = u.cmplx(mx,ny,mz, 1);
	wc = u.cmplx(mx,ny,mz, 2);

	f.cmplx(mx,ny,mz, 0) = rU*d_dx*uc + vc*rUy;
	f.cmplx(mx,ny,mz, 1) = rU*d_dx*vc;  
	f.cmplx(mx,ny,mz, 2) = rU*d_dx*wc;
      }
    }
  }

  vort.makePhysical();
  u.makePhysical();
  f.makePhysical();

  int Nx = u.Nx();
  int Ny = u.Ny();
  int Nz = u.Nz();

  Real omega_x;
  Real omega_y;
  Real omega_z;
  Real ru;
  Real rv;
  Real rw;

  // Loop should be efficient if we get 9 cache lines. 
  // Add in omega x u so that 
  // f = omega x U + Omega x u + omega x u
  for (int ny=0; ny<Ny; ++ny) { 
    //Real Uy = Ubasey[ny];
    for (int nx=0; nx<Nx; ++nx) {
      for (int nz=0; nz<Nz; ++nz) {
	ru = u(nx,ny,nz, 0);
	rv = u(nx,ny,nz, 1);
	rw = u(nx,ny,nz, 2);
	omega_x = vort(nx,ny,nz, 0);
	omega_y = vort(nx,ny,nz, 1);
	omega_z = vort(nx,ny,nz, 2);

	f(nx,ny,nz, 0) += omega_y*rw - omega_z*rv;
	f(nx,ny,nz, 1) += omega_z*ru - omega_x*rw;
	f(nx,ny,nz, 2) += omega_x*rv - omega_y*ru;
      }
    }
  }
  f.makeSpectral(); 
  
  // Possible extra transforms if entry state was Physical
  u.makeState(uxzstate, uystate);
  U.makeState(Ustate, trans);

  return;
}
  
void convectionNL(const FlowField& u_, const ChebyCoeff& U_, FlowField& f, FlowField& tmp) {

  FlowField& u = (FlowField&) u_;
  ChebyCoeff& U = (ChebyCoeff&) U_;
  FlowField& grad_u = tmp;
  
  assert(u.Nd() == 3);
  assert(U.numModes() == u.Ny());
  assert(U.a() == u.a() && U.b() == u.b());

  fieldstate uxzstate = u.xzstate();
  fieldstate uystate  = u.ystate();
  fieldstate Ustate   = U.state();

  if (!u.geomCongruent(f) || f.Nd() != 3)
    f.resize(u.Nx(), u.Ny(), u.Nz(), 3, u.Lx(), u.Lz(), u.a(), u.b());
  else
    f.setToZero();
  f.setState(Physical, Physical);
  
  if (!u.geomCongruent(grad_u) || grad_u.Nd() < 9)
    grad_u.resize(u.Nx(), u.Ny(), u.Nz(), 9, u.Lx(), u.Lz(), u.a(), u.b());

  U.makeSpectral();
  u.makeSpectral();
  u += U; 

  grad(u, grad_u);
  grad_u.makePhysical();
  u.makePhysical();

  int Nx = u.Nx();
  int Ny = u.Ny();
  int Nz = u.Nz();
  for (int i=0; i<3; ++i) 
    for (int j=0; j<3; ++j) {
      int ij = i3j(i,j);
      for (int ny=0; ny<Ny; ++ny) 
	for (int nx=0; nx<Nx; ++nx) 
	  for (int nz=0; nz<Nz; ++nz) 
	    f(nx,ny,nz,i) += u(nx,ny,nz,j)*grad_u(nx,ny,nz,ij);
    }

  f.makeSpectral();
  u.makeSpectral();
  u -= U;

  // Possible extra transforms if entry states were Physical
  u.makeState(uxzstate, uystate);
  U.makeState(Ustate);
}
  
void divergenceNL(const FlowField& u_, const ChebyCoeff& U_, FlowField& f, FlowField& tmp) {
  FlowField& u = (FlowField&) u_;
  ChebyCoeff& U = (ChebyCoeff&) U_;
  FlowField& uu = tmp;

  assert(u.Nd() == 3);
  assert(U.numModes() == u.Ny());
  assert(U.a() == u.a() && U.b() == u.b());

  fieldstate uxzstate = u.xzstate();
  fieldstate uystate  = u.ystate();
  fieldstate Ustate   = U.state();

  if (!u.geomCongruent(f) || f.Nd() != 3)
    f.resize(u.Nx(), u.Ny(), u.Nz(), 3, u.Lx(), u.Lz(), u.a(), u.b());

  if (!u.geomCongruent(uu) || uu.Nd() < 9)
    uu.resize(u.Nx(), u.Ny(), u.Nz(), 9, u.Lx(), u.Lz(), u.a(), u.b());

  // Possible extra transforms if entry states are Physical
  u.makeState(Spectral, uystate);
  U.makeState(uystate);
  u += U;

  u.makePhysical();
  outer(u,u, uu);

  uu.makeSpectral();
  div(uu, f);         // produces f spectral
  
  u.makeState(Spectral, uystate);
  u -= U;

  // Possible extra transforms if entry states are Physical
  u.makeState(uxzstate, uystate);
  U.makeState(Ustate);

  return;
}

// This function spells out the computation in low-level operations rather 
// than making calls to calling grad and outer, because the sequence of 
// computations for latter would require a few extra transforms on u.

// Thesis notes 4/22/01, 12/01/03
// Compute nonlinearity as 1/2 [u' grad u' + div (u'u')]
// where u' = u + U ex
void skewsymmetricNL(const FlowField& u_, const ChebyCoeff& U_, FlowField& f, FlowField& tmp) {
  
  FlowField& u = (FlowField&) u_;
  ChebyCoeff& U = (ChebyCoeff&) U_;

  assert(u.Nd() == 3);
  assert(U.numModes() == u.Ny());
  assert(U.a() == u.a() && U.b() == u.b());

  fieldstate uxzstate = u.xzstate();
  fieldstate uystate  = u.ystate();
  fieldstate Ustate   = U.state();

  if (!u.geomCongruent(f) || f.Nd() != 3)
    f.resize(u.Nx(), u.Ny(), u.Nz(), 3, u.Lx(), u.Lz(), u.a(), u.b());
  else
    f.setToZero();
  f.setState(Physical, Physical);

  if (!u.geomCongruent(tmp) || tmp.Nd() < 9)
    tmp.resize(u.Nx(), u.Ny(), u.Nz(), 9, u.Lx(), u.Lz(), u.a(), u.b());
  else
    tmp.setToZero();

  // Possible extra transforms if entry states are Physical
  u.makeSpectral();
  U.makeSpectral();
  u += U;

  // ====================================================================
  // Compute 1/2 u dotgrad u. (where u is set to former u + U)
  
  FlowField& grad_u = tmp;
  grad(u, grad_u);

  grad_u.makePhysical();
  u.makePhysical();

  int Nx = u.Nx();
  int Ny = u.Ny();
  int Nz = u.Nz();
  
  // Accumulate 1/2 u_j du_i/dx_j in f_i
  for (int i=0; i<3; ++i) 
    for (int j=0; j<3; ++j) {
      int ij = i3j(i,j);
      for (int ny=0; ny<Ny; ++ny)
	for (int nx=0; nx<Nx; ++nx) 
	  for (int nz=0; nz<Nz; ++nz) 
	    f(nx,ny,nz,i) += 0.5*u(nx,ny,nz,j)*grad_u(nx,ny,nz,ij);
    }

  // ================================================================
  // II. Add grad dot (u u) to f. Spell out loops because div(uu, f) 
  // would overwrite results already in f (and changing order of div
  // and convec calculations would require an extra transform)

  FlowField& uu = tmp;

  //outer(u,u,uu);

  for (int ny=0; ny<Ny; ++ny) {
    for (int nx=0; nx<Nx; ++nx) 
      for (int nz=0; nz<Nz; ++nz) {
	Real u0 = u(nx,ny,nz,0);
	Real u1 = u(nx,ny,nz,1);
	Real u2 = u(nx,ny,nz,2);
	uu(nx,ny,nz,0) = u0*u0;
	uu(nx,ny,nz,1) = tmp(nx,ny,nz,3) = u0*u1;
	uu(nx,ny,nz,2) = tmp(nx,ny,nz,6) = u0*u2;
	uu(nx,ny,nz,4) = u1*u1;
	uu(nx,ny,nz,5) = tmp(nx,ny,nz,7) = u1*u2;
	uu(nx,ny,nz,8) = u2*u2;
      }
  }

  uu.makeSpectral();
  f.makeSpectral();

  int Mx = u.Mx();
  int My = u.My();
  int Mz = u.Mz();
  int kxmax = u.kxmax();
  int kzmax = u.kzmax();
  Real Lx = u.Lx();
  Real Lz = u.Lz();

  ComplexChebyCoeff tmpProfile(My, u.a(), u.b(), Spectral);
  ComplexChebyCoeff tmpProfile_y(My, u.a(), u.b(), Spectral);

  // Now set f_i += d/dx_j (u_i u_j)
  for (int i=0; i<3; ++i) {
    int i0 = i3j(i,0);
    int i1 = i3j(i,1);
    int i2 = i3j(i,2);

    // Add in du_i/dx and du_i/dz, that is, d/dx_j (u_i u_j) for j=0,2
    for (int my=0; my<My; ++my)     
      for (int mx=0; mx<Mx; ++mx) {
	int kx = u.kx(mx);
	Complex d_dx(0.0, 2*pi*kx/Lx*zero_last_mode(kx,kxmax,1));
	for (int mz=0; mz<Mz; ++mz) {
	  int kz = u.kz(mz);
	  Complex d_dz(0.0, 2*pi*kz/Lz*zero_last_mode(kz,kzmax,1));
	  f.cmplx(mx,my,mz,i) 
	    += 0.5*(d_dx*uu.cmplx(mx,my,mz,i0)+d_dz*uu.cmplx(mx,my,mz,i2));
	}
      }
    // Add in du_i/dy, that is d/dx_j (u_i u_j) for j=1
    for (int mx=0; mx<Mx; ++mx) 
      for (int mz=0; mz<Mz; ++mz) {
	for (int my=0; my<My; ++my)    
	  tmpProfile.set(my, uu.cmplx(mx,my,mz,i1));
	diff(tmpProfile, tmpProfile_y);
	for (int my=0; my<My; ++my) 
	  f.cmplx(mx,my,mz,i) += 0.5*tmpProfile_y[my]; // j=1
      }
  }

  u.makeSpectral(); 
  U.makeSpectral(); 
  u -= U;

  // Possible extra transforms if entry states are Physical
  u.makeState(uxzstate, uystate);
  U.makeState(Ustate);
  
  return;
}


void linearizedNL(const FlowField& u_, const ChebyCoeff& U_, FlowField& f) {
  FlowField& u = (FlowField&) u_;
  ChebyCoeff& U = (ChebyCoeff&) U_;

  assert(u.Nd() == 3);
  assert(U.N() == u.Ny());
  assert(U.a() == u.a() && U.b() == u.b());

  fieldstate uxzstate = u.xzstate();
  fieldstate uystate  = u.ystate();
  fieldstate Ustate   = U.state();

  if (!u.geomCongruent(f) || f.Nd() != 3)
    f.resize(u.Nx(), u.Ny(), u.Nz(), 3, u.Lx(), u.Lz(), u.a(), u.b());
  else
    f.setToZero();
  f.setState(Spectral, Physical);

  ChebyTransform trans(U.N());
  U.makeSpectral(trans);
  ChebyCoeff Uy;
  diff(U,Uy);
  U.makePhysical(trans);
  Uy.makePhysical(trans);

  Complex cu;
  Complex cv;
  Complex cw;

  int Mx = u.Mx();
  int Ny = u.Ny();
  int Mz = u.Mz();
  Real Lx = u.Lx();
  int kxmax = u.kxmax();

  u.makeState(Spectral, Physical);
  for (int ny=0; ny<Ny; ++ny) 
    for (int mx=0; mx<Mx; ++mx) {
      int kx = u.kx(mx);
      Complex d_dx(0.0, 2*pi*kx/Lx*zero_last_mode(kx,kxmax,1));
      for (int mz=0; mz<Mz; ++mz) {
	cu = u.cmplx(mx,ny,mz, 0);
	cv = u.cmplx(mx,ny,mz, 1);
	cw = u.cmplx(mx,ny,mz, 2);

	f.cmplx(mx,ny,mz, 0) = U[ny]*d_dx*cu + cv*Uy[ny];
	f.cmplx(mx,ny,mz, 1) = U[ny]*d_dx*cv;  
	f.cmplx(mx,ny,mz, 2) = U[ny]*d_dx*cw;
      }
    }

  f.makeSpectral(); 
  u.makeState(uxzstate,uystate);
  U.makeState(Ustate);
  return;
}

void uq2p(const FlowField& u_, const FlowField& q, FlowField& p, 
	  NonlinearMethod nonl_method) {
  
  assert(u_.Nd() == 3);
  assert(q.Nd() == 1);
  assert(p.Nd() == 1);

  p = q; 
  if (nonl_method != Rotational)
    return;

  FlowField& u = (FlowField&) u_;
  fieldstate uxzstate = u.xzstate();
  fieldstate uystate = u.ystate();

  u.makePhysical();
  p.makePhysical();

  int Nx=u_.Nx();
  int Ny=u_.Ny();
  int Nz=u_.Nz();

  // Remove 1/2 u dot u or 1/2 (u+U) dot (u+U) from p
  for (int ny=0; ny<Ny; ++ny) 
    for (int nx=0; nx<Nx; ++nx)
      for (int nz=0; nz<Nz; ++nz) 
	p(nx,ny,nz,0) -= 0.5*(square(u(nx,ny,nz,0)) + 
			      square(u(nx,ny,nz,1)) + 
			      square(u(nx,ny,nz,2))); 

  u.makeState(uxzstate, uystate); 
  p.makeState(q.xzstate(), q.ystate()); 
}

void up2q(const FlowField& u_, const FlowField& p, FlowField& q, 
	  NonlinearMethod nonl_method) {

  assert(u_.Nd() == 3);
  assert(q.Nd() == 1);
  assert(p.Nd() == 1);

  q = p;
  if (nonl_method != Rotational)
    return;

  FlowField& u = (FlowField&) u_;
  fieldstate uxzstate = u.xzstate();
  fieldstate uystate = u.ystate();

  u.makePhysical();
  q.makePhysical();

  int Nx=u_.Nx();
  int Ny=u_.Ny();
  int Nz=u_.Nz();
  
  // Add 1/2 u dot u to q
  for (int ny=0; ny<Ny; ++ny) {
    for (int nx=0; nx<Nx; ++nx)
      for (int nz=0; nz<Nz; ++nz) 
	q(nx,ny,nz,0) += 0.5*(square(u(nx,ny,nz,0)) + 
			      square(u(nx,ny,nz,1)) + 
			      square(u(nx,ny,nz,2))); 
  }
  u.makeState(uxzstate, uystate); 
  q.makeState(p.xzstate(), p.ystate()); 
  return;
}

ostream& operator<<(ostream& os, NonlinearMethod nonl) {
  string s;
  switch(nonl) {
  case Rotational: s="Rotational"; break;
  case Convection: s="Convection"; break;
  case Divergence: s="Divergence"; break;
  case SkewSymmetric: s="SkewSymmetric"; break;
  case Alternating: s="Alternating"; break;
  case Alternating_: s="Alternating_"; break;
  case Linearized: s="Linearized"; break;
  default: s = "Invalid NonlinearMethod: please submit bug report";
  }
  os << s;
  return os;
}


/**************************************************************************
void calc_dedt(const FlowField& u, const ChebyCoeff& U, const FlowField& p, 
	       Real dPdx, Real nu, FlowField& dedt, FlowField& tmp3d_a, 
	       FlowField& tmp3d_b, FlowField& tmp3d_c, FlowField& tmp9d) {
  assert(u.Nd() == 3);
  assert(U.state() == Spectral);
  assert(u.xzstate() == Spectral && u.ystate() == Spectral);

  FlowField& utot   = tmp3d_a;
  FlowField& grad_p = tmp3d_b;
  FlowField& lapl_u = tmp3d_c;
  FlowField& grad_u = tmp9d;

  // Set utot = u + U. This will reduce number of terms to calculate on RHS
  utot = u;
  utot += U;
  
  // Calculate derivatives of u and p
  grad(utot,grad_u);
  grad(p,   grad_p);
  lapl(utot,lapl_u);

  utot.makePhysical();
  grad_p.makePhysical();
  grad_u.makePhysical();
  lapl_u.makePhysical();

  dedt.setToZero();
  dedt.setState(Physical, Physical);

  int Nx = utot.Nx();
  int Ny = utot.Ny();
  int Nz = utot.Nz();
  int Nd = utot.Nd();

  // Calculate terms with two sums on i,j
  for (int j=0; j<Nd; ++j) 
    for (int i=0; i<Nd; ++i) 
      for (int ny=0; ny<Ny; ++ny)
	for (int nx=0; nx<Nx; ++nx)
	  for (int nz=0; nz<Nz; ++nz)
	    dedt(nx,ny,nz,0) = 
	      nu*utot(nx,ny,nz,i)*lapl_u(nx,ny,nz,i)
	      - utot(nx,ny,nz,i)*utot(nx,ny,nz,j)*grad_u(nx,ny,nz,i3j(i,j));

  // Calculate terms with one sum on i
  for (int i=0; i<Nd; ++i) 
    for (int ny=0; ny<Ny; ++ny)
      for (int nx=0; nx<Nx; ++nx)
	for (int nz=0; nz<Nz; ++nz)
	  dedt(nx,ny,nz,0) -= utot(nx,ny,nz,i)*grad_p(nx,ny,nz,i);
  

  // Calculate terms with no sum 
  for (int ny=0; ny<Ny; ++ny)
    for (int nx=0; nx<Nx; ++nx)
      for (int nz=0; nz<Nz; ++nz)
	dedt(nx,ny,nz,0) -= utot(nx,ny,nz,0)*dPdx;

 dedt.makeSpectral();
}
***********************************************************************/  
  
