/* turbstats.h: a class for computing basic statistics of turbulent flows
 * Channelflow-0.9
 *  
 * Copyright (C) 2001-2005  John F. Gibson
 *
 * John F. Gibson
 * Center for Nonlinear Sciences
 * School of Physics
 * Georgia Institute of Technology
 * Atlanta, GA 30332-0430
 *
 * gibson@cns.physics.gatech.edu
 * jfg@member.fsf.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, U
 */

#ifndef CHANNELFLOW_TURBSTATS_H
#define CHANNELFLOW_TURBSTATS_H

#include "channelflow/mathdefs.h"
#include "channelflow/chebyshev.h"
#include "channelflow/basisfunc.h"
#include "channelflow/flowfield.h"

class TurbStats {
public:
  TurbStats();
  TurbStats(const std::string& filebase);
  TurbStats(const ChebyCoeff& Ubase, Real nu);

  void reset();
  void addData(FlowField& ubase, FlowField& tmp);
  void msave(const std::string& filebase, bool wallunits=false) const;

  // Terminology: utot = U + u = Ubase + ubase
  ChebyCoeff U() const;          // average of total flow
  ChebyCoeff dUdy() const;       
  ChebyCoeff Ubase() const;      // base flow (e.g. parabola)
  ChebyCoeff ubase() const;      // mean fluctuation on base flow
  ChebyCoeff uu() const;         
  ChebyCoeff uv() const;
  ChebyCoeff uw() const;
  ChebyCoeff vv() const;
  ChebyCoeff vw() const;
  ChebyCoeff ww() const;

  // wall unit stuff
  Real ustar() const;   
  Real parabolicReynolds() const;  // h Uparab/nu, center vel of parab w = flux
  Real bulkReynolds() const;       // h Ubulk/nu
  Real centerlineReynolds() const; // h Ucenterline/nu
  Vector yplus() const;

private:
  int count_;
  Real nu_;          // All ChebyCoeff quantities are sums for means, in utot
  ChebyCoeff Ubase_; // base flow (parabolas, etc).
  ChebyCoeff ubase_; // mean fluc above base flow: utot = Ubase + ubase
  ChebyCoeff U_;     // mean flow = avg(utot)
  ChebyCoeff uu_;    // sum_1^count utot utot
  ChebyCoeff uv_;
  ChebyCoeff uw_;
  ChebyCoeff vv_;
  ChebyCoeff vw_;
  ChebyCoeff ww_;
};

  


#endif // TURBSTATS
