/*
**  NSAttributedString+FindAndReplace.m
**
**  Copyright (c) 2003
**
**  Author: Yen-Ju Chen <yjchenx@hotmail.com>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "NSAttributedString+FindAndReplace.h"
#include "RegEx.h"
#include <AppKit/AppKit.h>

@implementation NSAttributedString (FindAndReplace)

- (NSRange) rangeOfString: (NSString *) regexp
                  inRange: (NSRange) range
{
  return [self rangeOfString: regexp
                     inRange: range
                    backward: NO
             caseInsensitive: NO];

}

- (NSRange) rangeOfString: (NSString *) regexp
                  inRange: (NSRange) range
                 backward: (BOOL) backward
{
  return [self rangeOfString: regexp
                     inRange: range
                    backward: backward
             caseInsensitive: NO];

}

- (NSRange) rangeOfString: (NSString *) regexp
                  inRange: (NSRange) range
                 backward: (BOOL) backward
          caseInsensitive: (BOOL) caseInsensitive;
{
  RegExPattern *pattern;
  NSRange resultRange;  

  if (caseInsensitive == NO)
    {
      pattern = [RegExPattern regexPattern: regexp];
    }
  else
    {
      pattern = [[RegExPattern alloc] initWithPattern: regexp
                                             options: NSCaseInsensitiveSearch];
      AUTORELEASE(pattern);
    }

  if (pattern == nil)
    return NSMakeRange(NSNotFound, 0);
  
  if (backward == NO)
    {
      resultRange = [RegExParser rangeOfPattern: pattern
                                      inString: [self string]
                                         range: range];
      return resultRange;
    }
  else
    {
      /* backward is tricky and expensive.
       * It search all the matches in range, and pick up the last one.
       */
      NSRange tempRange;

      tempRange = range;
      resultRange = NSMakeRange(NSNotFound, 0);

      while(1)
        {
          tempRange = [RegExParser rangeOfPattern: pattern
                                        inString: [self string]
                                           range: tempRange];

          if(tempRange.location == NSNotFound)
            break;
          else
            {
              resultRange = tempRange;
              tempRange.location = NSMaxRange(tempRange);
              tempRange.length = NSMaxRange(range) - tempRange.location;
            }
        }
      return resultRange;
    }
}

@end
