/*
Copyright (C)  2006  Daniele Zelante

This file is part of comf.

comf is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

comf is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with comf; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
/*@LICENSE*/
// $Id$

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <syslog.h>

#include "file.hxx"

#include "exception.hxx"
#include "mymacros.hxx"



#define DRT(FD)										\
{ 													\
	if(_drt) 										\
	{		 										\
		if(_drt(errno,FD,__PRETTY_FUNCTION__))		\
			continue;								\
	}												\
	else LOGERREXCEPTIONHERE(LOG_ERR);				\
}



COMF_NS_BEGIN

const void * MMAP_FAILED = reinterpret_cast<void*>(-1);
static const off_t FNPOS = static_cast<off_t>(-1);

File::File(const char * name, int flags, mode_t mode, t_destructretry drt) :
	_drt(drt)
{
	_fd = ::open(name,flags,mode);
	if(_fd==-1) THROWERREXCEPTIONHERE();
}
	

File::~File()
{
	while(true)
	{	
		if(-1==::close(_fd))
		{
			if(errno==EINTR) continue;
			DRT(_fd);
		}
		break;
	}
}


size_t File::write(const void * buffer, size_t size) const
{
	ssize_t written;
	while(true)
	{
		written = ::write(_fd,buffer,size);
		if(written==-1)
		{
			if(errno==EINTR) continue;
			THROWERREXCEPTIONHERE();
		}
		break;
	}
	return written;
}


size_t File::read(void * buffer, size_t size) const
{
	ssize_t readen;	
	while(true)
	{	
		readen = ::read(_fd,buffer,size);
		if(readen==-1)
		{
			if(errno==EINTR) continue;
			THROWERREXCEPTIONHERE();
		}
		break;
	}
	return readen;
}

void File::setOfs(off_t pos, int whence) const
{
	off_t x = lseek(_fd,pos,whence);
	if(x==FNPOS) THROWERREXCEPTIONHERE();
}

size_t File::getOfs() const
{
	off_t pos = lseek(_fd,0,SEEK_CUR);
	if(pos==FNPOS) THROWERREXCEPTIONHERE();
	return pos;
}

File::operator int() const
{
	return _fd;
}

size_t File::getSize() const
{
	struct stat a;
	ERRCALL(fstat(_fd,&a));
	return a.st_size;
}

void File::setSize(size_t n) const
{
	ERRCALL(ftruncate(_fd,n));
}



//********************** FileLock

//TODO check use of fcntl / flock / lockf

FileLock::FileLock(File & f, bool write, bool wait, t_destructretry drt) : 
	_drt(drt),
	_file(f),
	_file_use(this,_file)
{
	struct flock k;
	memset(&k,0,sizeof(k));
	k.l_whence = SEEK_SET;
	k.l_start = 0;
	k.l_len = 0;
	k.l_type = write ? F_WRLCK : F_RDLCK;
	
	while(true)
	{
		if(-1==fcntl(_file, wait ? F_SETLKW : F_SETLK, &k))
		{
			if(errno==EINTR) continue;
			_file.delUser(this);
			THROWERREXCEPTIONHERE();
		}
		break;
	}
}

FileLock::~FileLock()
{
	struct flock k;
	memset(&k,0,sizeof(k));
	k.l_whence = SEEK_SET;
	k.l_start = 0;
	k.l_len = 0;
	k.l_type = F_UNLCK;
	
	while(true)
	{
		if(-1==fcntl(_file,F_SETLK,&k))
		{
			if(errno==EINTR) continue;
			DRT(_file);
		}
		break;
	}

}


//***** FileMappingConstSegment

FileMappingConstSegment::FileMappingConstSegment(const char * filename, t_destructretry drt) :
	_drt(drt),
	_file(filename,O_RDONLY,0,_drt),
	_lock(_file,false,false,_drt)
{
	_size = _file.getSize();
	_ptr = static_cast<char*>(mmap(0,_size,PROT_READ,MAP_SHARED,_file,0));
	if(_ptr==MMAP_FAILED) THROWERREXCEPTIONHERE();
}

FileMappingConstSegment::~FileMappingConstSegment()
{
	while(true)
	{	
		if(-1==munmap(_ptr,_size)) DRT(_file);
		break;
	}
}

size_t FileMappingConstSegment::size() const
{
	return _size;
}

const char * FileMappingConstSegment::cptr() const
{
	return _ptr;
}


//***** FileMappingSegment

FileMappingSegment::FileMappingSegment(const char * filename, t_destructretry drt) :
	_drt(drt),
	_file(filename,O_RDWR,0,drt),
	_lock(_file,true,false,drt)
{
	_size = _file.getSize();
	_ptr = static_cast<char*>(mmap(0,_size,PROT_READ|PROT_WRITE,MAP_SHARED,_file,0));
	if(_ptr==MMAP_FAILED) THROWERREXCEPTIONHERE();
}

FileMappingSegment::~FileMappingSegment()
{
	while(true)
	{	
		if(-1==munmap(_ptr,_size)) DRT(_file);
		break;
	}
}

size_t FileMappingSegment::size() const
{
	return _size;
}

const char * FileMappingSegment::cptr() const
{
	return _ptr;
}

char * FileMappingSegment::ptr() const
{
	return _ptr;
}



//**** FileMappingDynamicSegment
FileMappingDynamicSegment::FileMappingDynamicSegment(
	const char * filename,
	bool create,
	mode_t mode,
	size_t growby,
	t_destructretry drt
) :
	_drt(drt),
	_file(filename,O_RDWR|(create?O_CREAT:0),mode,drt),
	_lock(_file,true,false,drt)
{
	struct stat s;
	fstat(_file,&s);
	if(!S_ISREG(s.st_mode)) THROWCOMFEXCEPTIONHERE();

	size_t gran = std::max(
		static_cast<size_t>(sysconf(_SC_PAGESIZE)), 
		static_cast<size_t>(s.st_blksize)
		);
	_growby = growby - growby % gran;
	_growby = std::max(_growby, gran);
	if(create) _truesize = 0, _file.setSize(growby);
	else _truesize = _file.getSize();
	map();
}

FileMappingDynamicSegment::~FileMappingDynamicSegment()
{	
	while(true)
	{
		if(-1==::munmap(_ptr,_size)) DRT(_file);
		break;
	}
	
	while(true)
	{
		if(-1==::ftruncate(_file,_truesize)) DRT(_file);
		break;
	}
}


void FileMappingDynamicSegment::map()
{
	_size = _file.getSize();
	_ptr = static_cast<char*>(mmap(0,_size,PROT_READ|PROT_WRITE,MAP_SHARED,_file,0));
	if(_ptr==MMAP_FAILED) THROWERREXCEPTIONHERE();
	ASSERT(_ptr);
}

void FileMappingDynamicSegment::unmap()
{
	ASSERT(_ptr);
	ERRCALL(munmap(_ptr,_size));
}

void FileMappingDynamicSegment::resize(size_t n)
{
	if(n==_truesize) return;

	if(n>_truesize)
	{
		if(n>_size)
		{
			unmap();
			_file.setSize(roundup(n));
			map();
		}
	}

	if(n<_size)
	{
		if(_size-n > 2*_growby)
		{
			unmap();
			_file.setSize(roundup(n));
			map();
		}

	}
	_truesize = n;
}


void FileMappingDynamicSegment::clear(size_t n)
{
	resize(n);
}

void FileMappingDynamicSegment::clear()
{
	unmap();
	_file.setSize(_growby);
	map();
	_truesize = 0;
}

size_t FileMappingDynamicSegment::roundup(size_t n)
{
	return n + _growby - (n % _growby);
}

size_t FileMappingDynamicSegment::size() const
{
	return _truesize;
}

const char * FileMappingDynamicSegment::cptr() const
{
	return _ptr;
}

char * FileMappingDynamicSegment::ptr() const
{
	return _ptr;
}


COMF_NS_END
