/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#ifndef CONEXUSDATA_H
#define CONEXUSDATA_H

#include <boost/shared_array.hpp>

namespace conexus {

/**
 * This structure represents a data block allocated via a smart pointer.
 *
 * Additional it provides several typedefs that may be used to ensure type consistency both within
 * the conexus library, and with applications using the conexus library.
 *
 * @author Rick L Vinyard Jr
 * @ingroup conexus
 */
struct Data {
    /**
     * @typedef Octet
     * Typedefs the concept of an octet as an unsigned 8 bit integer.
     */
    typedef u_int8_t Octet;

    /**
     * @typedef sptr
     * Provides a convenience typedef of a smart pointer to data.
     * Hint: access externally as Data::sptr
     */
    typedef boost::shared_array<Octet> sptr;

    /**
     * Default constructor initializes data size to zero and relies on smart pointer constructor to
     * initialize pointer to NULL.
     */
    Data(): size(0) { }

    /**
     * Constructor which accepts a pointer to data and size and creates a duplicate
     */
    Data(const void* d, size_t s): size(s) {
        data = sptr(new Octet[size]);
        memcpy(data.get(), d, size);
    }

    /**
     * Constructor which accepts a smart pointer to data and data size in octets.
     */
    Data(sptr d, size_t s): data(d), size(s) { }

    /**
     * Constructor which creates a data object of a given size in octets.
     */
    Data(size_t s): size(s) {
        data = sptr(new Octet[s]);
    }

    /**
     * Overloaded dereference operator provides direct access to actual data. If you want access
     * to the smart pointer, use the @em data member instead.
     */
    operator Octet*() {
        return data.get();
    }

    void reset() {
      data.reset();
      size = 0;
    };
    
    /**
     * Smart pointer to the underlying data.
     */
    sptr data;

    /**
     * Size of the data in bytes.
     */
    size_t size;
};

/**
 * This class represents a constant data block allocated via a smart pointer, hence the name CData as a shortened
 * form of ConstantData.
 *
 * Additional it provides several typedefs that may be used to ensure type consistency both within
 * the conexus library, and with applications using the conexus library.
 * @author Rick L Vinyard Jr
 * @ingroup conexus
 */
struct CData {
    /**
     * @typedef Octet
     * Typedefs the concept of an octet as an unsigned 8 bit integer.
     */
    typedef u_int8_t Octet;

    /**
     * @typedef sptr
     * Provides a convenience typedef of a smart pointer to constant data.
     * Hint: access externally as CData::sptr
     */
    typedef boost::shared_array<Octet> sptr;

    /**
     * Default constructor initializes data size to zero and relies on smart pointer constructor to
     * initialize pointer to NULL.
     */
    CData(): data(m_data), size(0) { }

    /**
     * Constructor which accepts a smart pointer to data and data size in bytes.
     */
    CData(const void* d, size_t s): data(m_data), size(s) {
      m_data = sptr(new Octet[size]);
      memcpy(m_data.get(), d, size);
    }

    /**
     * Overloaded dereference operator provides direct access to actual data. If you want access
     * to the smart pointer, use the @em data member instead.
     */
    operator const Octet*() {
        return data.get();
    }

    /**
     * Smart pointer to the underlying data.
     */
    const sptr& data;

    /**
     * Size of the data in bytes.
     */
    const size_t size;

  protected:
    boost::shared_array<Octet> m_data;
};



}

#endif
