/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#ifndef CONEXUSCONNECT_ERROR_H
#define CONEXUSCONNECT_ERROR_H

#include <conexus/error.h>

namespace conexus {

/**
 * Base class for all connect errors.
 * @author Rick L Vinyard Jr
 * @ingroup errors
 */
class connect_error: public conexus_error {
public:
connect_error(int e=0): conexus_error("Unknown error, socket not connected.",e,"connect") { }
connect_error(const std::string s, int e=0): conexus_error(s,e,"connect") { }
}
;

namespace error {


namespace connect {


/**
 * EBADF
 * Thrown when equivalent UNIX domain (AF_UNIX) API error is encountered. May be caught alone
 * or via parent class.
 * @author Rick L Vinyard Jr
 * @ingroup errors
 */
class bad_fd: public connect_error {
public:
  bad_fd(): connect_error("The file descriptor is not a valid index in the descriptor table.", EBADF) { }
}
;

/**
 * EFAULT
 * Thrown when equivalent UNIX domain (AF_UNIX) API error is encountered. May be caught alone
 * or via parent class.
 * @author Rick L Vinyard Jr
 * @ingroup errors
 */
class segfault: public connect_error {
public:
  segfault(): connect_error("The socket structure  address  is  outside  the  user’s  address space.", EFAULT) { }
}
;

/**
 * ENOTSOCK
 * Thrown when equivalent UNIX domain (AF_UNIX) API error is encountered. May be caught alone
 * or via parent class.
 * @author Rick L Vinyard Jr
 * @ingroup errors
 */
class not_socket: public connect_error {
public:
  not_socket(): connect_error("The file descriptor is not associated with a socket.", ENOTSOCK) { }
}
;

/**
 * EISCONN
 * Thrown when equivalent UNIX domain (AF_UNIX) API error is encountered. May be caught alone
 * or via parent class.
 * @author Rick L Vinyard Jr
 * @ingroup errors
 */
class connected_socket: public connect_error {
public:
  connected_socket(): connect_error("The socket is already connected.", EISCONN) { }
}
;

/**
 * ECONNREFUSED
 * Thrown when equivalent UNIX domain (AF_UNIX) API error is encountered. May be caught alone
 * or via parent class.
 * @author Rick L Vinyard Jr
 * @ingroup errors
 */
class connection_refused: public connect_error {
public:
  connection_refused(): connect_error("No one listening on the remote address.", ECONNREFUSED) { }
}
;

/**
 * ETIMEDOUT
 * Thrown when equivalent UNIX domain (AF_UNIX) API error is encountered. May be caught alone
 * or via parent class.
 * @author Rick L Vinyard Jr
 * @ingroup errors
 */
class timeout: public connect_error {
public:
  timeout(): connect_error("Timeout while attempting connection.", ETIMEDOUT) { }
}
;

/**
 * ENETUNREACH
 * Thrown when equivalent UNIX domain (AF_UNIX) API error is encountered. May be caught alone
 * or via parent class.
 * @author Rick L Vinyard Jr
 * @ingroup errors
 */
class network_unreachable: public connect_error {
public:
  network_unreachable(): connect_error("Network is unreachable.", ENETUNREACH) { }
}
;

/**
 * EADDRINUSE
 * Thrown when equivalent UNIX domain (AF_UNIX) API error is encountered. May be caught alone
 * or via parent class.
 * @author Rick L Vinyard Jr
 * @ingroup errors
 */
class address_in_use: public connect_error {
public:
  address_in_use(): connect_error("Local address is already in use.", EADDRINUSE) { }
}
;

/**
 * EINPROGRESS
 * Thrown when equivalent UNIX domain (AF_UNIX) API error is encountered. May be caught alone
 * or via parent class.
 * @author Rick L Vinyard Jr
 * @ingroup errors
 */
class not_immediate: public connect_error {
public:
  not_immediate(): connect_error("The socket is non-blocking and the connection cannot be completed immediately.", EINPROGRESS) { }
}
;

/**
 * EALREADY
 * Thrown when equivalent UNIX domain (AF_UNIX) API error is encountered. May be caught alone
 * or via parent class.
 * @author Rick L Vinyard Jr
 * @ingroup errors
 */
class in_progress: public connect_error {
public:
  in_progress(): connect_error("The socket is non-blocking and a previous connection attempt has not yet been completed.", EALREADY) { }
}
;

/**
 * EAGAIN
 * Thrown when equivalent UNIX domain (AF_UNIX) API error is encountered. May be caught alone
 * or via parent class.
 * @author Rick L Vinyard Jr
 * @ingroup errors
 */
class try_again: public connect_error {
public:
  try_again(): connect_error("No more free local ports or insufficient entries in the routing cache.", EAGAIN) { }
}
;

/**
 * EAFNOSUPPORT
 * Thrown when equivalent UNIX domain (AF_UNIX) API error is encountered. May be caught alone
 * or via parent class.
 * @author Rick L Vinyard Jr
 * @ingroup errors
 */
class af_not_supported: public connect_error {
public:
  af_not_supported(): connect_error("The passed address didn’t have the correct address family in its sa_family field.", EAFNOSUPPORT) { }
}
;

/**
 * EACCES or EPERM
 * Thrown when equivalent UNIX domain (AF_UNIX) API error is encountered. May be caught alone
 * or via parent class.
 * @author Rick L Vinyard Jr
 * @ingroup errors
 */
class access: public connect_error {
public:
  access(): connect_error("The user tried to connect to a broadcast address without having the socket broadcast flag enabled or the connection request failed because of a local firewall rule.", EACCES) { }
}
;

/**
 * Thrown when attempting to autoconnect to a socket that is not capable of
 * autoconnecting or does not have autoconnection enabled.
 * @author Rick L Vinyard Jr
 * @ingroup errors
 */
class no_remote_address: public connect_error {
public:
  no_remote_address(): connect_error("Socket cannot auto connect because a default remote address has not been set.", -1) { }
};

      /**
       * Thrown when connect is requested of the Socket class
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
class no_socket_connect: public connect_error
{
public:
no_socket_connect(): connect_error("The Socket class connect method cannot be called without an address.", -1)
  { }
};

      /**
       * Thrown when connect is requested and a socket cannot be opened
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
class not_open: public connect_error
{
public:
not_open(): connect_error("Socket could not be opened for connect method.", -1)
  { }
};



}
}
}

#endif // CONEXUSCONNECT_ERROR_H
