/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#ifndef CONEXUSOPEN_ERROR_H
#define CONEXUSOPEN_ERROR_H

#include <conexus/error.h>

namespace conexus {

/**
 * Base class for all socket creation errors.
 * @author Rick L Vinyard Jr
 * @ingroup errors
 */
                  class open_error: public conexus_error {
public:
                  open_error(int e=0): conexus_error("Unknown error, socket cannot be created.",e,"open") { }
                  open_error(const std::string s, int e=0): conexus_error(s,e,"open") { }
};

namespace error {


namespace open {

/**
 * EPROTONOSUPPORT: Thrown when equivalent BSD socket API error is encountered. May be caught alone
 * or via parent class.
 * @author Rick L Vinyard Jr
 * @ingroup errors
 */
class protocol_not_supported: public open_error {
public:
  protocol_not_supported(): open_error("The  protocol  type  or  the specified protocol is not supported within this domain.", EPROTONOSUPPORT) { }
};

/**
 * EAFNOSUPPORT: Thrown when equivalent BSD socket API error is encountered. May be caught alone
 * or via parent class.
 * @author Rick L Vinyard Jr
 * @ingroup errors
 */
                class af_not_supported: public open_error {
public:
  af_not_supported(): open_error("The implementation does not support the specified  address  family.", EAFNOSUPPORT) { }
};

/**
 * ENFILE: Thrown when equivalent BSD socket API error is encountered. May be caught alone
 * or via parent class.
 * @author Rick L Vinyard Jr
 * @ingroup errors
 */
                class kernel_memory: public open_error {
public:
  kernel_memory(): open_error("Not enough kernel memory to allocate a new socket structure.", ENFILE) { }
};

/**
 * EMFILE: Thrown when equivalent BSD socket API error is encountered. May be caught alone
 * or via parent class.
 * @author Rick L Vinyard Jr
 * @ingroup errors
 */
                class process_file: public open_error {
public:
  process_file(): open_error("Process file table overflow.", EMFILE) { }
};

/**
 * EACCES: Thrown when equivalent BSD socket API error is encountered. May be caught alone
 * or via parent class.
 * @author Rick L Vinyard Jr
 * @ingroup errors
 */
                class access: public open_error {
public:
  access(): open_error("Permission to create a socket of the specified type and/or protocol is denied.", EACCES) { }
};

/**
 * ENOBUFS or ENOMEM: Thrown when equivalent BSD socket API error is encountered. May be caught alone
 * or via parent class.
 * @author Rick L Vinyard Jr
 * @ingroup errors
 */
                class no_memory: public open_error {
public:
  no_memory(): open_error("Insufficient memory is available. The socket cannot be created until sufficient resources are freed.", ENOBUFS) { }
};

/**
 * EINVAL: Thrown when equivalent BSD socket API error is encountered. May be caught alone
 * or via parent class.
 * @author Rick L Vinyard Jr
 * @ingroup errors
 */
class invalid_protocol: public open_error {
public:
  invalid_protocol(): open_error("Unknown protocol, or protocol family not available.", EINVAL) { }
};

/**
 * Thrown when auto open is requested and a device is not available
 * @author Rick L Vinyard Jr
 * @ingroup errors
 */
class no_device: public open_error {
public:
  no_device(): open_error("open requested and device not available or not set.", -1) { }
};

}


}
}

#endif // CONEXUSCREATE_ERROR_H
