/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#include "fdserver.h"

#include <sys/select.h>
#include <algorithm>

using namespace conexus;

FDServer::FDServer()
{
  m_timeval.tv_sec = 0;
  m_timeval.tv_usec = 5000;
  pthread_mutex_init(&m_mutex_ioset, NULL);
}


FDServer::~FDServer()
{}

void FDServer::set_responsiveness(long r)
{
  m_timeval.tv_usec = r;
}

long FDServer::get_responsiveness()
{
  return m_timeval.tv_usec;
}

void FDServer::add_io(FileDescriptor& io)
{
  pthread_mutex_lock(&m_mutex_ioset);
  m_io.insert(&io);
  pthread_mutex_unlock(&m_mutex_ioset);
}

void FDServer::add_socket(Socket& socket)
{
  switch (socket.get_state())
    {
    case Socket::CLOSED:
    case Socket::OPENED:
      socket.bind();
      break;
    case Socket::BOUND:
    case Socket::CONNECTED:
    case Socket::ACCEPTED:
    case Socket::LISTENING:
      break;
    }

  add_io(socket);
}

void FDServer::remove_io(FileDescriptor& io)
{
  pthread_mutex_lock(&m_mutex_ioset);
  m_io.erase(&io);
  pthread_mutex_unlock(&m_mutex_ioset);
}

void FDServer::remove_socket(Socket& socket)
{
  remove_io(socket);
}

void FDServer::create_read_set()
{
  std::set
    <FileDescriptor*>::iterator i;
  FD_ZERO(&m_readfd);
  m_n = -1;
  for (i = m_io.begin(); i != m_io.end(); i++)
    {
      FD_SET((*i)->get_fd(), &m_readfd);
      m_n = std::max(m_n, (*i)->get_fd()+1);
    }
}

void FDServer::service_thread_main()
{
  int n;
  Data d, dcpy;
  std::set
    <FileDescriptor*>::iterator i;
  struct timeval tv;

  while (!m_terminate)
    {
      pthread_mutex_lock(&m_mutex_ioset);
      create_read_set();
      if (m_timeval.tv_usec <= 0)
        n = select(m_n, &m_readfd, NULL, NULL, NULL);
      else
        {
          tv = m_timeval;
          n = select(m_n, &m_readfd, NULL, NULL, &tv);
        }
      if (n > 0)
        for (i = m_io.begin(); i != m_io.end(); i++)
          if (FD_ISSET((*i)->get_fd(), &m_readfd))
            queue_received_data((*i)->read());

      pthread_mutex_unlock(&m_mutex_ioset);
    }
}

