/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#include "io.h"

using namespace conexus;

IO::IO(bool close_on_destruction):
    m_close_on_destruction(close_on_destruction),
    m_state(CLOSED),
    m_readable(true),
    m_writable(true),
    m_read_blocked(false),
    m_write_blocked(false)
{
}


IO::~IO()
{
}

ssize_t IO::write(Data data)
{
  return write(data.data.get(), data.size);
}

ssize_t IO::write(CData data)
{
  return write(data.data.get(), data.size);
}

void IO::change_state( unsigned long states ) throw (state_error)
{
  if (states == UNCHANGED)
    return;

  // check for inconsistent states
  if ( states&CLOSED && states&OPENED)
    throw error::state::inconsistent();

  // perform the proper actions
  if ( is_open() && states&CLOSED ) {
    close();
    if ( is_open() )
      throw error::state::failed();
  }
  else if ( is_closed() && states&OPENED ) {
    open();
    if ( is_closed() )
      throw error::state::failed();
  }

}

void IO::set_state_opened() {
  if (is_open())
    return;
  m_state &= ~CLOSED;
  m_state |= OPENED;
  m_signal_opened.emit();
}

void IO::set_state_closed() {
  if (is_closed())
    return;
  m_state &= ~OPENED;
  m_state |= CLOSED;
  m_signal_closed.emit();
}

void IO::block_read(bool read_block) {
  m_read_blocked = read_block;
  m_signal_read_write_block_changed.emit(m_read_blocked, m_write_blocked);
}

void IO::block_write(bool write_block) {
  m_write_blocked = write_block;
  m_signal_read_write_block_changed.emit(m_read_blocked, m_write_blocked);
}

void IO::block_read_write(bool read_block, bool write_block) {
  m_read_blocked = read_block;
  m_write_blocked = write_block;
  m_signal_read_write_block_changed.emit(m_read_blocked, m_write_blocked);
}

bool IO::close_on_destruction() const
  {
    return m_close_on_destruction;
  }


void IO::set_close_on_destruction(bool value)
{
  m_close_on_destruction = value;
}

void IO::close_and_reopen( unsigned long new_state )
{
  unsigned long prev_state = get_state();
  change_state(CLOSED);
  if (new_state == UNCHANGED)
    change_state(prev_state);
  else
    change_state(new_state);
}

conexus::IO& operator<<(conexus::IO& io, conexus::Data d) {
  io.write(d);
  return io;
}

conexus::IO& operator<<(conexus::IO& io, conexus::CData d) {
  io.write(d);
  return io;
}

conexus::IO& operator>>(conexus::IO& io, conexus::Data d) {
  d = io.read();
  return io;
}

