/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#ifndef CONEXUS_IPIP_ADDRESS_H
#define CONEXUS_IPIP_ADDRESS_H

#include <string>

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <conexus/address.h>

namespace conexus {

/**
 * @namespace conexus::IP
 * The conexus::IP namespace contains classes capable of communicating at
 * OSI Layers 3 (Network) and 4 (Transport) which includes support for
 * raw IP, TCP/IP and UDP/IP communications [ For more information see IP(7) ]
 */
namespace IP {

/**
 * IPv4 network address.
 *
 * This object wraps the BSD socket API IPv4 network address structure sockaddr_in
 * and provides accessor methods for the members. Many of the related functions for
 * manipulating the sockaddr_in structure are presented in the class interface as
 * methods.
 *
 * Several extensions to the API functionality are also provided, such as the boolean
 * methods identifying traffic characteristics from address type.
 *
 * Accessors will provide get/set methods that will utilize parameters in @em local @em host
 * byte order rather than @em network byte order which is typically stored in the sockaddr_in
 * structure. These conversions rely upon the standard hton/ntoh functions to perform
 * their conversions.
 *
 * This class does not provide a strict encapsulation of the sockaddr_in structure as two public
 * methods are provided to directly access and potentially modify the underlying sockaddr_in.
 *
 * @author Rick L Vinyard Jr
 * @ingroup conexus
 */
class Address: public conexus::Address {
public:

    /**
     * Default constructor, or if parameters are used constructs an address from
     * a numeric address and an optional port number.
     *
     * Sets default family to AF_INET.
     *
     * @param address Numeric address in local host byte order.
     * @param port Port number in local host byte order.
     */
    Address(in_addr_t address=INADDR_ANY, u_int16_t port=0);

    /**
     * Constructs an address from a string which may be a hostname or in dotted decimal and
     * an optional port.
     *
     * Sets default family to AF_INET.
     *
     * @param address May be in dotted decimal or hostname form (will perform DNS query if latter).
     * @param port Port number in local host byte order.
     */
    Address(const std::string address, u_int16_t port=0);

    /**
     * Constructs an address from a network mask string (not CIDR), a hostname string
     * and an optional port number.
     *
     * Sets default family to AF_INET.
     *
     * @param net The network mask in dotted decimal. Must be a proper class A, B or C
     *    netmask and not a CIDR mask.
     * @param local The local address as dotted decimal or hostname.
     * @param port Port number in local host byte order.
     */
    Address(const std::string net, const std::string local, u_int16_t port=0);

    Address(struct sockaddr_in& addr);

    Address(struct sockaddr& addr);

    ~Address();

    /**
     * Returns the sin_family member of the encapsulated sockaddr_in structure which
     * is probably the default AF_INET.
     */
    sa_family_t get_family() const;

    /**
     * Sets the sin_family member of the encapsulated sockaddr_in structure. Unless
     * you know what you're doing you probably want to leave the family at AF_INET.
     */
    void set_family(sa_family_t);

    /**
     * Returns the port number associated with this address in local host byte order.
     */
    u_int16_t get_port() const;

    /**
     * Sets the port number associated with this address.
     * @param p Port number in host byte order.
     */
    void set_port(u_int16_t p);

    /**
     * Returns the address stored in the sockaddr_in.sin_addr.s_addr members
     * in local host byte order.
     */
    in_addr_t get_address() const;

    /**
     * Returns the network number part of this Internet address in local host byte
     * order.
     */
    in_addr_t get_network() const;

    /**
     * Returns the local host address part of this Internet address in local host byte
     * order.
     */
    in_addr_t get_local() const;

    /**
     * Sets the address stored in the sockaddr_in.sin_addr.s_addr member. The parameter
     * is in local host byte order, but will be converted to network byte order for
     * internal storage via htonl().
     * @param addr New address in local host byte order.
     */
    void set_address(in_addr_t addr);

    /**
     * Sets the address in the underlying sockaddr_in given the network number @em net with
     * the  local  address @em local in network @em net, both in local host byte order.
     */
    void set_address(in_addr_t net, in_addr_t local);

    /**
     * Sets the address in the underlying sockaddr_in given the network number @em net in local host
     * byte order while keeping the local address @em local intact.
     */
    void set_network(in_addr_t net);

    /**
     * Sets the address in the underlying sockaddr_in given the local address @em local in
     * local host byte order, while maintaining the same network mask.
     */
    void set_local(in_addr_t local);

    /**
     * Performs a DNS query of the provided string and sets this address equal to the
     * results of the DNS query.
     */
    void set_hostname(const std::string);

    /**
     * Determine whether the provided string is a valid hostname by performing
     * a DNS query.
     */
    bool is_valid_hostname(const std::string) const;

    /**
     * Performs a DNS query on the currently set address and returns the hostname string
     */
    std::string get_hostname() const;

    /**
     * Returns this address as a dotted decimal string.
     */
    std::string get_address_string() const;

    /**
     * Returns the network component as a dotted decimal string.
     */
    std::string get_network_string() const;

    /**
     * Returns the local network component of this address as a dotted decimal string.
     */
    std::string get_local_string() const;

    /**
     * Sets the address to the provided dotted decimal address.
     */
    void set_address(const std::string addr);

    /**
     * Sets the address to the provided dotted decimal address.
     */
    void set_address(const std::string net, const std::string local);

    /**
     * Sets the network component of the address to the provided dotted decimal address,
     * while leaving the local component intact.
     */
    void set_network_address(const std::string net);

    /**
     * Sets the local host address to the provided dotted decimal address,
     * while leaving the network component intact.
     */
    void set_local_address(const std::string local);

    /**
     * Provides a non-constant reference to the underlying sockaddr_in structure.
     */
    struct sockaddr_in& get_sockaddr_in();

    /**
     * Provides a non-constant pointer to the underlying sockaddr_in structure.
     */
    struct sockaddr_in* get_sockaddr_in_ptr();

    /**
     * True if this address specifies a private network.
     *
     * For reference, private networks are defined as:
     *  - 10.0.0.0        to   10.255.255.255  (10/8 prefix)
     *  - 172.16.0.0      to   172.31.255.255  (172.16/12 prefix)
     *  - 192.168.0.0     to   192.168.255.255 (192.168/16 prefix)
     */
    bool is_private() const;

    /**
     * True if this address is a reserved address specified as either 0.0.0.0/16 or class E (11110).
     */
    bool is_reserved() const;

    /**
     * True of this address is the loopback address with prefix 127.0.0.1/24.
     */
    bool is_loopback() const;

    /**
     * True if this is a valid broadcast address.
     */
    bool is_broadcast() const;

    /**
     * True if this is a valid multicast address.
     */
    bool is_multicast() const;

    /**
     * Overrides the virtual parent method and provides a pointer to the underlying sockaddr_in structure.
     */
    struct sockaddr* get_raw_address();

    /**
     * Overrides the virtual parent method and provides the size in bytes of the underlying sockaddr_in structure.
     */
    socklen_t get_raw_address_size() const;

protected:
  /**
   * The sockaddr_in structure encapsulated by this class.
   */
  struct sockaddr_in m_sockaddr_in;
};

}

}

#endif
