/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#ifndef CONEXUS_IPIP_IP_H
#define CONEXUS_IPIP_IP_H

#include <conexus/socket.h>
#include <conexus/ip_address.h>

namespace conexus {

namespace IP {

/**
 * This class serves as the base class for all IPv4 communication classes.
 *
 * All child classes such as UDP and TCP inherit the association of a local
 * host address from this class as well as the concept of interface association,
 * multicasting capabilities and of course, ports.
 *
 * This class also stores an internal address object representing a local interface
 * object. All BSD socket API functions (such as bind) that require a sockaddr_in
 * structure will be provided this internal attribute. Therefore, it is the
 * responsibility of children such as UDP and TCP to make the appropriate modifications
 * to this protected attribute.
 *
 * @author Rick L Vinyard Jr
 * @ingroup conexus
 */
class IP : public Socket {
public:
  /**
   * Default constructor which sets the socket domain to PF_INET but does not provide
   * a socket type or protocol.
   */
    IP(int type=-1, int protocol=0);

  virtual ~IP() throw();

    /**
     * Attempts to bind this socket to the internal address
     */
    virtual void bind() throw(bind_error);

    /**
     * Sets the interface of this socket to the interface associated with
     * the address specified in @em addr.
     */
    virtual void set_interface(const std::string addr);

    /**
     * Sets the local port associated with this socket.
     */
    virtual void set_local_port(u_int16_t port);

    /**
     * Sets the remote port associated with this socket.
     */
    virtual void set_remote_port(u_int16_t port);

  virtual const Address& get_local_address();

  virtual const Address& get_remote_address();

  virtual void set_remote_address(Address addr);

  virtual void set_local_address(Address addr);

protected:
    Address m_local_address;
    Address m_remote_address;
    bool m_remote_address_set;

};

}

}

#endif
