/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#ifndef CONEXUS_IPIP_UDP_H
#define CONEXUS_IPIP_UDP_H

#include <conexus/ip_ip.h>

namespace conexus {

namespace IP {

/**
 * This class encapsulates a socket used for UDP/IP communications.
 *
 * @author Rick L Vinyard Jr
 * @ingroup conexus
 */
class UDP : public IP
{
public:
    UDP();

  virtual ~UDP() throw();

    /**
     * Associates this socket with the interface specified by @em addr.
     *
     * If this is a broadcast address, also sets the broadcast option on the socket.
     */
    virtual void set_interface(const std::string addr);

    /**
     * Sets the UDP port associated with this socket.
     */
    virtual void set_local_port(u_int16_t port);

    /**
     * Sets the UDP port associated with this socket.
     */
    virtual void set_remote_port(u_int16_t port);

    /**
     * Returns the size in bytes of the next pending datagram, or 0 when no datagram
     * is pending.
     */
    int get_pending_input_datagram_size();

    /**
     * Returns the size in bytes of the local send queue.
     */
    int get_pending_output_datagram_size();

  /**
   * Overloads the Socket ancestor send method to ensure that the socket is connected
   * before trying to send.
   */
    virtual ssize_t write(const void* data, size_t size) throw (write_error);

    /**
     * Receives a single datagram from the UDP socket up to s bytes in size; allocates
     * any memory necessary as a smart pointer contained in the returned Data object.
     *
     * If s < 0 then receives the next pending datagram, however large it may be.
     */
    virtual Data read(size_t s = -1)  throw (read_error);

    virtual void set_remote_address(Address addr);
    virtual void set_local_address(Address addr);

    virtual void connect() throw (connect_error);
    virtual void connect(Address& a) throw (connect_error) { Socket::connect(a); }

  /**
   * There are some circumstances in which it may be desirable to set a
   * default address and have the send method transparently call the sendto method
   * without connecting rather than connecting and calling the send method.
   *
   * A specific example of this is when the remote address/port may occassionally
   * be unavailable. In this case the destination may generate an ICMP UNREACHABLE
   * message, and in accordance with RFC 1122 the local machine will fail any
   * future sends on a connected port, throwing a write::connection_refused error.
   * If you want to ensure that the datagram is sent contrary to RFC 1122 you could
   * set the default address, set send without connect and call send without throwing
   * connection errors.
   *
   * But, since this is contrary to RFC 1122 behavior you should really think about
   * your design before setting this.
   */

  void set_write_without_connect(bool value=true);


  bool get_write_without_connect() const;


protected:
  bool m_write_without_connect;
};

}

}

#endif
