/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#include "ll_packet.h"

#include <netinet/in.h>
#include <sys/ioctl.h>
#include <net/if.h>

using namespace conexus::LL;

Packet::Packet(int protocol, int type): Socket(PF_PACKET, type, htons(protocol))
{
  m_protocol = protocol;
}


Packet::~Packet() throw()
{
}


void Packet::bind() throw (bind_error)
{
  m_address.set_protocol(m_protocol);
  m_address.set_interface(1);
  Socket::bind(m_address);
}

ssize_t Packet::write(const void* data, size_t size) throw (write_error) {
  return Socket::writeto(m_address, data, size);
}

conexus::Data Packet::read() throw (read_error) {
  return Socket::read(ETH_FRAME_LEN);
}

void Packet::set_interface(int ifindex) {
  set_interface(get_interface_name(ifindex));
}

void Packet::set_interface(const std::string ifname) {
  if ( is_closed() )
    change_state(OPENED);
  struct ifreq req;
  strncpy(req.ifr_name, ifname.c_str(), IFNAMSIZ);
  if (ioctl(m_fd, SIOCGIFINDEX, &req) >= 0) {
    m_address.m_sockaddr_ll.sll_ifindex = req.ifr_ifindex;
    if (ioctl(m_fd, SIOCGIFHWADDR, &req) >= 0) {
      m_address.m_sockaddr_ll.sll_hatype = req.ifr_hwaddr.sa_family;
      m_address.m_sockaddr_ll.sll_halen = 8;
      memcpy(m_address.m_sockaddr_ll.sll_addr, req.ifr_hwaddr.sa_data, 8);
    }
  }
}

int Packet::get_interface_index(const std::string name) {
  if ( is_closed() )
    change_state(OPENED);
  int ifindex = -1;
  struct ifreq req;
  strncpy(req.ifr_name, name.c_str(), IFNAMSIZ);
  if (ioctl(m_fd, SIOCGIFINDEX, &req) >= 0)
    ifindex = req.ifr_ifindex;
  return ifindex;
}

std::string Packet::get_interface_name(int ifindex) {
  if ( is_closed() )
    change_state(OPENED);
  struct ifreq req;
  std::string ifname;
  req.ifr_ifindex = ifindex;
  if (ioctl(m_fd, SIOCGIFNAME, &req) >= 0)
    ifname = req.ifr_name;
  return ifname;
}


#define MAX_REQ_SIZE 100

std::vector<std::string> Packet::get_interface_names() {
  if ( is_closed() )
    change_state(OPENED);
  struct ifreq* req;
  struct ifconf conf;
  char buf[sizeof(struct ifreq)*MAX_REQ_SIZE];
  std::vector<std::string> ifnames;

  conf.ifc_len = sizeof(buf);
  conf.ifc_buf = (caddr_t)buf;

  if (ioctl(m_fd, SIOCGIFCONF, (caddr_t)&conf) >= 0)
    for (req = (struct ifreq*)conf.ifc_buf; (caddr_t)req < conf.ifc_buf+conf.ifc_len; req++)
      ifnames.push_back(std::string(req->ifr_name));

  return ifnames;
}
