/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#ifndef CONEXUSPACKETCAPTURE_H
#define CONEXUSPACKETCAPTURE_H

#include <vector>
#include <map>
#include <pcap.h>

#include <conexus/io.h>
#include <conexus/ip_address.h>

namespace conexus {

/**
@author Rick L Vinyard Jr
*/
class PacketCapture : public IO
{
public:
  PacketCapture(std::string device=std::string(), int snaplen=65535, bool promiscuous=false, int timeout_ms=0);

    ~PacketCapture();

    virtual void open() throw (open_error);
    virtual void close() throw (close_error);

  virtual Data read(size_t s) throw (read_error);

    virtual ssize_t write(CData data);
  virtual ssize_t write(const void* data, size_t size) throw (write_error);
    virtual ssize_t write(Data data);

    typedef enum Address {ADDRESS, NETMASK, BROADCAST, DESTINATION} Address;

    struct Device {
      Device(): loopback(false) { }
      std::string name;
      std::string description;
      std::map<unsigned, IP::Address> addresses;
      bool loopback;
    };

    typedef std::vector<Device> Devices;

    static Devices get_all_devices();

    std::string device() const;
    void set_device(const std::string& value=std::string());

  int snapshot_length() const;
  void set_snapshot_length(const int& value=65535);

  bool promiscuous() const;
  void set_promiscuous(bool value=false);

  int timeout_ms() const;
  void set_timeout_ms(const int& value=0);

  virtual void stop();
  virtual void set_responsiveness(long r);
  virtual long get_responsiveness();


  protected:
    char m_errbuf[PCAP_ERRBUF_SIZE];
    pcap_t* m_pcapd;
    std::string m_device;
    int m_snapshot_length;
    bool m_promiscuous;
    int m_timeout_ms;
    struct pcap_pkthdr* m_capture_header;
    uint8_t* m_capture_data;
    long m_responsiveness;

    pthread_mutex_t m_mutex_working;


    static char m_static_errbuf[PCAP_ERRBUF_SIZE];

    static void pcap_callback(u_char* object, const struct pcap_pkthdr* pkthdr, const u_char* packet);

    virtual void service_thread_main();

};

}

#endif
