/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#include "serial.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

using namespace conexus;

Serial::Serial(unsigned long readwrite):
  FileDescriptor()
{
  change_state(readwrite&(READ|WRITE));
}

Serial::~Serial()
{
}

ssize_t Serial::write( const void * data, size_t size ) throw (write_error)
{
  if (is_write_blocked())
    return 0;

  if ( is_closed() )
    try {
    change_state(OPENED|(get_state()&(READ|WRITE)));
    } catch (error::state::failed) {
      throw error::write::not_opened();
    }

  return ::write(m_fd, data, size);
  // TODO come back and check error codes
}

Data Serial::read( size_t s ) throw (read_error)
{
  Data d;
  if (is_read_blocked())
    return d;

  if ( is_closed() )
    try {
    change_state(OPENED|(get_state()&(READ|WRITE)));
    } catch (error::state::failed) {
      throw error::read::not_opened();
    }

  d.data = Data::sptr(new Data::Octet[s]);
  d.size = ::read(m_fd, d.data.get(), s);
  // TODO come back and check error codes
  return d;
}

void Serial::open( const std::string name, int state ) throw (open_error)
{
  int flags=O_NOCTTY|O_ASYNC;

  if ( is_open() )
    change_state(CLOSED);

  if ( state & (READ|WRITE) )
    flags |= O_RDWR;
  else if ( state & READ )
    flags |= O_RDONLY;
  else if ( state & WRITE )
    flags |= O_WRONLY;

  m_fd = ::open(name.c_str(), flags);
  // TODO come back and check error codes
  set_state_opened();
}

void Serial::open( ) throw( open_error )
{
  if (m_device == std::string())
    throw(error::open::no_device());
  open(m_device, get_state() );
}

std::string Serial::get_device( )
{
  return m_device;
}

void Serial::set_device( std::string d, int state ) throw (open_error)
{
  // if the device is the same and the state is the same, then leave
  if (m_device == d && (m_state==state || state==UNCHANGED))
    return;

  m_device = d;
  m_signal_device_changed.emit();
  change_state(state);
}

void Serial::change_state( long unsigned new_state ) throw (state_error)
{
  if (new_state == UNCHANGED)
    return;

  // if the READ or WRITE flags changed at a minimum we need to close the port to
  // force a reopen
  if ( m_state&(READ|WRITE) != new_state&(READ|WRITE) )
    IO::change_state(CLOSED);

  // change the READ and WRITE flags
  // zero out the READWRITE bits, then bitwise-or in only those two bits
  set_state_readable(new_state&READ);
  set_state_writable(new_state&WRITE);

  // check to see if the port needs to be opened
  IO::change_state(new_state);
}

void Serial::set_state_readable(bool b) {
  if (is_readable() && b)
    return;
  if (!is_readable() && !b)
    return;

  if (b)
    m_state |= READ;
  else
    m_state &= ~READ;
  m_signal_readable_changed.emit(b);
}

void Serial::set_state_writable(bool b) {
  if (is_writable() && b)
    return;
  if (!is_writable() && !b)
    return;

  if (b)
    m_state |= WRITE;
  else
    m_state &= ~WRITE;
  m_signal_writable_changed.emit(b);
}
