/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#include "server.h"

#include <sys/select.h>

using namespace conexus;

Server::Server() : m_running(false), m_dispatcher(NULL), m_use_dispatcher(false)
{
  pthread_mutex_init(&m_mutex_ioset, NULL);
  pthread_mutex_init(&m_mutex_signal_data, NULL);
  pthread_mutex_init(&m_mutex_data_queue, NULL);
}


Server::~Server()
{
  if (m_running)
    stop();
}

Server::iterator Server::connect_to_data(const slot_type& slot)
{
  iterator i;
  pthread_mutex_lock(&m_mutex_signal_data);
  i = m_signal_data.connect(slot);
  pthread_mutex_unlock(&m_mutex_signal_data);
  return i;
}

Server::citerator Server::connect_to_const_data(const cslot_type& slot)
{
  citerator i;
  i = m_signal_cdata.connect(slot);
  return i;
}

void Server::start(bool use_dispatcher)
{
  if (m_running)
    return;

  m_use_dispatcher = use_dispatcher;
  m_terminate = false;
  m_running = true;
  if (m_use_dispatcher)
    {
      m_dispatcher = new Glib::Dispatcher();
      m_dispatcher_connection = m_dispatcher->connect(sigc::mem_fun(*this, &Server::signal_data_received));
    }
  pthread_attr_init(&m_thread_attr);
  pthread_create(&m_thread, &m_thread_attr, &Server::service_thread_main_static_proxy, this);
}

void Server::stop()
{
  m_terminate = true;
  pthread_join(m_thread, NULL);
  m_running = false;
  if (m_use_dispatcher) {
  m_use_dispatcher = false;
  m_dispatcher_connection.disconnect();
  delete m_dispatcher;
  m_dispatcher = NULL;
  }
}

void* Server::service_thread_main_static_proxy(void * p)
{
  Server* s = (Server*)p;
  s->service_thread_main();
}

void Server::service_thread_main_proxy()
{
  service_thread_main();
}

void Server::queue_received_data( Data d )
{
  pthread_mutex_lock(&m_mutex_data_queue);
  m_data_queue.push(d);
  pthread_mutex_unlock(&m_mutex_data_queue);
  if ( pthread_mutex_trylock(&m_mutex_signal_data) == 0 )
    {
      pthread_mutex_unlock(&m_mutex_signal_data);
      if (m_use_dispatcher)
        m_dispatcher->emit();
      else
        signal_data_received();
    }
}

void Server::signal_data_received( )
{
  Data current;
  pthread_mutex_lock(&m_mutex_signal_data);
  pthread_mutex_lock(&m_mutex_data_queue);
  while ( ! m_data_queue.empty() )
    {
      current = m_data_queue.front();
      m_data_queue.pop();
      pthread_mutex_unlock(&m_mutex_data_queue);
      if (m_signal_cdata.slots().begin() != m_signal_cdata.slots().end())
        m_signal_cdata.emit(CData(current.data.get(), current.size));
      iterator next;
      next = m_signal_data.slots().begin();
      next++;
      for (iterator i = m_signal_data.slots().begin(); i != m_signal_data.slots().end(); i++)
        {
          if (next != m_signal_data.slots().end())
            {
              (*i)(Data(current.data.get(), current.size));
              next++;
            }
          else
            {
              (*i)(current);
              current.reset();
            }
        }
      pthread_mutex_lock(&m_mutex_data_queue);
    }
  pthread_mutex_unlock(&m_mutex_data_queue);
  pthread_mutex_unlock(&m_mutex_signal_data);
}
