/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#ifndef CONEXUSSERVER_H
#define CONEXUSSERVER_H

#include <set>
#include <queue>

#include <sys/select.h>
#include <pthread.h>

#include <sigc++/sigc++.h>

#include <glibmm/dispatcher.h>
#include <glibmm/thread.h>

#include <conexus/data.h>

namespace conexus {

/**
@author Rick L Vinyard Jr
 * @ingroup conexus
 */
class Server: public sigc::trackable {
public:
  typedef sigc::slot<void,Data> slot_type;
  typedef sigc::slot<void,CData> cslot_type;
  typedef sigc::slot_list<slot_type> slot_list_type;
  typedef sigc::slot_list<cslot_type> cslot_list_type;
  typedef slot_list_type::iterator iterator;
  typedef cslot_list_type::iterator citerator;

  Server();
  ~Server();

  virtual void start(bool use_dispatcher=false);
  virtual void stop();

  iterator connect_to_data(const slot_type& slot);
  citerator connect_to_const_data(const cslot_type& slot);

    /**
   * Sets the responsiveness to add_*, remove_* and stop commands in microseconds;
   * lower is more responsive; larger is less responsive; default is 5000 microseconds (or 5 ms).
   *
   * In general this refers to the timeout period of the @em select() call used internally.
   * Each cycle of the server I/O thread acquires several mutex locks at the beginning
   * of the cycle and releases them at the end of the cycle.
   *
   * This value shortens the cycle period, thereby making the server more responsive to commands at
   * the cost of increasing overhead.
     */
  virtual void set_responsiveness(long r) = 0;

  /**
   * Returns the server responsiveness level.
   */
  virtual long get_responsiveness() = 0;


protected:
  int m_n;
  pthread_t m_thread;
  pthread_attr_t m_thread_attr;
  bool m_terminate;
  bool m_running;
  sigc::signal<void, Data> m_signal_data;
  sigc::signal<void, CData> m_signal_cdata;
  sigc::connection m_dispatcher_connection;
  Glib::Dispatcher* m_dispatcher;
  bool m_use_dispatcher;
  std::queue<Data> m_data_queue;


  pthread_mutex_t m_mutex_ioset;
  pthread_mutex_t m_mutex_signal_data;
  pthread_mutex_t m_mutex_data_queue;

  virtual void service_thread_main() { }

  void queue_received_data( Data d );
  void signal_data_received( );

private:
  static void* service_thread_main_static_proxy(void*);
  void service_thread_main_proxy();

};

}

#endif
