/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#include "tty.h"

#include <termios.h>
#include <sys/ioctl.h>

using namespace conexus;

TTY::TTY()
        : Serial() {
    cfmakeraw(&m_termios);
}


TTY::~TTY() {
    ::tcsetattr(m_fd, TCSANOW, &m_origtermios);
}

void TTY::open(const std::string name, int s)  throw (open_error){
    Serial::open(name, s);
    ::tcgetattr(m_fd, &m_origtermios);
}


void TTY::drain( ) {
    ::tcdrain(m_fd);
}

void TTY::flush_input( ) {
    ::tcflush(m_fd, TCIFLUSH);
}

void TTY::flush_output( ) {
    ::tcflush(m_fd, TCOFLUSH);
}

void TTY::flush( ) {
    ::tcflush(m_fd, TCIOFLUSH);
}

void TTY::suspend_output( ) {
    ::tcflow(m_fd, TCOOFF);
}

void TTY::restart_output( ) {
    ::tcflow(m_fd, TCOON);
}

void TTY::suspend_input( ) {
    ::tcflow(m_fd, TCIOFF);
}

void TTY::restart_input( ) {
    ::tcflow(m_fd, TCION);
}

void TTY::tcsetattr( SetOption option ) {
    ::tcsetattr(m_fd, option, &m_termios);
}

void TTY::set_input_speed( speed_t speed ) {
    ::cfsetispeed(&m_termios, speed);
    tcsetattr(SET_NOW);
}

void TTY::set_output_speed( speed_t speed ) {
    ::cfsetospeed(&m_termios, speed);
    tcsetattr(SET_NOW);
}

void TTY::set_input_modes(tcflag_t iflag, SetOption option) {
    m_termios.c_iflag = iflag;
    tcsetattr(option);
}

void TTY::set_output_modes(tcflag_t oflag, SetOption option) {
    m_termios.c_oflag = oflag;
    tcsetattr(option);
}

void TTY::set_control_modes(tcflag_t cflag, SetOption option) {
    m_termios.c_cflag = cflag;
    tcsetattr(option);
}

void TTY::set_local_modes(tcflag_t lflag, SetOption option) {
    m_termios.c_lflag = lflag;
    tcsetattr(option);
}

void TTY::set_control_characters(int index, cc_t value, SetOption option) {
    m_termios.c_cc[index] = value;
    tcsetattr(option);
}

speed_t TTY::get_input_speed( ) {
    return ::cfgetispeed(&m_termios);
}

speed_t TTY::get_output_speed( ) {
    return ::cfgetospeed(&m_termios);
}

struct termios TTY::get_attributes() {
  struct termios attr;
  tcgetattr(m_fd, &attr);
  return attr;
}

// ssize_t TTY::write( const void * data, size_t size ) {
//     return ::write(m_fd, data, size);
// }

Data TTY::read( size_t s ) throw (read_error) {
    int n;
    ioctl(m_fd, TIOCINQ, &n);
  return Serial::read(n);
}
