/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#include "serialcombobox.h"

#include <iostream>
#include <sstream>

using namespace conexusgtk;

SerialComboBox::SerialComboBox()
 : Gtk::ComboBox(),
 m_connected(NULL),
 m_hal_ctx_status(true)
{
  m_reftreemodel = Gtk::ListStore::create(m_modelcolumns);
  set_model(m_reftreemodel);
  pack_start(m_modelcolumns.description);
  pack_start(m_modelcolumns.device);

  dbus_error_init(&m_dbus_error);
  if ((m_hal_ctx = libhal_ctx_new()) == NULL) {
    std::cerr << "error: libhal_ctx_new" << std::endl;
    m_hal_ctx_status = false;
  }

  if (!libhal_ctx_set_dbus_connection(m_hal_ctx, dbus_bus_get(DBUS_BUS_SYSTEM, &m_dbus_error))) {
    std::cerr << "error: libhal_ctx_set_dbus_connection: " << m_dbus_error.name << ": " << m_dbus_error.message << std::endl;
    m_hal_ctx_status = false;
  }

  if (!libhal_ctx_init(m_hal_ctx, &m_dbus_error)) {
    std::cerr << "error: libhal_ctx_init: " << m_dbus_error.name << ": " << m_dbus_error.message << std::endl;
    m_hal_ctx_status = false;
  }

  load_from_hal();

}


SerialComboBox::~SerialComboBox()
{
  if (!libhal_ctx_shutdown(m_hal_ctx, &m_dbus_error))
    std::cerr << "error: libhal_ctx_init: " << m_dbus_error.name << ": " << m_dbus_error.message << std::endl;

  if (!libhal_ctx_free(m_hal_ctx))
    std::cerr << "error: libhal_ctx_free" << std::endl;
}

void SerialComboBox::load_from_hal( )
{
  int num_udis;
  char** udis;
  char* device;
  uint32_t port;
  std::ostringstream oss;
  Gtk::TreeModel::Row row;

  if (!m_hal_ctx_status)
    return;

  udis = libhal_find_device_by_capability(m_hal_ctx, "serial", &num_udis, &m_dbus_error);

  if (dbus_error_is_set(&m_dbus_error)) {
    std::cerr << "error: " << m_dbus_error.name << ": " << m_dbus_error.message << std::endl;
    return;
  }

  m_reftreemodel->clear();

  if (num_udis == 0)
    return;

  row = *(m_reftreemodel->append());
  row[m_modelcolumns.description] = "No device selected";

  for (int i=0; i < num_udis; i++) {
    device = libhal_device_get_property_string(m_hal_ctx, udis[i], "serial.device", &m_dbus_error);
    if (dbus_error_is_set(&m_dbus_error)) {
      std::cerr << "error: " << m_dbus_error.name << ": " << m_dbus_error.message << std::endl;
      continue;
    }
    port = libhal_device_get_property_int(m_hal_ctx, udis[i], "serial.port", &m_dbus_error);
    if (dbus_error_is_set(&m_dbus_error)) {
      std::cerr << "error: " << m_dbus_error.name << ": " << m_dbus_error.message << std::endl;
      continue;
    }
    row = *(m_reftreemodel->append());
    oss.str("");
    oss << "Serial Port " << port;
    row[m_modelcolumns.description] = oss.str();
    row[m_modelcolumns.device] = device;
    libhal_free_string(device);
  }

  libhal_free_string_array(udis);

  if (get_active_row_number() == -1)
    set_active(0);
}

bool SerialComboBox::connect( conexus::Serial & serial, bool replace )
{
  bool set = false;
  Gtk::TreeModel::Row row;

  if (m_connected != NULL && !replace)
    return false;
  Gtk::TreeModel::iterator iter;
  for (iter = m_reftreemodel->children().begin(); iter != m_reftreemodel->children().end(); iter++) {
    if ( (*iter)[m_modelcolumns.device] == serial.get_device() ) {
      set = true;
      set_active(iter);
    }
  }
  if (!set && serial.get_device().size() > 0 ) {
    row = *(m_reftreemodel->append());
    row[m_modelcolumns.device] = serial.get_device();
    set_active(m_reftreemodel->children().end()-1);
  }
  else if (!set)
    set_active(0);

  m_connected = &serial;

  return true;
}

conexus::Serial * SerialComboBox::get_connected( )
{
  return m_connected;
}

bool SerialComboBox::disconnect( )
{
  m_connected = NULL;
}

std::string SerialComboBox::get_device( )
{
  Glib::ustring device;
  if (get_active_row_number() == -1)
    return std::string();
  device = get_active()->operator[](m_modelcolumns.device);
  return device;
}

sigc::signal< void > & SerialComboBox::signal_device_changed( )
{
  return m_signal_device_changed;
}

void SerialComboBox::on_changed( )
{
  if (m_connected != NULL)
    m_connected->set_device(get_device(), conexus::Serial::UNCHANGED);
  Gtk::ComboBox::on_changed();
}


