/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/

/**
 * This program is a reimplementation of miniterm.c for the conexus library
 * which contains the following copyright notation.
 *
 *  AUTHOR: Sven Goldt (goldt@math.tu-berlin.de)
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 * This is like all programs in the Linux Programmer's Guide meant
 * as a simple practical demonstration.
 * It can be used as a base for a real terminal program.
 */

#include <conexus/conexus.h>
#include <iostream>


#include <termios.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/signal.h>
#include <stdlib.h>
#include <sys/wait.h>

#define BAUDRATE B115200
#define MODEMDEVICE "/dev/ttyS0"
#define ENDMINITERM 2 /* ctrl-b to quit miniterm */

#define _POSIX_SOURCE 1 /* POSIX compliant source */

using namespace conexus;
using namespace std;

volatile bool STOP=false;

void child_handler(int s) {
    STOP=true;
}

void print_data(Data d);

int main() {
    int fd,c;
    TTY tty;
    FDServer server;
    struct termios oldstdio, newstdio;
    struct sigaction sa;

  conexus::init();

    /*
      Open modem device for reading and writing and not as controlling tty
      because we don't want to get killed if linenoise sends CTRL-C.
    */
    tty.open(MODEMDEVICE);

    /*
       Set bps rate and hardware flow control and 8n1 (8bit,no parity,1 stopbit).
       Also don't hangup automatically and ignore modem status.
       Finally enable receiving characters.
     */
    tty.set_control_modes(BAUDRATE | CRTSCTS | CS8 | CLOCAL | CREAD, TTY::SET_FLUSH);

    /*
      Ignore bytes with parity errors and make terminal raw and dumb.
    */
    tty.set_input_modes(IGNPAR);

    /*
      Raw output.
    */
    tty.set_output_modes(0);

    /*
      Don't echo characters because if you connect to a host it or your
      modem will echo characters for you. Don't generate signals.
    */
    tty.set_local_modes(0);

    /* blocking read until 1 char arrives */
    tty.set_control_characters(VMIN, 1);
    tty.set_control_characters(VTIME, 0);

    /*
      Strange, but if you uncomment this command miniterm will not work
      even if you stop canonical mode for stdout.
    */
    tcgetattr(1,&oldstdio);
    newstdio = tty.get_attributes();
    tcsetattr(1,TCSANOW,&newstdio); /* stdout settings like modem settings */

    /* terminal settings done, now handle in/ouput */
    switch (fork()) {
    case 0: /* child */
        /* user input */
        close(1); /* stdout not needed */
        for (c=getchar(); c!= ENDMINITERM ; c=getchar())
            tty.write(&c,1);
        tty.close();
        tcsetattr(1,TCSANOW,&oldstdio);
        exit(0); /* will send a SIGCHLD to the parent */
        break;
    case -1:
        perror("fork");
        tty.close();
        tcsetattr(1,TCSANOW,&oldstdio);
        exit(-1);
    default: /* parent */
        close(0); /* stdin not needed */
        sa.sa_handler = child_handler;
        sa.sa_flags = 0;
        sigaction(SIGCHLD,&sa,NULL); /* handle dying child */
        server.connect_to_data(sigc::ptr_fun(&print_data));
        server.add_io(tty);
        server.start();
        while (!STOP); /* modem input handler */
        server.stop();
        wait(NULL); /* wait for child to die or it will become a zombie */
        break;
    }
    return 0;
}

void print_data(Data d) {
    for (int i=0; i < d.size; i++)
        cout << (char)(*((char*)d.data.get()+i));
    cout << flush;
}

