/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#include <conexus/conexus.h>

#include <sys/types.h>
#include <unistd.h>
#include <iostream>

/**
 * This is an example of the conexus library server class that watches two
 * UDP objects and calls all three print functions for any data received
 * by either UDP object.
 *
 * This example illustrates how to add input objects to the server, and
 * also how to add multiple callback slots.
 */

// prototypes of the three print callback functions
void print_data1(conexus::Data d);
void print_data2(conexus::Data d);
void print_data3(conexus::Data d);

int main() {
  conexus::init();
  // Declare the two udp objects and the server object
  conexus::IP::UDP udp1, udp2;

  // Set each udp object to two seperate ports
  udp1.set_local_port(1500);
  udp2.set_local_port(1501);

  std::vector<std::pair<std::string,std::string> > ifs = conexus::get_interface_ip_addresses();
  std::vector<std::pair<std::string,std::string> >::iterator iter;
  for (iter = ifs.begin(); iter != ifs.end(); iter++)
    std::cout << "Interface: " << iter->first << "\t\tAddress: " << iter->second << std::endl;

  std::vector<std::string> ifnames = conexus::get_interface_names();
  for (std::vector<std::string>::iterator i = ifnames.begin(); i != ifnames.end(); i++)
    std::cout << *i << " : " << conexus::get_interface_index(*i) << std::endl;

  std::cout << conexus::get_interface_name(0) << std::endl;

  // The server connect method connects a provided sigc++ slot that will be called
  // back when the server receives any data.
  udp1.connect_to_data(sigc::ptr_fun(&print_data1));
  udp1.connect_to_data(sigc::ptr_fun(&print_data2));
  udp1.connect_to_data(sigc::ptr_fun(&print_data3));
  udp2.connect_to_data(sigc::ptr_fun(&print_data1));
  udp2.connect_to_data(sigc::ptr_fun(&print_data2));
  udp2.connect_to_data(sigc::ptr_fun(&print_data3));

  // Sets the response time in usec
  udp1.set_responsiveness(5);
  udp2.set_responsiveness(5);

  // Start the server. The server will spawn a separate thread to service
  // received data, so this call will immediately return after the thread
  // is spawned.
  udp1.start();
  udp2.start();

std::cout << "Main thread pid: " << pthread_self() << std::endl;
  // Set up a loop that will run for 20 seconds and print the time every 5
  // seconds. Since the server is threaded, the sleep(1) call will not effect
  // the servicing thread.
  std::cout << "Starting..." << std::endl;
  for (int i=1; i <= 20; i++) {
    if (i%5 == 0)
      std::cout << "Time: " << i << std::endl;
    sleep(1);
  }

  // Stop the server and prepare for shutdown
  udp1.stop();
  udp2.stop();

  return 0;
}

// These are the three callback functions. Each simply prints out the data received.

void print_data1(conexus::Data d) {
  static conexus::Data sd;
  std::cout << "Responding thread pid: " << pthread_self() << std::endl;
  sd = d;
  std::cout << "<1> Received " << d.size << " bytes of data (" << *((uint64_t*)d.data.get()) << ") [" << d << "]\n";
}

void print_data2(conexus::Data d) {
  static conexus::Data sd;
  sd = d;
  std::cout << "<2> Received " << d.size << " bytes of data (" << (uint64_t)d.data.get() << ") [" << d << "]\n";
}

void print_data3(conexus::Data d) {
  static conexus::Data sd;
  sd = d;
  std::cout << "<3> Received " << d.size << " bytes of data (" << (uint64_t)d.data.get() << ") [" << d << "]\n";
}
