/* Configuration file parser -*- tab-width: 4; -*-
 *
 * Copyright (c) 2002, Martin Hedenfalk <mhe@home.se>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <string.h>
#include <stdlib.h>
#include <assert.h>
#include <limits.h>
#include <errno.h>
#include <pwd.h>
#include <sys/types.h>
#include <unistd.h>

#include "confuse.h"

#if ENABLE_NLS
# include <libintl.h>
# define _(str) dgettext(PACKAGE, str)
#else
# define _(str) str
#endif
#define N_(str) str

extern FILE *yyin;

const char *confuse_version = PACKAGE_VERSION;
const char *confuse_copyright = PACKAGE_STRING" by Martin Hedenfalk <mhe@home.se>";
const char *confuse_author = "Martin Hedenfalk <mhe@home.se>";

extern int yylex(cfg_t *cfg);
char * yylval;
extern int cfg_lexer_include(cfg_t *cfg, const char *fname);

cfgopt_t *cfg_getopt(cfg_t *cfg, const char *name)
{
	int i;

	assert(cfg && cfg->name && name);

	for(i = 0; cfg->opts[i].name; i++) {
		if(is_set(CFGF_NOCASE, cfg->flags)) {
			if(strcasecmp(cfg->opts[i].name, name) == 0) {
				return &cfg->opts[i];
			}
		} else {
			if(strcmp(cfg->opts[i].name, name) == 0) {
				return &cfg->opts[i];
			}
		}
	}
	cfg_error(cfg, _("no such option '%s'"), name);
	return 0;
}

const char * cfg_title(cfg_t *cfg)
{
	return cfg->title;
}

unsigned int cfg_size(cfg_t *cfg, const char *name)
{
	cfgopt_t *opt = cfg_getopt(cfg, name);
	if(opt)
		return opt->nvalues;
	return 0;
}

signed long cfg_getnint(cfg_t *cfg, const char *name, unsigned int index)
{
	cfgopt_t *opt = cfg_getopt(cfg, name);
	if(opt) {
		assert(opt->type == CFGT_INT);
		if(opt->nvalues == 0)
			return (signed long)opt->def;
		else {
			assert(index < opt->nvalues);
			return opt->values[index]->number;
		}
	} else
		return 0;
}

signed long cfg_getint(cfg_t *cfg, const char *name)
{
	return cfg_getnint(cfg, name, 0);
}

double cfg_getnfloat(cfg_t *cfg, const char *name, unsigned int index)
{
	cfgopt_t *opt = cfg_getopt(cfg, name);
	if(opt) {
		assert(opt->type == CFGT_FLOAT);
		if(opt->nvalues == 0)
			return opt->fpdef;
		else {
			assert(index < opt->nvalues);
			return opt->values[index]->fpnumber;
		}
	} else
		return 0;
}

double cfg_getfloat(cfg_t *cfg, const char *name)
{
	return cfg_getnfloat(cfg, name, 0);
}

cfgbool_t cfg_getnbool(cfg_t *cfg, const char *name, unsigned int index)
{
	cfgopt_t *opt = cfg_getopt(cfg, name);
	if(opt) {
		assert(opt->type == CFGT_BOOL);
		if(opt->nvalues == 0)
			return (cfgbool_t)opt->def;
		else {
			assert(index < opt->nvalues);
			return opt->values[index]->boolean;
		}
	} else
		return 0;
}

cfgbool_t cfg_getbool(cfg_t *cfg, const char *name)
{
	return cfg_getnbool(cfg, name, 0);
}

char *cfg_getnstr(cfg_t *cfg, const char *name, unsigned int index)
{
	cfgopt_t *opt = cfg_getopt(cfg, name);
	if(opt) {
		assert(opt->type == CFGT_STR);
		if(opt->nvalues == 0) {
			return (char *)opt->def;
		} else {
			assert(index < opt->nvalues);
			return opt->values[index]->string;
		}
	}
	return 0;
}

char *cfg_getstr(cfg_t *cfg, const char *name)
{
	return cfg_getnstr(cfg, name, 0);
}

cfg_t *cfg_getnsec(cfg_t *cfg, const char *name, unsigned int index)
{
	cfgopt_t *opt = cfg_getopt(cfg, name);
	if(opt) {
		assert(opt->type == CFGT_SEC);
		assert(opt->values);
		assert(index < opt->nvalues);
		return opt->values[index]->section;
	}
	return 0;
}

cfg_t *cfg_gettsec(cfg_t *cfg, const char *name, const char *title)
{
	unsigned int i, n;

	n = cfg_size(cfg, name);
	for(i = 0; i < n; i++) {
		cfg_t *sec = cfg_getnsec(cfg, name, i);
		assert(sec && sec->title);
		if(is_set(CFGF_NOCASE, cfg->flags)) {
			if(strcasecmp(title, sec->title) == 0)
				return sec;
		} else {
			if(strcmp(title, sec->title) == 0)
				return sec;
		}
	}
	return 0;
}

cfg_t *cfg_getsec(cfg_t *cfg, const char *name)
{
	return cfg_getnsec(cfg, name, 0);
}

static cfgval_t *cfg_addval(cfgopt_t *opt)
{
	opt->values = (cfgval_t **)realloc(opt->values,
									   (opt->nvalues+1) * sizeof(cfgval_t *));
	assert(opt->values);
	opt->values[opt->nvalues] = (cfgval_t *)malloc(sizeof(cfgval_t));
	memset(opt->values[opt->nvalues], 0, sizeof(cfgval_t));
	return opt->values[opt->nvalues++];
}

static cfgopt_t *cfg_dupopts(cfgopt_t *opts)
{
	int n;
	cfgopt_t *dupopts;

	for(n = 0; opts[n].name; n++)
		/* do nothing */ ;
	dupopts = (cfgopt_t *)malloc(++n * sizeof(cfgopt_t));
	memcpy(dupopts, opts, n * sizeof(cfgopt_t));
	return dupopts;
}

static int parse_boolean(const char *s)
{
	if(strcasecmp(s, "true") == 0
	   || strcasecmp(s, "on") == 0
	   || strcasecmp(s, "yes") == 0)
		return 1;
	else if(strcasecmp(s, "false") == 0
			|| strcasecmp(s, "off") == 0
			|| strcasecmp(s, "no") == 0)
		return 0;
	return -1;
}

cfgval_t *cfg_setopt(cfg_t *cfg, cfgopt_t *opt, char *value)
{
	cfgval_t *val;
	int b;
	char *endptr;

	assert(cfg && opt);

	if(opt->simple_value) {
		assert(opt->type != CFGT_SEC);
		val = (cfgval_t *)opt->simple_value;
	} else {
		if(opt->nvalues == 0 || is_set(CFGF_MULTI, opt->flags) ||
		   is_set(CFGF_LIST, opt->flags))
			val = cfg_addval(opt);
		else
			val = opt->values[0];
	}

	switch(opt->type) {
		case CFGT_INT:
			val->number = strtol(value, &endptr, 0);
			if(*endptr != '\0') {
				cfg_error(cfg, _("invalid integer value for option '%s'"),
						  opt->name);
				return 0;
			}
			if(val->number == LONG_MIN || val->number == LONG_MAX) {
				cfg_error(cfg,
						  _("integer value for option '%s' is out of range"),
						  opt->name);
				return 0;
			}
			break;
		case CFGT_FLOAT:
			val->fpnumber = strtod(value, &endptr);
			if(*endptr != '\0') {
				cfg_error(cfg,
						  _("invalid floating point value for option '%s'"),
						  opt->name);
				return 0;
			}
			if(errno == ERANGE) {
				cfg_error(cfg,
				  _("floating point value for option '%s' is out of range"),
						  opt->name);
				return 0;
			}
			break;
		case CFGT_STR:
			if(val->string)
				free(val->string);
			val->string = strdup(value);
			break;
		case CFGT_SEC:
			val->section = (cfg_t *)malloc(sizeof(cfg_t));
			assert(val->section);
			memset(val->section, 0, sizeof(cfg_t));
			val->section->name = strdup(opt->name);
			val->section->opts = cfg_dupopts(opt->subopts);
			val->section->flags = cfg->flags;
			val->section->flags |= CFGF_ALLOCATED;
			val->section->filename = cfg->filename;
			val->section->line = cfg->line;
			val->section->errfunc = cfg->errfunc;
			val->section->title = value;
			break;
		case CFGT_BOOL:
			b = parse_boolean(value);
			if(b == -1) {
				cfg_error(cfg, _("invalid boolean value for option '%s'"),
						  opt->name);
				return 0;
			}
			val->boolean = (cfgbool_t)b;
			break;
		default:
			cfg_error(cfg, "internal error in cfg_setopt()");
			break;
	}
	return val;
}

void cfg_free_val(cfgopt_t *opt)
{
	int i;

	assert(opt);
	for(i = 0; i < opt->nvalues; i++) {
		if(opt->type == CFGT_STR)
			free(opt->values[i]->string);
		else if(opt->type == CFGT_SEC)
			cfg_free(opt->values[i]->section);
		free(opt->values[i]);
	}
	free(opt->values);
	opt->values = 0;
	opt->nvalues = 0;
}

cfg_errfunc_t cfg_set_error_function(cfg_t *cfg, cfg_errfunc_t errfunc)
{
	cfg_errfunc_t old = cfg->errfunc;
	cfg->errfunc = errfunc;
	return old;
}

void cfg_error(cfg_t *cfg, const char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);

	if(cfg->errfunc) {
		(*cfg->errfunc)(cfg, fmt, ap);
	} else {
		if(cfg && cfg->filename && cfg->line)
			fprintf(stderr, "%s:%d: ", cfg->filename, cfg->line);
		else if(cfg && cfg->filename)
			fprintf(stderr, "%s: ", cfg->filename);
		vfprintf(stderr, fmt, ap);
		fprintf(stderr, "\n");
	}

	va_end(ap);
}

static int cfg_parse_internal(cfg_t *cfg, int level)
{
	int state = 0, next_state = 0;
	char *opttitle = 0;
	cfgopt_t *opt = 0;
	cfgval_t *val;
	cfgbool_t append_value;
	cfgopt_t funcopt = CFG_END();

	while(1) {
		int tok = yylex(cfg);

		if(tok == 0) {
			/* lexer.l should have called cfg_error */
			return 1;
		}

		if(tok == EOF) {
			if(state != 0) {
				cfg_error(cfg, _("premature end of file"));
				return 1;
			}
			return 0;
		}

		switch(state) {
			case 0: /* expecting an option name */
				if(tok == '}') {
					if(level == 0) {
						cfg_error(cfg, _("unexpected closing brace"));
						return 1;
					}
					return 0;
				}
				if(tok != CFGT_STR) {
					cfg_error(cfg, _("unexpected token '%s'"), yylval);
					return 1;
				}
				opt = cfg_getopt(cfg, yylval);
				if(opt == 0)
					return 1;
				if(opt->type == CFGT_SEC) {
					if(is_set(CFGF_TITLE, opt->flags))
						state = 6;
					else
						state = 5;
				} else if(opt->type == CFGT_FUNC) {
					state = 7;
				} else
					state = 1;
				break;
			case 1: /* expecting an equal sign or plus-equal sign */
				append_value = cfg_false;
				if(tok == '+') {
					if(!is_set(CFGF_LIST, opt->flags)) {
						cfg_error(cfg,
								  _("attempt to append to non-list option %s"),
								  opt->name);
						return 1;
					}
					append_value = cfg_true;
				} else if(tok != '=') {
					cfg_error(cfg, _("missing equal sign after option '%s'"),
							  opt->name);
					return 1;
				}
				if(is_set(CFGF_LIST, opt->flags)) {
					if(!append_value)
						cfg_free_val(opt);
					state = 3;
				} else {
					state = 2;
					next_state = 0;
				}
				break;
			case 2: /* expecting an option value */
				if(tok == '}' && is_set(CFGF_LIST, opt->flags)) {
					state = 0;
					break;
				}

				if(tok != CFGT_STR) {
					cfg_error(cfg, _("unexpected token '%s'"), yylval);
					return 1;
				}

				if(cfg_setopt(cfg, opt, yylval) == 0)
					return 1;
				state = next_state;
				break;
			case 3: /* expecting an opening brace for a list option */
				if(tok != '{') {
					cfg_error(cfg, _("missing opening brace for option '%s'"),
							  opt->name);
					return 1;
				}
				state = 2;
				next_state = 4;
				break;

			case 4: /* expecting a separator for a list option, or
					 * closing (list) brace */
				if(tok == ',') {
					state = 2;
					next_state = 4;
				} else if(tok == '}')
					state = 0;
				else {
					cfg_error(cfg, _("unexpected token '%s'"), yylval);
					return 1;
				}
				break;
			case 5: /* expecting an opening brace for a section */
				if(tok != '{') {
					cfg_error(cfg, _("missing opening brace for section '%s'"),
							  opt->name);
					return 1;
				}

				val = cfg_setopt(cfg, opt, opttitle);
				opttitle = 0;
				if(!val)
					return 1;

				if(cfg_parse_internal(val->section, level+1) != 0)
					return 1;
				cfg->line = val->section->line;
				state = 0;
				break;
			case 6: /* expecting a title for a section */
				if(tok != CFGT_STR) {
					cfg_error(cfg, _("missing title for section '%s'"),
							  opt->name);
					return 1;
				} else
					opttitle = strdup(yylval);
				state = 5;
				break;
			case 7: /* expecting a opening parenthesis for a function */
				if(tok != '(') {
					cfg_error(cfg, _("missing parenthesis for function '%s'"),
							  opt->name);
					return 1;
				}
				if(funcopt.nvalues >= opt->nparams)
					state = 10;
				else
					state = 8;
				break;
			case 8: /* expecting a function parameter */
				if(tok != CFGT_STR) {
					cfg_error(cfg, _("missing parameter for function '%s'"),
							  opt->name);
					return 1;
				}

				val = cfg_addval(&funcopt);
				val->string = strdup(yylval);
				if(funcopt.nvalues >= opt->nparams)
					state = 10;
				else
					state = 9;
				break;
			case 9: /* expecting a comma in a function call */
				if(tok != ',') {
					cfg_error(cfg, _("missing comma for function '%s'"),
							  opt->name);
					return 1;
				}
				state = 8;
				break;
			case 10: /* expecting a closing parenthesis in a function call */
				if(tok == ')') {
					int ret;
					const char **argv;
					int i;

					argv = (const char **)malloc(funcopt.nvalues *
												 sizeof(char *));
					for(i = 0; i < funcopt.nvalues; i++)
						argv[i] = funcopt.values[i]->string;
					ret = (*opt->func)(cfg, funcopt.nvalues, argv);
					cfg_free_val(&funcopt);
					free(argv);
					if(ret != 0)
						return 1;
					state = 0;
				} else {
					cfg_error(cfg, _("missing parenthesis for function '%s'"),
							  opt->name);
					return 1;
				}
				break;
		}
	}
}

cfg_t *cfg_init(cfgopt_t *opts, cfg_flag_t flags)
{
	cfg_t *cfg;

	cfg = (cfg_t *)malloc(sizeof(cfg_t));
	cfg->name = "root";
	cfg->opts = opts;
	cfg->flags = flags;
	cfg->filename = 0;
	cfg->line = 0;

#if ENABLE_NLS
	setlocale(LC_MESSAGES, "");
	setlocale(LC_CTYPE, "");
	bindtextdomain(PACKAGE, LOCALEDIR);
#endif

	return cfg;
}

char *cfg_tilde_expand(const char *filename)
{
	char *expanded = 0;

	/* do tilde expansion
	 */
	if(filename[0] == '~') {
		struct passwd *passwd = 0;
		const char *f = 0;

		if(filename[1] == '/') {
			passwd = getpwuid(geteuid());
			f = filename + 1;
		} else {
			char *user;
			char *e = strchr(filename, '/');
			if(e) {
				f = e;
				user = malloc(f - filename);
				strncpy(user, filename + 1, f - filename - 1);
				passwd = getpwnam(filename + 1);
				free(user);
			}
		}
		if(passwd) {
			expanded = malloc(strlen(passwd->pw_dir) + strlen(f) + 1);
			strcpy(expanded, passwd->pw_dir);
			strcat(expanded, f);
		}
	} else
		expanded = strdup(filename);
	return expanded;
}

int cfg_parse(cfg_t *cfg, const char *filename)
{
	assert(cfg && filename);

	cfg->filename = cfg_tilde_expand(filename);
	if(cfg->filename == 0) {
		cfg_error(cfg, _("%s: can't expand home directory"), filename);
		return 1;
	}

	cfg->line = 1;

	yyin = fopen(cfg->filename, "r");
	if(yyin == 0)
		return -1;

	return cfg_parse_internal(cfg, 0);
}

void cfg_free(cfg_t *cfg)
{
	int i;

	assert(cfg);

	for(i = 0; cfg->opts[i].name; ++i)
		cfg_free_val(&cfg->opts[i]);

	if(is_set(CFGF_ALLOCATED, cfg->flags)) {
		free(cfg->name);
	   free(cfg->opts);
	} else
		free(cfg->filename);

	free(cfg);
}

int cfg_include(cfg_t *cfg, int argc, const char **argv)
{
	assert(argc == 1);
	return cfg_lexer_include(cfg, argv[0]);
}
